#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import datetime

from si_tests import settings
from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
import si_tests.utils.waiters as helpers
from si_tests import logger

LOG = logger.logger


class K8sService(K8sNamespacedResource):
    resource_type = 'service'

    @property
    def type(self):
        return self.data['spec'].get('type', None)

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_service(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_service(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_service(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_service(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, **kwargs):
        self._manager.api.delete_namespaced_service(
            self.name, self.namespace, **kwargs)

    def get_ip(self):
        """Return service internal IP"""
        return self.read().spec.cluster_ip

    def wait_for_external_ip(self, log_path=settings.ARTIFACTS_DIR,
                             timeout=500, interval=3):
        try:
            helpers.wait(
                lambda: self.get_external_ip() is not None,
                timeout=timeout, interval=interval,
                timeout_msg='Timeout waiting, service {0} '
                            'does not got external address'.format(
                    self.name))
        except Exception as e:
            svc_describe = str(self.read())
            timestamp = datetime.datetime.now().strftime('%Y%m%d%H%M%S')
            logfile = '{path}svc-describe-{svs_name}-{svc_ns}-{timestamp}.log'\
                .format(path=log_path, svs_name=self.name,
                        svc_ns=self.namespace, timestamp=timestamp
                        )
            with open(logfile, 'w') as f:
                f.write(svc_describe)
            LOG.error(svc_describe)
            LOG.error(e)
            raise
        return self

    def get_external_ip(self):
        """Return service external IP or None"""
        ingress = self.read().status.load_balancer.ingress
        if ingress:
            return ingress[0].ip or ingress[0].hostname
        else:
            return None

    def get_external_hostname(self):
        """Return service external hostname or None"""
        ingress = self.read().status.load_balancer.ingress
        return ingress[0].hostname if ingress else None

    def get_external_addr(self):
        """Return service external IP or None"""
        ingress = self.read().status.load_balancer.ingress
        return ingress[0].ip if ingress else None

    def get_ports(self):
        """Return list of dicts

        Example:
          - name: http
            nodePort: 30695
            port: 80
            protocol: TCP
            targetPort: http
        """
        return self.read().spec.ports


class K8sServiceManager(K8sBaseManager):
    resource_class = K8sService

    @property
    def api(self):
        return self._cluster.api_core

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_service(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_service_for_all_namespaces(**kwargs)

    def describe_fixed_resources(self):
        """Collect the release-independent properties of the resources

        To check that the upgrades don't change any collected here property.
        """
        data = {}
        for obj in self._list_all().to_dict()['items']:
            name = '{0}/{1}/{2}'.format(obj['metadata']['namespace'],
                                        self.resource_type,
                                        obj['metadata']['name'])
            ingress = obj['status']['load_balancer']['ingress']
            if ingress:
                ingress_address = (ingress[0].get('ip') or
                                   ingress[0].get('hostname'))
                data[name] = ingress_address
        return data
