#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
from tabulate import tabulate

from si_tests.clients.k8s import base
from si_tests.utils import k8s_utils, waiters
from si_tests import logger
from si_tests import settings

LOG = logger.logger


class V1K0rdentRegion(base.BaseModel):
    pass


class V1K0rdentRegionList(base.BaseModelList):
    pass


class K0rdentRegion(base.K8sClusterScopedResource):
    resource_type = 'region'
    model = V1K0rdentRegion

    @property
    def kubeconfig(self):
        return self.data['spec'].get('kubeConfig', {}).get('name', None)

    @property
    def clustername(self):
        # NOTE(va4st) We're assuming that all the things happen inside k0rdent installation and all clusters are
        # deployed by k0rdent - e.g the format of cluster/kubeconfig is always the same
        return self.kubeconfig.split('-kubeconfig')[0]

    @property
    def kubeconfig_secret(self):
        return self._manager._cluster.secrets.get(self.kubeconfig, namespace=settings.KCM_NAMESPACE)

    @property
    def kubeconfig_data(self):
        """Decode and return kubeconfig data from secret."""
        _, data = k8s_utils.get_kubeconfig_from_secret(self._manager._cluster, secret=self.kubeconfig_secret)
        return data

    @property
    def providers(self):
        """List of providers configured for the region."""
        providers = self.data['spec'].get('providers', [])
        return [provider['name'] for provider in providers] if providers else []

    @property
    def clusterdeployment(self):
        # TODO(va4st) add clusterdeployment lookup after 1.5.0 oss
        raise NotImplementedError

    @property
    def credentials(self):
        """List of credentials configured for the region.

        :return: List
        """
        return [cred for cred in self._manager._cluster.k0rdent_credentials.list_all() if cred.region == self.name]

    @property
    def ready(self):
        result = self.get_conditions(verbose=True)
        if result['not_ready']:
            return False
        else:
            return True

    def get_conditions(self, verbose):
        result = {
            'ready': [],
            'not_ready': []
        }
        data_status = self.data.get('status') or {}
        conditions = data_status.get('conditions', [])
        if not conditions:
            result['not_ready'].append("No conditions in helmrelease status yet")

        for condition in conditions:
            if condition['status'] in ['True', 'False']:
                ready = eval(condition['status'])
            else:
                ready = False

            condition_type = condition.get('type')
            if ready:
                result['ready'].append(condition_type)
            else:
                result['not_ready'].append(condition_type)

            if verbose:
                condition_message = condition.get('message', '')
                LOG.info(f"Condition ({condition_type}) have message ({condition_message}) and "
                         f"status ({condition['status']})")
        if verbose:
            comp_statuses = self.collect_components_statuses()
            headers = ["Component", "Ready", "Template", "Message"]
            data = [[stat['component'],
                     stat['ready'],
                     stat['template'],
                     stat['message']]
                    for stat in comp_statuses]
            message = tabulate(data, tablefmt="presto", headers=headers)
            LOG.info(f"\n{message}\n")
        return result

    def collect_components_statuses(self):
        data_status = self.data.get('status') or {}
        components = data_status.get('components', {})
        components_data = []
        for component in components.keys():
            components_data.append({
                'component': component,
                'ready': True if components[component].get('success', '') else False,
                'template': components[component].get('template', ''),
                'message': components[component].get('error', '')
            })
        return components_data

    def wait_for_readiness(self, timeout=900, interval=15):
        """Wait for region to become ready"""
        LOG.info(f"Waiting for region {self.name} to become ready")
        timeout_msg = f'Timeout waiting for region {self.name} to become ready'
        waiters.wait(lambda: self.ready,
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=timeout_msg)


class K0rdentRegionManager(base.K8sBaseManager):
    model = V1K0rdentRegionList
    resource_class = K0rdentRegion
    resource_group = 'k0rdent.mirantis.com'
    resource_version = 'v1beta1'
    resource_plural = 'regions'
