#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s import base
from si_tests.utils import utils, waiters
from si_tests import logger

LOG = logger.logger


class V1K0rdentMultiClusterService(base.BaseModel):
    pass


class V1K0rdentMultiClusterServiceList(base.BaseModelList):
    pass


class K0rdentMultiClusterService(base.K8sClusterScopedResource):
    resource_type = 'multiclusterservice'
    model = V1K0rdentMultiClusterService

    def get_conditions(self):
        """Returns k0rdent-specific list for the MCS object condition"""
        data_status = self.data.get('status') or {}
        return data_status.get('conditions') or []

    def get_cluster_conditions(self, cluster_name, cluster_namespace):
        """Returns k0rdent-specific list for the services condition related to the specified cluster"""
        data_status = self.data.get('status') or {}
        clusters_conditions = data_status.get('services') or []
        for cluster_conditions in clusters_conditions:
            # For kcm >= 1.3.0 , status.services are empty in MultiClusterServices. MCS services status
            # is appended to the related ClusterDeployments status.services.
            # In this case, to check the services readiness please use cld.check.check_cluster_readiness().
            if (cluster_conditions.get('clusterName') == cluster_name and
                    cluster_conditions.get('clusterNamespace') == cluster_namespace):
                return cluster_conditions.get('conditions') or []
        return None

    def wait_for_readiness(self, timeout=1800, interval=15, expected_fails=None, verbose=True):
        def _get_mcs_readiness():
            conditions = self.get_conditions()
            if not conditions:
                return False
            result = utils.check_conditions(conditions, expected_fails=expected_fails, verbose=verbose)
            if result['not_ready']:
                return False
            else:
                return True

        LOG.info(f"Wait for readiness of the MultiCluseterService '{self.name}' conditions ...")
        timeout_msg = f"Timeout waiting for MultiClusterService '{self.name}' status.conditions readiness"
        waiters.wait(
                _get_mcs_readiness,
                timeout=timeout,
                interval=interval,
                timeout_msg=timeout_msg)

    def wait_for_services_readiness(self, cluster_name, cluster_namespace,
                                    timeout=1800, interval=15, expected_fails=None, verbose=True):
        def _get_mcs_services_readiness():
            conditions = self.get_cluster_conditions(cluster_name, cluster_namespace)
            result = utils.check_conditions(conditions, expected_fails=expected_fails, verbose=verbose)
            if result['not_ready']:
                return False
            else:
                return True

        LOG.info(f"Wait for readiness of the services for cluster {cluster_namespace}/{cluster_name} "
                 f"deployed from MultiCluseterService '{self.name}' ...")
        timeout_msg = (f"Timeout waiting for MultiClusterService '{self.name}' status.services readiness "
                       f"for the cluster {cluster_namespace}/{cluster_name}")
        waiters.wait(
                _get_mcs_services_readiness,
                timeout=timeout,
                interval=interval,
                timeout_msg=timeout_msg)


class K0rdentMultiClusterServiceManager(base.K8sBaseManager):
    model = V1K0rdentMultiClusterServiceList
    resource_class = K0rdentMultiClusterService
    resource_group = 'k0rdent.mirantis.com'
    resource_version = 'v1beta1'
    resource_plural = 'multiclusterservices'
