#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import tempfile
import yaml
import json

import kubernetes
from kubernetes import client

from si_tests.clients.k8s.capi_awsclusters import AWSClusterManager
from si_tests.clients.k8s.capi_awsmanagedclusters import AWSManagedClusterManager
from si_tests.clients.k8s.capi_azureclusters import AzureClusterManager
from si_tests.clients.k8s.capi_vsphereclusters import VSphereClusterManager
from si_tests.clients.k8s.capi_gcpclusters import GCPClusterManager
from si_tests.clients.k8s.capi_gcpmanagedclusters import GCPManagedClusterManager
from si_tests.clients.k8s.capi_gcpmanagedcontrolplanes import GCPManagedControlPlaneManager
from si_tests.clients.k8s.capi_gcpmanagedmachinepools import GCPManagedMachinePoolManager
from si_tests.clients.k8s.capi_clusters import ClusterManager
from si_tests.clients.k8s.capi_machinepools import MachinePoolManager
from si_tests.clients.k8s.capi_awsclusterstaticidentity import AWSClusterStaticIdentityManager
from si_tests.clients.k8s.capi_azureclusteridentity import AzureClusterIdentityManager
from si_tests.clients.k8s.capi_azureasomanagedclusters import AzureASOManagedClusterManager
from si_tests.clients.k8s.capi_metal3clusters import Metal3ClusterManager
from si_tests.clients.k8s.capi_remoteclusters import RemoteClusterManager
from si_tests.clients.k8s.capi_azureasomanagedmachinepools import AzureASOManagedMachinePoolManager
from si_tests.clients.k8s.capi_machines import MachineManager
from si_tests.clients.k8s.capi_vsphereclusteridentity import VSphereClusterIdentityManager
from si_tests.clients.k8s.capi_openstackclusters import OpenStackClusterManager
from si_tests.clients.k8s.capi_openstackclusteridentities import OpenStackClusterIdentityManager
from si_tests.clients.k8s.capi_infrastructureproviders import InfrastructureProviderManager
from si_tests.clients.k8s.k0_credential import K0rdentCredentialManager
from si_tests.clients.k8s.k0_clusterdeployment import K0rdentClusterDeploymentManager
from si_tests.clients.k8s.k0_clustertemplate import K0rdentClusterTemplateManager
from si_tests.clients.k8s.k0_management import K0rdentManagementManager
from si_tests.clients.k8s.k0_servicetemplate import K0rdentServiceTemplateManager
from si_tests.clients.k8s.k0_multiclusterservice import K0rdentMultiClusterServiceManager
from si_tests.clients.k8s.k0_providerinterfaces import K0rdentProviderInterfaceManager
from si_tests.clients.k8s.k0_providertemplates import K0rdentProviderTemplateManager
from si_tests.clients.k8s.k0_releases import K0rdentReleaseManager
from si_tests.clients.k8s.k0_regions import K0rdentRegionManager
from si_tests.clients.k8s.k0_managementbackup import K0rdentManagementBackupManager
from si_tests.clients.k8s.m3_baremetalhost import M3BaremetalHostManager
from si_tests.clients.k8s.velero_restore import VeleroRestoreManager
from si_tests.clients.k8s.velero_backup import VeleroBackupManager
from si_tests.clients.k8s.velero_backupstoragelocations import VeleroBackupStorageLocationManager
from si_tests.clients.k8s.clusterrole import K8sClusterRoleManager
from si_tests.clients.k8s.clusterrolebindings import \
    K8sClusterRoleBindingManager
from si_tests.clients.k8s.rolebindings import \
    K8sRoleBindingManager
from si_tests.clients.k8s.componentstatuses import \
    K8sComponentStatusManager
from si_tests.clients.k8s.configmaps import K8sConfigMapManager
from si_tests.clients.k8s.customresourcedefinitions import K8sCustomResourceDefinitionManager
from si_tests.clients.k8s.daemonsets import K8sDaemonSetManager
from si_tests.clients.k8s.deployments import K8sDeploymentManager
from si_tests.clients.k8s.endpoints import K8sEndpointsManager
from si_tests.clients.k8s.events import K8sEventManager
from si_tests.clients.k8s.horizontalpodautoscalers import \
    K8sHorizontalPodAutoscalerManager
from si_tests.clients.k8s.poddisruptionbudget import \
    K8sPodDisruptionBudgetManager
from si_tests.clients.k8s.ingresses import K8sIngressManager
from si_tests.clients.k8s.jobs import K8sJobManager
from si_tests.clients.k8s.cronjob import K8sCronJobManager
from si_tests.clients.k8s.limitranges import K8sLimitRangeManager
from si_tests.clients.k8s.namespaces import K8sNamespaceManager
from si_tests.clients.k8s.networkpolicies import K8sNetworkPolicyManager
from si_tests.clients.k8s.nodes import K8sNodeManager
from si_tests.clients.k8s.opentelemetry_collectors import OpenTelemetryCollectorManager
from si_tests.clients.k8s.persistentvolumeclaims import \
    K8sPersistentVolumeClaimManager
from si_tests.clients.k8s.persistentvolumes import \
    K8sPersistentVolumeManager
from si_tests.clients.k8s.pods import K8sPodManager
from si_tests.clients.k8s.replicasets import K8sReplicaSetManager
from si_tests.clients.k8s.replicationcontrollers import \
    K8sReplicationControllerManager
from si_tests.clients.k8s.resourcequotas import K8sResourceQuotaManager
from si_tests.clients.k8s.secrets import K8sSecretManager
from si_tests.clients.k8s.serviceaccounts import \
    K8sServiceAccountManager
from si_tests.clients.k8s.services import K8sServiceManager
from si_tests.clients.k8s.statefulsets import K8sStatefulSetManager
from si_tests.clients.k8s.updategroups import UpdateGroupManager
from si_tests.clients.k8s.storageclass import K8sStorageClassManager
from si_tests.clients.k8s.leases import LeasesManager
from si_tests.clients.k8s.helmreleases import HelmReleaseManager
from si_tests.clients.k8s.helmrepositories import HelmRepositoryManager
from si_tests.clients.k8s.gitrepositories import GitRepositoryManager
from si_tests.clients.k8s.grafanadashboards import GrafanaDashboardManager
from si_tests.clients.k8s.grafanadatasources import GrafanaDatasourceManager
from si_tests.clients.k8s.argocd_application import ArgocdApplicationManager
from si_tests import logger

LOG = logger.logger


class K8sCluster(object):
    def __init__(self, schema="https", user=None, password=None, ca=None,
                 host='localhost', port='443', default_namespace='default',
                 config_data=None, kubeconfig=None, cluster_name='kaas',
                 idp_ca=None, client_id="kaas", idp_issuer_url=None):
        self.__api_client = None
        self.default_namespace = default_namespace
        self.__schema = schema
        self.__user = user
        self.__password = password
        self.__ca = ca
        self.__host = host
        self.__port = port
        self.__config_data = config_data
        self.__kubeconfig = kubeconfig
        self.__cluster_name = cluster_name
        self.__idp_ca = idp_ca
        self.__client_id = client_id
        self.__idp_issuer_url = idp_issuer_url

        self.login()
        self.init_apis()
        self.init_managers()

    def update_kubeconfig(self, kubeconfig):
        """Update path to the KUBECONFIG file and relogin"""
        self.__kubeconfig = kubeconfig
        self.login()
        self.init_apis()
        self.init_managers()

    def login(self):
        if self.__kubeconfig:
            # Load original kubeconfig
            with open(self.__kubeconfig, 'r') as f:
                kubeconfig_data = yaml.safe_load(f.read())
            # Patch kubeconfig
            for user in kubeconfig_data.get('users', []):
                auth_provider = user.get('user', {}).get('auth-provider', {})
                if auth_provider.get('name', '') == 'oidc':
                    auth_provider_config = auth_provider.get('config')
                    if auth_provider_config and 'client-secret' not in auth_provider_config:
                        LOG.warning(f"Patching 'kubeconfig' OIDC config for user {user.get('name')}")
                        auth_provider_config['client-secret'] = 'si-fake-secret'
            # Init kubernetes client using patched kubeconfig
            with tempfile.NamedTemporaryFile(mode="w") as tmp:
                tmp.write(yaml.safe_dump(kubeconfig_data))
                tmp.flush()
                kubernetes.config.load_kube_config(config_file=tmp.name)
            self.__api_client = client.ApiClient()
        else:
            configuration = type.__call__(client.Configuration)
            config_data = self.__config_data
            loader = kubernetes.config.kube_config.KubeConfigLoader(
                config_data
            )
            LOG.debug("Initialized kubeconfig: {}".format(
                json.dumps(config_data, indent=4)))
            loader.load_and_set(configuration)
            self.__api_client = client.ApiClient(configuration=configuration)

    def init_apis(self):
        self.api_core = client.CoreV1Api(self.api_client)
        self.api_apps = client.AppsV1Api(self.api_client)
        self.api_custom = client.CustomObjectsApi(self.api_client)
        self.api_autoscaling = client.AutoscalingV1Api(self.api_client)
        self.api_batch = client.BatchV1Api(self.api_client)
        self.api_batch_v1beta1 = client.BatchApi(self.api_client)
        self.api_policy_v1beta1 = client.PolicyApi(self.api_client)
        self.api_rbac_auth = client.RbacAuthorizationV1Api(self.api_client)
        self.api_version = client.VersionApi(self.api_client)
        self.api_apis = client.ApisApi(self.api_client)
        self.api_admission = client.AdmissionregistrationV1Api(self.api_client)
        self.api_storage = client.StorageV1Api(self.api_client)
        self.api_networking = client.NetworkingV1Api(self.api_client)
        self.api_apiextensions_v1 = client.ApiextensionsV1Api(self.api_client)

    def init_managers(self):
        self.awsclusters = AWSClusterManager(self)
        self.awsmanagedclusters = AWSManagedClusterManager(self)
        self.awsclusterstaticidentities = AWSClusterStaticIdentityManager(self)
        self.azureclusters = AzureClusterManager(self)
        self.azureasomanagedclusters = AzureASOManagedClusterManager(self)
        self.metal3clusters = Metal3ClusterManager(self)
        self.remoteclusters = RemoteClusterManager(self)
        self.azureasomanagedmachinepools = AzureASOManagedMachinePoolManager(self)
        self.azureclusteridentities = AzureClusterIdentityManager(self)
        self.vsphereclusters = VSphereClusterManager(self)
        self.gcpclusters = GCPClusterManager(self)
        self.gcpmanagedclusters = GCPManagedClusterManager(self)
        self.gcpmanagedcontrolplanes = GCPManagedControlPlaneManager(self)
        self.gcpmanagedmachinepools = GCPManagedMachinePoolManager(self)
        self.vsphereclusteridentities = VSphereClusterIdentityManager(self)
        self.openstackclusters = OpenStackClusterManager(self)
        self.openstackclusteridentities = OpenStackClusterIdentityManager(self)
        self.capi_clusters = ClusterManager(self)
        self.capi_machines = MachineManager(self)
        self.capi_infrastructureproviders = InfrastructureProviderManager(self)
        self.capi_machinepools = MachinePoolManager(self)
        self.crds = K8sCustomResourceDefinitionManager(self)
        self.nodes = K8sNodeManager(self)
        self.pods = K8sPodManager(self)
        self.endpoints = K8sEndpointsManager(self)
        self.namespaces = K8sNamespaceManager(self)
        self.services = K8sServiceManager(self)
        self.serviceaccounts = K8sServiceAccountManager(self)
        self.secrets = K8sSecretManager(self)
        self.statefulsets = K8sStatefulSetManager(self)
        self.events = K8sEventManager(self)
        self.limitranges = K8sLimitRangeManager(self)
        self.jobs = K8sJobManager(self)
        self.cronjobs = K8sCronJobManager(self)
        self.daemonsets = K8sDaemonSetManager(self)
        self.ingresses = K8sIngressManager(self)
        self.deployments = K8sDeploymentManager(self)
        self.horizontalpodautoscalers = K8sHorizontalPodAutoscalerManager(self)
        self.componentstatuses = K8sComponentStatusManager(self)
        self.resourcequotas = K8sResourceQuotaManager(self)
        self.replicationcontrollers = K8sReplicationControllerManager(self)
        self.pvolumeclaims = K8sPersistentVolumeClaimManager(self)
        self.pvolumes = K8sPersistentVolumeManager(self)
        self.replicasets = K8sReplicaSetManager(self)
        self.networkpolicies = K8sNetworkPolicyManager(self)
        self.opentelemetry_collectors = OpenTelemetryCollectorManager(self)
        self.clusterrole = K8sClusterRoleManager(self)
        self.clusterrolebindings = K8sClusterRoleBindingManager(self)
        self.rolebindings = K8sRoleBindingManager(self)
        self.configmaps = K8sConfigMapManager(self)
        self.pdb = K8sPodDisruptionBudgetManager(self)
        self.storageclass = K8sStorageClassManager(self)
        self.leases = LeasesManager(self)
        self.updategroups = UpdateGroupManager(self)
        self.k0rdent_credentials = K0rdentCredentialManager(self)
        self.k0rdent_cluster_deployments = K0rdentClusterDeploymentManager(self)
        self.k0rdent_cluster_templates = K0rdentClusterTemplateManager(self)
        self.k0rdent_managements = K0rdentManagementManager(self)
        self.k0rdent_service_templates = K0rdentServiceTemplateManager(self)
        self.k0rdent_multi_cluster_services = K0rdentMultiClusterServiceManager(self)
        self.k0rdent_provider_interfaces = K0rdentProviderInterfaceManager(self)
        self.k0rdent_provider_templates = K0rdentProviderTemplateManager(self)
        self.k0rdent_releases = K0rdentReleaseManager(self)
        self.k0rdent_regions = K0rdentRegionManager(self)
        self.k0rdent_management_backups = K0rdentManagementBackupManager(self)
        self.metal3_baremetalhosts = M3BaremetalHostManager(self)
        self.helmreleases = HelmReleaseManager(self)
        self.helmrepositories = HelmRepositoryManager(self)
        self.gitrepositories = GitRepositoryManager(self)
        self.grafanadashboards = GrafanaDashboardManager(self)
        self.grafanadatasources = GrafanaDatasourceManager(self)
        self.argocd_applications = ArgocdApplicationManager(self)
        self.velero_restores = VeleroRestoreManager(self)
        self.velero_backups = VeleroBackupManager(self)
        self.velero_backupstoragelocations = VeleroBackupStorageLocationManager(self)

    @property
    def api_client(self):
        return self.__api_client
