import time
from retry import retry

from si_tests.clients.gcore.box_api import Configuration
from si_tests.clients.gcore.box_api import ApiClient
# APIs for box-api endpoints
from si_tests.clients.gcore.box_api import APIKeysApi
from si_tests.clients.gcore.box_api import AppsApi
from si_tests.clients.gcore.box_api import AuditLogsApi
from si_tests.clients.gcore.box_api import AutodiscoveryApi
from si_tests.clients.gcore.box_api import CapacityApi
from si_tests.clients.gcore.box_api import DashboardApi
from si_tests.clients.gcore.box_api import FlavorsApi
from si_tests.clients.gcore.box_api import FlavorsManagementApi
from si_tests.clients.gcore.box_api import InferencesApi
from si_tests.clients.gcore.box_api import MetricsApi
from si_tests.clients.gcore.box_api import NamespacesApi
from si_tests.clients.gcore.box_api import NodeGroupsApi
from si_tests.clients.gcore.box_api import NodesApi
from si_tests.clients.gcore.box_api import PodsApi
from si_tests.clients.gcore.box_api import ProjectsApi
from si_tests.clients.gcore.box_api import PullSecretsApi
from si_tests.clients.gcore.box_api import QuotasApi
from si_tests.clients.gcore.box_api import RegionsApi
from si_tests.clients.gcore.box_api import RegistriesApi
from si_tests.clients.gcore.box_api import RegistryUsersApi
from si_tests.clients.gcore.box_api import SecretsApi
from si_tests.clients.gcore.box_api import SlurmApi
from si_tests.clients.gcore.box_api import TLSSecretsApi
from si_tests.clients.gcore.box_api import UserGroupsApi
from si_tests.clients.gcore.box_api import UsersApi
from si_tests.clients.gcore.box_api import VolumesApi
from si_tests.clients.gcore.box_api import V1AccessRule
from si_tests.clients.gcore.box_api import V1CreateInferenceRequest
from si_tests.clients.gcore.box_api import V1UpdateInferenceRequest
from si_tests.clients.gcore.box_api import V1CreateProjectRequest
from si_tests.clients.gcore.box_api import V1CreateGroupRequest
from si_tests.clients.gcore.box_api import V1CreateUserRequest
from si_tests.clients.gcore.box_api import V1CreateRegistryRequest
from si_tests.clients.gcore.box_api import V1CreateRegistryUserRequest
from si_tests.clients.gcore.box_api import V1CreateApiKeyRequest
from si_tests.clients.gcore.box_api import V1UpdateApiKeyRequest
from si_tests.clients.gcore.box_api import V1CreateFlavorRequest
from si_tests.clients.gcore.box_api import V1CreateVolumeRequest
from si_tests.clients.gcore.box_api import V1EmptyDirVolumeSource
from si_tests.clients.gcore.box_api import V1ImageVolumeSource
from si_tests.clients.gcore.box_api import V1PVCVolumeSource
from si_tests.clients.gcore.box_api import V1CreateNodeGroupRequest
from si_tests.clients.gcore.box_api import ApiServicesInferenceV1Volume
from si_tests.clients.gcore.box_api import V1RegistryRef
from si_tests.clients.gcore.box_api import V1RegistryImageRef
from si_tests.clients.gcore.box_api import V1RegistryUserRef
from si_tests.clients.gcore.box_api import SchemasAppDeploymentCreateRequest
from si_tests.clients.keycloak import keycloak_client
from si_tests import settings
from si_tests import logger

LOG = logger.logger


class BoxApiClient(object):

    def __init__(self,
                 keycloak_url,
                 box_api_url,
                 username,
                 password,
                 client_id=settings.KSI_INFERENCE_BOX_KEYCLOAK_CLIENT_ID,
                 realm_name=settings.KSI_INFERENCE_BOX_KEYCLOAK_REALM_NAME,
                 verify=False):
        self.keycloak_url = keycloak_url
        self.box_api_url = box_api_url
        self.username = username
        self.password = password
        self.client_id = client_id
        self.realm_name = realm_name
        self.verify = verify
        self.__keycloak_client = None
        self.__token = None
        self.id_token = None
        self.init_apis()

    def get_api_client(self):
        """Returns a client for box-api"""
        config = Configuration(host=self.box_api_url)
        config.verify_ssl = self.verify
        self.api_client = ApiClient(configuration=config)

        # replace rest_client.request with a hook that will refresh openid token
        self.__orig_request = self.api_client.rest_client.request
        self.api_client.rest_client.request = self.__request_hook
        return self.api_client

    @retry(Exception, delay=2, tries=3, logger=LOG)
    def __request_hook(self, *args, **kwargs):
        """Hook for api client, to refresh openid token before api calls"""
        id_token = self.get_id_token()
        kwargs['headers']["Authorization"] = f"Bearer {id_token}"
        return self.__orig_request(*args, **kwargs)

    def get_id_token(self):
        """Authorize with username/password and get or refresh openid token"""
        if self.__token is None or self._login_is_expired():
            LOG.info(f"Get new OpenID auth token for {self.keycloak_url}")
            self.__token = self.keycloak_client.get_openid_token()
            self._last_login_time = int(time.time())
            self._last_refresh_time = int(time.time())
            self.login_expire = self.__token.get('refresh_expires_in', 0)
            self.refresh_expire = self.__token.get('expires_in', 0)
            self.id_token = self.__token['id_token']
            LOG.debug(f"OpenID token data:\n{self.__token}")
            LOG.info(f"OpenID login expires in: {self.login_expire}")
            LOG.info(f"OpenID token refresh in: {self.refresh_expire}")

        elif self._refresh_is_expired():
            LOG.info(f"Refresh OpenID auth token for {self.keycloak_url}")
            self.__token = self.keycloak_client.refresh_token(self.__token)
            self._last_refresh_time = int(time.time())
            self.id_token = self.__token['id_token']
            LOG.debug(f"OpenID token data:\n{self.__token}")

        return self.id_token

    def _login_is_expired(self):
        buffer = 30  # seconds before actual expiration
        return (time.time() - self._last_login_time) >= (self.login_expire - buffer)

    def _refresh_is_expired(self):
        buffer = 30  # seconds before actual expiration
        return (time.time() - self._last_refresh_time) >= (self.refresh_expire - buffer)

    @property
    def keycloak_client(self):
        if self.__keycloak_client is None:
            self.__keycloak_client = keycloak_client.KeycloakUserClient(self.keycloak_url,
                                                                        self.username,
                                                                        self.password,
                                                                        client_id=self.client_id,
                                                                        realm_name=self.realm_name)
        return self.__keycloak_client

    def init_apis(self):
        """Init all APIs for box-api endpoints"""
        api_client = self.get_api_client()
        self.api_keys = APIKeysApi(api_client)
        # Integrate V1CreateApiKeyRequest and V1UpdateApiKeyRequest into 'api_keys'
        self.api_keys.V1CreateApiKeyRequest = V1CreateApiKeyRequest
        self.api_keys.V1UpdateApiKeyRequest = V1UpdateApiKeyRequest
        self.apps = AppsApi(api_client)
        # Integrate SchemasAppDeploymentCreateRequest into 'apps'
        self.apps.SchemasAppDeploymentCreateRequest = SchemasAppDeploymentCreateRequest
        self.audit_logs = AuditLogsApi(api_client)
        self.autodiscovery = AutodiscoveryApi(api_client)
        self.capacity = CapacityApi(api_client)
        self.dashboard = DashboardApi(api_client)
        self.flavors = FlavorsApi(api_client)
        self.flavorsmanagement = FlavorsManagementApi(api_client)
        # Integrate V1CreateFlavorRequest into 'flavors'
        self.flavorsmanagement.V1CreateFlavorRequest = V1CreateFlavorRequest
        self.inferences = InferencesApi(api_client)
        # Integrate extra objects into 'inferences'
        self.inferences.ApiServicesInferenceV1Volume = ApiServicesInferenceV1Volume
        self.inferences.V1CreateInferenceRequest = V1CreateInferenceRequest
        self.inferences.V1UpdateInferenceRequest = V1UpdateInferenceRequest
        self.inferences.V1RegistryRef = V1RegistryRef
        self.inferences.V1RegistryImageRef = V1RegistryImageRef
        self.inferences.V1RegistryUserRef = V1RegistryUserRef
        self.metrics = MetricsApi(api_client)
        self.namespaces = NamespacesApi(api_client)  # Deprecated, use 'projects' instead
        self.node_groups = NodeGroupsApi(api_client)
        # Integrate V1CreateNodeGroupRequest into 'node_groups'
        self.node_groups.V1CreateNodeGroupRequest = V1CreateNodeGroupRequest
        self.nodes = NodesApi(api_client)
        self.pods = PodsApi(api_client)
        self.projects = ProjectsApi(api_client)
        # Integrate V1CreateProjectRequest into 'projects'
        self.projects.V1CreateProjectRequest = V1CreateProjectRequest
        self.pull_secrets = PullSecretsApi(api_client)
        self.quotas = QuotasApi(api_client)
        self.regions = RegionsApi(api_client)
        self.registries = RegistriesApi(api_client)
        # Integrate V1CreateRegistryRequest into 'registries'
        self.registries.V1CreateRegistryRequest = V1CreateRegistryRequest
        self.registry_users = RegistryUsersApi(api_client)
        # Integrate V1CreateRegistryUserRequest into 'registry_users'
        self.registry_users.V1CreateRegistryUserRequest = V1CreateRegistryUserRequest
        self.secrets_api = SecretsApi(api_client)
        self.slurm = SlurmApi(api_client)
        self.tls_secrets = TLSSecretsApi(api_client)
        self.user_groups = UserGroupsApi(api_client)
        # Integrate V1CreateGroupRequest and V1AccessRule into 'user_groups'
        self.user_groups.V1CreateGroupRequest = V1CreateGroupRequest
        self.user_groups.V1AccessRule = V1AccessRule
        self.users = UsersApi(api_client)
        # Integrate V1CreateUserRequest and V1AccessRule into 'users'
        self.users.V1CreateUserRequest = V1CreateUserRequest
        self.users.V1AccessRule = V1AccessRule
        self.volumes = VolumesApi(api_client)
        # Integrate V1CreateVolumeRequest into 'volumes'
        self.volumes.V1CreateVolumeRequest = V1CreateVolumeRequest
        self.volumes.V1EmptyDirVolumeSource = V1EmptyDirVolumeSource
        self.volumes.V1ImageVolumeSource = V1ImageVolumeSource
        self.volumes.V1PVCVolumeSource = V1PVCVolumeSource
