import pytest

from si_tests import logger
from si_tests import settings
from qase.pytest import qase

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@qase.title("Test Restore of KCM from Backup")
@qase.suite("Bootstrap")
def test_restore(kcm_bootstrap_manager, show_step):
    """Restore KCM from Velero backup

    Scenario:
        1. Install k0rdent enterprise with restore configuration
        2. Create Velero AWS credentials secret
        3. Create backup storage location
        4. Wait for backup storage location to be ready
        5. Wait for Velero to reconcile existing backups
        6. Apply restore from specified backup
        7. Wait for restore completion
        8. Verify management objects are restored
    """

    # Validate required settings
    assert settings.AWS_ACCESS_KEY_ID, "AWS_ACCESS_KEY_ID is required"
    assert settings.AWS_SECRET_ACCESS_KEY, "AWS_SECRET_ACCESS_KEY is required"
    assert settings.KSI_VELERO_BACKUP_NAME, "KSI_VELERO_BACKUP_NAME is required"
    assert settings.KSI_VELERO_BACKUP_STORAGE_BUCKET, "KSI_VELERO_BACKUP_STORAGE_BUCKET is required"

    show_step(1)
    LOG.info("Installing k0rdent enterprise with restore configuration")
    kcm_bootstrap_manager.install_kcm(mode="restore")

    # Get namespace manager
    kcm_ns = kcm_bootstrap_manager.kcm_mgr.get_or_create_namespace(settings.KCM_NAMESPACE)

    show_step(2)
    LOG.info("Creating Velero AWS credentials secret")
    secret = kcm_ns.create_velero_aws_secret()

    show_step(3)
    LOG.info(f"Creating backup storage location with bucket: {settings.KSI_VELERO_BACKUP_STORAGE_BUCKET}")
    bls = kcm_ns.create_backup_storage_location(secret)

    show_step(4)
    LOG.info("Waiting for backup storage location to be ready")
    bls.wait_for_availability()

    show_step(5)
    LOG.info("Waiting for Velero to reconcile existing backups")
    # Wait for Velero backups to be synced using the check function
    kcm_bootstrap_manager.check.wait_velero_backups_sync(
        backup_name=settings.KSI_VELERO_BACKUP_NAME,
        timeout=120
    )

    show_step(6)
    LOG.info(f"Applying restore from backup: {settings.KSI_VELERO_BACKUP_NAME}")
    restore = kcm_ns.apply_restore(
        restore_name=settings.KSI_VELERO_RESTORE_NAME,
        backup_name=settings.KSI_VELERO_BACKUP_NAME
    )

    show_step(7)
    LOG.info("Waiting for restore to complete")
    restore.wait_for_completion()

    show_step(8)
    LOG.info("Verifying management objects are restored")
    kcm_bootstrap_manager.check.wait_for_mgmt_object_readiness()
    kcm_bootstrap_manager.check.check_k8s_pods()
    kcm_bootstrap_manager.check.check_actual_expected_pods()

    LOG.info("Restore completed successfully!")
