import pytest
import requests

from si_tests import settings
from si_tests import logger
# from si_tests.managers.harbor_manager import HarborManager

LOG = logger.logger


# Global variables to store test resources for dependency chain
test_resources = {
    'box_cluster_name': settings.KSI_INFERENCE_BOX_CLUSTER_NAME,
    'project_name': settings.KSI_INFERENCE_TEST_PROJECT_NAME,
    'nodegroup_name': settings.KSI_INFERENCE_TEST_NODEGROUP_NAME,
    'usergroup_name': settings.KSI_INFERENCE_TEST_USERGROUP_NAME,
    'user_name': settings.KSI_INFERENCE_TEST_USER_NAME,

    'apikey_name': settings.KSI_INFERENCE_TEST_APIKEY_NAME,
    'app_model_flavor_name': settings.KSI_INFERENCE_TEST_FLAVOR_NAME,
    'app_model_flavor_cpu': settings.KSI_INFERENCE_TEST_FLAVOR_CPU,
    'app_model_flavor_memory': settings.KSI_INFERENCE_TEST_FLAVOR_MEMORY,
    'app_model_flavor_gpu': settings.KSI_INFERENCE_TEST_FLAVOR_GPU or None,
    'app_model_flavor_gpu_memory': settings.KSI_INFERENCE_TEST_FLAVOR_GPU_MEMORY or None,
    'app_model_flavor_gpu_model': settings.KSI_INFERENCE_TEST_FLAVOR_GPU_MODEL or None,
    'app_model_flavor_is_gpu_shared': settings.KSI_INFERENCE_TEST_FLAVOR_IS_GPU_SHARED,

    'app_ui_flavor_name': 'test-flavor-1vcpu-2gib',
    'app_ui_flavor_cpu': '1',
    'app_ui_flavor_memory': '2Gi',
    'app_ui_flavor_gpu': '0',
    'app_ui_flavor_gpu_memory': '0',
    'app_ui_flavor_gpu_model': None,
    'app_ui_flavor_is_gpu_shared': False,

    'mlapp_name': settings.KSI_INFERENCE_TEST_MLAPP_NAME,
    'mlapptemplate_name': settings.KSI_INFERENCE_TEST_MLAPPTEMPLATE_NAME,

    # Resources that get created during test execution
    'box_api_client': None,
    'inference_manager': None,
    'inference_user_manager': None,
    'usergroup': None,
    'user': None,
    'user_password': None,
    'apikey': None,
    'project': None,
    # 'nodegroup': None,            # nodegroups are unsupported for mlapps in box 1.0.1
    'model_flavor': None,
    'ui_flavor': None,
    'application': None,
    # 'inference_disabled': False,  # disable(pause) is unsupported for mlapps in box 1.0.1
    'inferences': None
}


@pytest.fixture(scope='session')
def setup_test_resources(box_api_client, inference_manager):
    """Setup basic test resources - step 1"""
    # Store the clients for reuse
    test_resources['box_api_client'] = box_api_client
    test_resources['inference_manager'] = inference_manager


@pytest.mark.usefixtures('setup_test_resources')
def test_step_1_show_available_resources(inference_manager):
    """Step 1: Show available resources from box-api"""
    LOG.banner("=== Step 1: Show available resources ===")
    # This is a simple check that the API works
    try:
        inference_manager.show_capacity()
        inference_manager.show_gpu_nodes()
        LOG.info("* Available resources shown successfully")
    except Exception as e:
        pytest.fail(f"Failed to show available resources: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_2_get_or_create_project(inference_manager):
    """Step 2: Get or create a Project"""
    LOG.banner("=== Step 2: Get or create a Project ===")
    try:
        project = inference_manager.get_or_create_project(test_resources['project_name'])
        test_resources['project'] = project
        LOG.info(f"* Project '{test_resources['project_name']}' created/retrieved successfully")
    except Exception as e:
        pytest.fail(f"Failed to get or create project: {str(e)}")
    inference_manager.show_projects()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_3_create_model_flavor(inference_manager):
    """Step 3: Create a Flavor to run a model on GPU"""
    LOG.banner("=== Step 3: Create a model Flavor ===")
    try:
        flavor = inference_manager.get_or_create_flavor(
            flavor_name=test_resources['app_model_flavor_name'],
            cpu=test_resources['app_model_flavor_cpu'],
            memory=test_resources['app_model_flavor_memory'],
            gpu=test_resources['app_model_flavor_gpu'],
            gpu_memory=test_resources['app_model_flavor_gpu_memory'],
            gpu_model=test_resources['app_model_flavor_gpu_model'],
            is_gpu_shared=test_resources['app_model_flavor_is_gpu_shared']
        )
        test_resources['model_flavor'] = flavor
        LOG.info("* Model Flavor created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create model flavor: {str(e)}")
    inference_manager.show_flavors()
    inference_manager.show_capacity()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_4_create_ui_flavor(inference_manager):
    """Step 4: Create a Flavor to run a UI application"""
    LOG.banner("=== Step 4: Create an UI Flavor ===")
    try:
        flavor = inference_manager.get_or_create_flavor(
            flavor_name=test_resources['app_ui_flavor_name'],
            cpu=test_resources['app_ui_flavor_cpu'],
            memory=test_resources['app_ui_flavor_memory'],
            gpu=test_resources['app_ui_flavor_gpu'],
            gpu_memory=test_resources['app_ui_flavor_gpu_memory'],
            gpu_model=test_resources['app_ui_flavor_gpu_model'],
            is_gpu_shared=test_resources['app_ui_flavor_is_gpu_shared']
        )
        test_resources['ui_flavor'] = flavor
        LOG.info("* UI Flavor created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create UI flavor: {str(e)}")
    inference_manager.show_flavors()
    inference_manager.show_capacity()

# # TODO(ddmitriev): uncomment test to create a NodeGroup, when mlappdeployment spec will support NodeGroups
# @pytest.mark.usefixtures('setup_test_resources')
# def test_step_5_create_nodegroup(inference_manager):
#     """Step 5: Create a NodeGroup for all available GPU nodes"""
#     LOG.banner("=== Step 5: Create a NodeGroup ===")
#     if not test_resources['project']:
#         LOG.error("*** SKIP: Required: Project")
#         pytest.skip("Required: Project")
#     try:
#         gpu_nodes = inference_manager.list_gpu_nodes()
#         # NodeGroup nodes: { region1_name: [node1_1, node1_2, node1_3],
#         #                    region2_name: [node2_1, node2_2, node2_3], ...}
#         nodegroup_nodes = {}
#         for gpu_node in gpu_nodes:
#             if gpu_node.region_name not in nodegroup_nodes:
#                 nodegroup_nodes[gpu_node.region_name] = []
#             nodegroup_nodes[gpu_node.region_name].append(gpu_node.name)
#         # custom_labels = {
#         #     'aaa': 'bbb'
#         # }
#         nodegroup = inference_manager.get_or_create_nodegroup(
#             nodegroup_name=test_resources['nodegroup_name'],
#             project_names=[test_resources['project_name']],
#             node_refs=nodegroup_nodes,
#         #   custom_labels=custom_labels,
#         )
#         test_resources['nodegroup'] = nodegroup
#         LOG.info("* NodeGroup created successfully")
#     except Exception as e:
#         pytest.fail(f"Failed to create NodeGroup: {str(e)}")
#     inference_manager.show_nodegroups()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_6_get_or_create_usergroup(inference_manager):
    """Step 6: Get or create a new UserGroup"""
    LOG.banner("=== Step 6: Get or create a new UserGroup ===")
    if not test_resources['project']:
        LOG.error("*** SKIP: Required: Project")
        pytest.skip("Required: Project")

    try:
        usergroup = inference_manager.get_or_create_usergroup(
            test_resources['usergroup_name'],
            test_resources['project_name']
        )
        test_resources['usergroup'] = usergroup
        LOG.info(f"* UserGroup '{test_resources['usergroup_name']}' created/retrieved successfully")
    except Exception as e:
        pytest.fail(f"Failed to get or create usergroup: {str(e)}")
    inference_manager.show_usergroups()


@pytest.mark.usefixtures('setup_test_resources')
def test_step_7_create_user(inference_manager):
    """Step 7: Create a new Box User"""
    LOG.banner("=== Step 7: Create a new User ===")
    if not test_resources['usergroup']:
        LOG.error("*** SKIP: Required: UserGroup")
        pytest.skip("Required: UserGroup")

    try:
        user = inference_manager.get_or_create_user(
            test_resources['user_name'],
            [test_resources['usergroup_name']],
            test_resources['project_name']
        )
        test_resources['user'] = user
        test_resources['user_password'] = inference_manager.get_user_password(test_resources['user_name'])
        LOG.info(f"* User '{test_resources['user_name']}' created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create user: {str(e)}")
    inference_manager.show_users()
    LOG.info(f"===== USER PASSWORD: {test_resources['user_password']}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_8_init_new_box_api_client(box_api_client, inference_manager):
    """Step 8: Init new box-api client with User credentials"""
    LOG.banner("=== Step 8: Init new box-api client with User credentials ===")
    if not test_resources['user'] or not test_resources['user_password']:
        LOG.error("*** SKIP: Required: Box User credentials")
        pytest.skip("Required: Box User credentials")

    try:
        user_client = box_api_client.__class__(
            keycloak_url=box_api_client.keycloak_url,
            box_api_url=box_api_client.box_api_url,
            username=test_resources['user_name'],
            password=test_resources['user_password'],
            client_id=box_api_client.client_id,
            realm_name=box_api_client.realm_name,
            verify=box_api_client.verify
        )
        test_resources['inference_user_manager'] = inference_manager.__class__(box_api_client=user_client)
        LOG.info("* New box-api client initialized successfully for test User")
    except Exception as e:
        pytest.fail(f"Failed to initialize new box-api client for test User: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_9_create_apikey(inference_manager):
    """Step 9: Create an APIKey (from User)"""
    LOG.banner("=== Step 9: Create an APIKey ===")
    if not test_resources['inference_user_manager']:
        LOG.error("*** SKIP: Required: Box User manager")
        pytest.skip("Required: Box User manager")

    try:
        apikey = test_resources['inference_user_manager'].create_or_recreate_apikey(
            test_resources['apikey_name'],
            test_resources['project_name']
        )
        test_resources['apikey'] = apikey
        LOG.info(f"* APIKey '{test_resources['apikey_name']}' created successfully")
    except Exception as e:
        pytest.fail(f"Failed to create APIKey: {str(e)}")
    test_resources['inference_user_manager'].show_apikeys(test_resources['project_name'])
    LOG.info(f"====== APIKEY '{test_resources['apikey_name']}' secret value: '{test_resources['apikey'].secret}'")

# TODO(ddmitriev): Add registry cases for apps when MLApps will have customization
#                  for the app images sources (vllm image, model image, webui image),
#                  to run all the App with images from the local Registry


@pytest.mark.usefixtures('setup_test_resources')
def test_step_10_create_application(inference_manager, kcm_manager):
    """Step 10: Create MLApp object using the created User, Flavor, NodeGroup, ApiKey"""
    LOG.banner("=== Step 10: Create Application ===")

    if not test_resources['project']:
        LOG.error("*** SKIP: Required: Project")
        pytest.skip("Required: Project")
    if not test_resources['model_flavor']:
        LOG.error("*** SKIP: Required: model Flavor")
        pytest.skip("Required: model Flavor")
    if not test_resources['ui_flavor']:
        LOG.error("*** SKIP: Required: ui Flavor")
        pytest.skip("Required: ui Flavor")
    if not test_resources['inference_user_manager']:
        LOG.error("*** SKIP: Required: Box User manager")
        pytest.skip("Required: Box User manager")
    if not test_resources['apikey']:
        LOG.error("*** SKIP: Required: ApiKey")
        pytest.skip("Required: ApiKey")
    try:
        # Create inference using the volume and registry image

        components_configuration = {
            "model": {
              "exposed": True,
              "flavor": test_resources['app_model_flavor_name'],
              "parameter_overrides": {},
              "scale": {
                "min": 1,
                "max": 1,
              },
            },
            "ui": {
              "exposed": True,
              "flavor": test_resources['app_ui_flavor_name'],
              "parameter_overrides": {},
              "scale": {
                "min": 1,
                "max": 1,
              },
            },
        }

        mlapp = test_resources['inference_user_manager'].create_application(
            project_name=test_resources['project_name'],
            mlapp_name=test_resources['mlapp_name'],
            mlapptemplate_name=test_resources['mlapptemplate_name'],
            inference_regions=[test_resources['box_cluster_name']],
            api_keys=[test_resources['apikey_name']],
            components_configuration=components_configuration,
            dry_run=None)

        test_resources['application'] = mlapp
        LOG.info("* Application created successfully")

        LOG.info("* Wait for Inferences created by the application")
        test_resources['inference_user_manager'].wait_inferences_with_mlapp(
            project_name=test_resources['project_name'],
            mlapp_name=test_resources['mlapp_name'])

        inferences = test_resources['inference_user_manager'].get_inferences_by_mlapp(
            project_name=test_resources['project_name'],
            mlapp_name=test_resources['mlapp_name'])

        LOG.info("=== Wait for Inferences of the deployed MLAppDeployment to be ready ===")
        ns = kcm_manager.get_namespace(settings.TARGET_NAMESPACE)
        cld = ns.get_cluster_deployment(settings.TARGET_CLD)
#        for inference in inferences:
#            test_resources['inference_user_manager'].wait_inferences_ready(
#                project_name=test_resources['project_name'],
#                inference_names=[inference.name],
#                k8sclient=cld.k8sclient
#            )

        test_resources['inference_user_manager'].wait_inferences_ready(
            project_name=test_resources['project_name'],
            inference_names=[inference.name for inference in inferences],
            k8sclient=cld.k8sclient
        )
        test_resources['inferences'] = inferences
        LOG.info("* Application is ready")
    except Exception as e:
        pytest.fail(f"Failed to create application: {str(e)}")


@pytest.mark.usefixtures('setup_test_resources')
def test_step_11_validate_apikey_access(inference_manager):
    """
    Step 11: Check that APIKey is working while accessing the Inference endpoint,
    and the endpoint is restricted without the APIKey
    """
    LOG.banner("=== Step 11: Validate APIKey access ===")
    if not test_resources['inferences']:
        LOG.error("*** SKIP: Required: Inferences are up and running")
        pytest.skip("Required: Inferences are up and running")

    # Try to access the Inference external address using the ApiKey for authorization
    apikey_secret = test_resources['apikey'].secret

    inference_name = None
    for inference in test_resources['inferences']:
        if inference.api_keys:
            inference_name = inference.name
    assert inference_name, "No Inference objects found with configured 'api_keys' for ingress"

    addresses = test_resources['inference_user_manager'].get_inference_addresses(
        project_name=test_resources['project_name'],
        inference_name=inference_name
    )
    LOG.info(f"* Inference with ApiKey: '{inference_name}'")
    try:

        # Try to access the Inference external address using the ApiKey for authorization
        for region, address in addresses.items():
            LOG.info(f"Testing access to address '{address}' in region '{region}' with API key...")
            response = requests.get(address, headers={'X-API-Key': apikey_secret}, verify=False)

            # Validate that access with valid API key returns 200 (success)
            if response.status_code != 200 and response.status_code != 404:
                pytest.fail(f"Expected 200 or 404 when accessing with valid API key, but got {response.status_code}")

            LOG.info(f"* Access with API key successful: Status code {response.status_code}")
            LOG.info(f"Response content:\n{response.text}")

        # Try to access the same address without ApiKey and ensure access is forbidden
        for region, address in addresses.items():
            LOG.info(f"Testing access to address '{address}' in region '{region}' without API key...")
            response = requests.get(address, verify=False)

            # Validate that access without API key returns 401 or 403 (forbidden/unauthorized)
            if response.status_code not in [401, 403]:
                pytest.fail(f"Expected 401 or 403 when accessing without API key, but got {response.status_code}")

            LOG.info(f"* Access without API key returned expected status code {response.status_code}")

    except Exception as e:
        pytest.fail(f"Failed to validate APIKey access: {str(e)}")


# TODO(ddmitriev): add disable/enable tests when mlappdeployment will support such options
