import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, k8s_utils, kcm_utils

LOG = logger.logger


@pytest.mark.create_cd
@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures('store_cluster_kubeconfig')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KCM_CLUSTER_DEPLOYMENT_NAME}"])
def test_create_azure_hosted_cluster(kcm_manager, show_step, _):
    """Deploy hosted clusterdeployment on Azure

    Scenario:
        1. Collect env data and determine regional/host cluster
        2. Gather host cluster based on environment/region data
        3. Generate unique names and auth parameters
        4. Gather network config of the host cluster for hosted deployment
        5. Create or get NS and check cluster templates presence
        6. Create auth
        7. Prepare clusterdeployment config
        8. Create clusterdeployment
        9. Wait for clusterdeployment progress to finish
        10. Check expected pods, its statuses and k8s nodes
        11. Set default storageclass if not present
        12. Sync CoreDNS Corefile with management cluster
    """
    show_step(1)
    cluster_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    namespace_name = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE
    host_cluster_ns, host_cluster_name = kcm_utils.determine_hosted_parameters(kcm_manager)

    show_step(2)
    host_cluster, host_kcm_manager, cld_manager = kcm_utils.determine_host_clusters(kcm_manager,
                                                                                    host_cluster_ns,
                                                                                    host_cluster_name)

    show_step(3)
    azure_cluster_id_name = f"ksi-azure-cluster-id-{utils.gen_random_string(4)}"
    auth_secret_name = f"{azure_cluster_id_name}-secret"
    cred_name = f"{azure_cluster_id_name}-cred"

    show_step(4)
    # Gathering network config of the host cluster for hosted deployment using mgmt cluster
    LOG.info(f"Gathering network config of the host cluster {host_cluster_ns}/{host_cluster_name} ")
    infracluster = host_cluster.clusterobject.infracluster
    location = infracluster.data['spec']['location']
    subscription_id = infracluster.data['spec']['subscriptionID']
    resource_group = infracluster.data['spec']['resourceGroup']
    vnet_name = infracluster.data['spec']['networkSpec']['vnet']['name']
    node_subnet = [subnet for subnet in infracluster.data['spec']['networkSpec']['subnets']
                   if subnet['name'] == f"{host_cluster_name}-node-subnet"][0]
    node_subnet_name = node_subnet['name']
    route_table_name = node_subnet['routeTable']['name']
    security_group_name = node_subnet['securityGroup']['name']

    show_step(5)
    ns = cld_manager.get_or_create_namespace(namespace_name)
    host_cluster_ns = host_kcm_manager.get_or_create_namespace(host_cluster_ns)

    clustertemplate = (settings.KCM_CLUSTER_TEMPLATE_NAME or
                       ns.get_latest_template_by_prefix('azure-hosted', strategy='chart'))
    available_templates = [tmpl.name for tmpl in ns.get_cluster_templates()]
    assert clustertemplate in available_templates, (f"Clustertemplate {clustertemplate} not found in namespace "
                                                    f"{ns.name}. Available templates: {available_templates}")

    show_step(6)
    auth_sec = ns.create_azure_auth_secret(name=auth_secret_name)
    azure_cluster_id = ns.create_azure_cluster_identity(azure_cluster_id_name, auth_sec)
    cred = ns.create_credential(name=cred_name, cred_object=azure_cluster_id)
    sveltos_template = host_cluster_ns.create_sveltos_resource_template_cm(azure_cluster_id)
    assert sveltos_template.read(), "sveltos resource template not created"

    show_step(7)
    config = ns.get_cluster_config(provider='azure-hosted',
                                   location=location,
                                   subscription_id=subscription_id,
                                   resource_group=resource_group,
                                   vnet_name=vnet_name,
                                   node_subnet_name=node_subnet_name,
                                   route_table_name=route_table_name,
                                   security_group_name=security_group_name)

    show_step(8)
    cld = ns.create_cluster_deployment(name=cluster_name,
                                       template=clustertemplate,
                                       credential=cred,
                                       config=config)
    LOG.info(f"Created clusterdeployment {cld.namespace}/{cld.name}")

    show_step(9)
    cld.check.check_cluster_readiness(expected_condition_cld_fails={'Paused': ''})

    show_step(10)
    cld.check.check_k8s_nodes()
    cld.check.check_k8s_pods()
    cld.check.check_actual_expected_pods()

    show_step(11)
    if not cld.is_default_storageclass_present():
        sc = cld.create_azuredisk_storageclass()
        sc.make_default()
        assert sc.is_default, f"StorageClass {sc.name} is not default"

    show_step(12)
    # TODO(va4st): Need a better way to determine the needs for CoreDNS sync (e.g some marker in mgmt config maybe?)
    if (not cld_manager.kof.kof_is_deployed() and cld_manager.kof.kof_ms_deployed()
            and cld_manager.source == 'custom-enterprise'):
        corefile = k8s_utils.get_corefile(cld_manager.api)
        k8s_utils.override_coredns_configmap(cld.k8sclient, corefile)
        coredns_deployment = cld.k8sclient.deployments.get(name='coredns', namespace='kube-system')
        coredns_deployment.rollout_restart()
