import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, k8s_utils, kcm_utils

LOG = logger.logger


@pytest.mark.create_cd
@pytest.mark.aws
@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures('store_cluster_kubeconfig')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KCM_CLUSTER_DEPLOYMENT_NAME}"])
def test_create_aws_standalone_cluster(kcm_manager, show_step, _):
    """Deploy managed clusterdeployment on AWS

    Scenario:
        1. Collect env data and determine regional cluster if used
        2. Gather host cluster based on environment/region data
        3. Generate unique names and auth parameters
        4. Create or get NS and check cluster templates presence
        5. Create auth
        6. Prepare clusterdeployment config
        7. Create clusterdeployment
        8. Wait for clusterdeployment progress to finish
        9. Check expected pods, its statuses and k8s nodes
        10. Sync CoreDNS Corefile with management cluster
    """
    show_step(1)
    # TODO(va4st): Move to separated test with auth actions, determine auth parameters dynamically
    cluster_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    namespace_name = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE

    show_step(2)
    host_kcm_manager = kcm_utils.determine_regional_objects(kcm_manager)

    show_step(3)
    aws_static_id_name = f"ksi-aws-auth-static-{utils.gen_random_string(4)}"
    auth_secret_name = f"{aws_static_id_name}-secret"
    cred_name = f"{aws_static_id_name}-cred"

    show_step(4)
    ns = kcm_manager.get_or_create_namespace(namespace_name)
    host_cluster_ns = host_kcm_manager.get_or_create_namespace(namespace_name)

    clustertemplate = (settings.KCM_CLUSTER_TEMPLATE_NAME or
                       ns.get_latest_template_by_prefix('aws-standalone-cp', strategy='chart'))
    available_templates = [tmpl.name for tmpl in ns.get_cluster_templates()]
    assert clustertemplate in available_templates, (f"Clustertemplate {clustertemplate} not found in namespace "
                                                    f"{ns.name}. Available templates: {available_templates}")

    show_step(5)
    auth_sec = ns.create_aws_auth_secret(name=auth_secret_name)
    aws_static_id = ns.create_aws_cluster_static_identity(aws_static_id_name, auth_sec)
    cred = ns.create_credential(name=cred_name, cred_object=aws_static_id)
    sveltos_template = host_cluster_ns.create_sveltos_resource_template_cm(aws_static_id)
    assert sveltos_template.read(), "sveltos resource template not created"

    show_step(6)
    # TODO(va4st): make configurable through settings and add parameters verification before testrun
    config = ns.get_cluster_config(provider='aws')

    child_labels = kcm_manager.kof.get_child_labels()
    config['clusterAnnotations'] = kcm_manager.kof.get_child_annotations()

    show_step(7)
    cld = ns.create_cluster_deployment(name=cluster_name,
                                       template=clustertemplate,
                                       credential=cred,
                                       config=config,
                                       labels=child_labels)
    LOG.info(f"Created clusterdeployment {cld.namespace}/{cld.name}")

    show_step(8)
    cld.check.check_cluster_readiness(expected_condition_cld_fails={'Paused': ''})

    show_step(9)
    cld.check.check_k8s_nodes()
    cld.check.check_k8s_pods()
    cld.check.check_actual_expected_pods()

    show_step(10)
    # TODO(va4st): Need a better way to determine the needs for CoreDNS sync (e.g some marker in mgmt config maybe?)
    if (not kcm_manager.kof.kof_is_deployed() and kcm_manager.kof.kof_ms_deployed()
            and kcm_manager.source == 'custom-enterprise'):
        corefile = k8s_utils.get_corefile(kcm_manager.api)
        k8s_utils.override_coredns_configmap(cld.k8sclient, corefile)
        coredns_deployment = cld.k8sclient.deployments.get(name='coredns', namespace='kube-system')
        coredns_deployment.rollout_restart()
