import pytest

from si_tests import logger
from si_tests import settings

LOG = logger.logger


def remove_cluster_collection(clds_list):
    LOG.warning(f"Async removal of clusterdeployments: {clds_list}")

    if settings.KSI_NESTED_CLUSTER_CHECK_ENABLED:

        LOG.info("Checking for nested CLDs.")
        kcm_clds = [cl for cl in clds_list if cl.k8sclient.k0rdent_cluster_deployments.available]

        # check each one for nested clusterdeployments and gather them
        nested_clds = []
        for kcm_cld in kcm_clds:
            nest_clds = kcm_cld.k8sclient.k0rdent_cluster_deployments.list_all()
            if nest_clds:
                LOG.warning(f"{kcm_cld.namespace}/{kcm_cld.name} has {len(nest_clds)} nested CLDs: {nest_clds}")
            nested_clds.extend(nest_clds)

        assert len(nested_clds) == 0, (
            f"Found {len(nested_clds)} nested CLD(s)"
            "Remove nested CLDs before continuing."
        )
        LOG.info("No nested CLDs found.")

    for cl in clds_list:
        LOG.info(f"Initialising resource chain for cluster {cl.namespace}/{cl.name}")
        # (va4st): initialize resource chain before removal start for each cluster to check leftovers
        _ = cl.clusterobject.infracluster

    for cl in clds_list:
        try:
            LOG.info(f"Checking for LoadBalancers in cluster {cl.namespace}/{cl.name}")
            services = cl.k8sclient.services.list_all()
            lb_services = [svc for svc in services if svc.type == 'LoadBalancer']
            if lb_services:
                LOG.info(
                    f"Found {len(lb_services)} LoadBalancers in cluster {cl.namespace}/{cl.name}"
                    )
                for svc in lb_services:
                    LOG.info(
                        f"Removing LoadBalancers/LoadBalancer/ {svc.namespace}/{svc.name} "
                        f"from cluster {cl.namespace}/{cl.name}"
                    )
                    svc.delete()
            else:
                LOG.info(f"No LoadBalancers found in cluster {cl.namespace}/{cl.name}")
        except Exception as e:
            LOG.warning(f"Failed to cleanup LoadBalancers from cluster {cl.namespace}/{cl.name}: {e}")

    if settings.KSI_CLEANUP_CHILD_PVCS:
        for cl in clds_list:
            cl.check.delete_cleanup_and_check_pvcs()
            LOG.info(f"PVCs were removed from clusterdeployment '{cl.name}'")

    for cl in clds_list:
        cl.delete()
        LOG.info(f"Request for removal {cl.namespace}/{cl.name} was sent.")

    LOG.info('Additionally checking for leftovers after clusterdeployments removal')
    for cl in clds_list:
        cl.check.check_clusterdeployment_deleted()
    # (va4st): Async removal of clds make us able not to wait for removal of each cluster separately. But it's
    # still need to check anyway that everything was cleared up including capi objects.


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
def test_cleanup_kcm(kcm_bootstrap_manager, show_step):
    """Cleanup KCM chart out of k8s cluster

    Scenario:
        1. Verify kcm installed inside provided cluster
        2. Initialize kcm manager
        3. Check regional entities and remove if present (optional)
        4. Check clusterdeployments presence and remove if present (optional)
        5. Check credential objects and remove if present (optional)
        6. Check and remove any static identities (optional)
        7. Check and remove auth configmaps if exists (optional) and secrets (if exists)
        8. Check and remove kof (if exists)
        9. Remove management object
        10. Uninstall helm kcm release
        11. Remove kcm-system and projectsveltos namespaces
        12. Remove all clusterroles and clusterrolebindings related to sveltos and CAPI
        13. Remove CRDs
    """

    show_step(1)
    assert kcm_bootstrap_manager.is_kcm_installed(), 'KCM chart is not installed in cluster'

    show_step(2)
    kcm_manager = kcm_bootstrap_manager.check.kcm_mgr

    if settings.REMOVE_RESOURCES_ON_KCM_CLEANUP:
        if kcm_manager.condition.oss_1_4_0_and_above():
            show_step(3)
            LOG.info("Starting regional entities gathering")
            regional_allocated_clusters = []
            regional_credentials = []
            regions = kcm_manager.api.k0rdent_regions.list_all()
            for region in regions:
                regional_credentials.extend(kcm_manager.list_regional_credentials(region.name))
            for credential in regional_credentials:
                regional_allocated_clusters.extend(kcm_manager.list_clusterdeployments_for_credential(credential))
            # NOTE(va4st): First round of cld removal. Removing clusters allocated to regions
            if regional_allocated_clusters:
                LOG.info(f"Found clusterdeployments allocated to regions: {regional_allocated_clusters}")
                remove_cluster_collection(regional_allocated_clusters)
            # NOTE(va4st): Removing regional creds after clusters removal
            if regional_credentials:
                LOG.info(f"Removing regional credentials: {regional_credentials}")
                for reg_cred in regional_credentials:
                    LOG.info(f"Removing regional credential {reg_cred.namespace}/{reg_cred.name}")
                    reg_cred.delete()
            if regions:
                LOG.info(f"Regional objects found: {regions}")
                # NOTE(va4st): Removing regions asynchronously after regional creds removal
                for region in regions:
                    LOG.info(f"Removing region {region.name}")
                    region.delete()
                # NOTE(va4st): Checking that regions removed
                for region in regions:
                    kcm_bootstrap_manager.check.check_region_removed(region)

        show_step(4)
        clds = kcm_manager.list_all_clusterdeployments()
        # TODO(va4st): Add regions removal before clusterdeployments.
        #  Should remove by following scheme: clusters with regional creds -> regional creds
        #  -> regional objects -> all other clusters -> all creds
        if clds:
            LOG.info(f"Found clusterdeployments {clds}.")
            remove_cluster_collection(clds)

        show_step(5)
        creds = kcm_manager.list_all_credentials()
        if creds:
            LOG.info(f"Credentials found: {creds}")
            for cred in creds:
                LOG.info(f"Removing Credential {cred.namespace}/{cred.name}")
                cred.delete()

        show_step(6)
        awssis = kcm_manager.api.awsclusterstaticidentities.list_all()
        if awssis:
            LOG.info(f"AWS Static Identities found: {awssis}")
            for awssi in awssis:
                awssi.delete()
                LOG.info(f"AWSSI {awssi.name} removed")

        azcids = kcm_manager.api.azureclusteridentities.list_all()
        if azcids:
            LOG.info(f"Azure Cluster IDs found: {azcids}")
            for azcid in azcids:
                azcid.delete()
                LOG.info(f"Azure cID {azcid.namespace}/{azcid.name} removed")

        vspcids = kcm_manager.api.vsphereclusteridentities.list_all()
        if vspcids:
            LOG.info(f"vSphere Cluster IDs found: {vspcids}")
            for vspcid in vspcids:
                vspcid.delete()
                LOG.info(f"vSphere cID {vspcid.name} removed")

        show_step(7)
        # TODO(va4st): Temporal cleanup by name. When separated auth manager will be introduced - need to change
        # TODO: that behavior to remove auth configs not by name filtering but by labels
        auth_cms = kcm_manager.api.configmaps.list_all(name_prefix='ksi')
        if auth_cms:
            LOG.info(f"ConfigMaps contains ksi auth creds found: {auth_cms}")
            for cm in auth_cms:
                cm.delete()
                LOG.info(f"ConfigMap {cm.namespace}/{cm.name} removed")

        auth_secrets = kcm_manager.api.secrets.list_all(name_prefix='ksi')
        if auth_secrets:
            LOG.info(f"Secrets contains ksi auth creds found: {auth_secrets}")
            for secr in auth_secrets:
                secr.delete()
                LOG.info(f"Secret {secr.namespace}/{secr.name} removed")

    show_step(8)
    if kcm_manager.kof.is_kof_mothership_present():
        kcm_bootstrap_manager.check.cleanup_grafana_entities()

    if settings.KSI_KOF_ISTIO_NAMESPACE not in [ns.name for ns in kcm_manager.get_namespaces()]:
        pass
    else:
        LOG.info(f"{settings.KSI_KOF_ISTIO_NAMESPACE} found. Double check that kof also will be removed")
        kof_istio_ns = kcm_manager.get_namespace(settings.KSI_KOF_ISTIO_NAMESPACE)
        if kcm_manager.kof.is_kof_istio_present():
            kcm_bootstrap_manager.check.uninstall_kof_istio_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        kof_istio_ns.delete(a_sync=False)

    if settings.KSI_KOF_NAMESPACE not in [ns.name for ns in kcm_manager.get_namespaces()]:
        pass
    else:
        # full copy of test_cleanup_kof_mothership
        LOG.info(f"{settings.KSI_KOF_NAMESPACE} found. Double check that kof also will be removed")
        kof_ns = kcm_manager.get_namespace(settings.KSI_KOF_NAMESPACE)
        if kcm_manager.kof.is_kof_child_present():
            kcm_bootstrap_manager.check.uninstall_kof_child_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_regional_present():
            kcm_bootstrap_manager.check.uninstall_kof_regional_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_collectors_present():
            kcm_bootstrap_manager.check.uninstall_kof_collectors_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_mothership_present():
            kcm_bootstrap_manager.check.uninstall_kof_mothership_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        if kcm_manager.kof.is_kof_operators_present():
            kcm_bootstrap_manager.check.uninstall_kof_op_and_wait()
            kcm_manager.refresh_expected_objects()
            kcm_bootstrap_manager.check.check_actual_expected_pods()

        kof_ns.delete(a_sync=False)

    show_step(9)
    kcm_bootstrap_manager.check.delete_mgmt_object_and_wait()

    show_step(10)
    kcm_bootstrap_manager.check.uninstall_kcm_and_wait()

    show_step(11)
    kcm_bootstrap_manager.check.kcm_namespace.delete()
    nss = kcm_manager.get_namespaces()
    if settings.SVELTOS_NAMESPACE in [ns.name for ns in nss]:
        LOG.info(f"{settings.SVELTOS_NAMESPACE} present")
        sveltos_ns = kcm_manager.get_namespace(settings.SVELTOS_NAMESPACE)
        sveltos_ns.delete()
        sveltos_ns.wait_for_deletion()
    if settings.KSI_SVELTOS_MGMT_NAMESPACE in [ns.name for ns in nss]:
        LOG.info(f"{settings.KSI_SVELTOS_MGMT_NAMESPACE} present")
        sveltos_ns = kcm_manager.get_namespace(settings.KSI_SVELTOS_MGMT_NAMESPACE)
        sveltos_ns.delete()
        sveltos_ns.wait_for_deletion()

    show_step(12)
    capi_label = 'clusterctl.cluster.x-k8s.io'
    capi_clusterroles = kcm_manager.api.clusterrole.list_all(label_selector=capi_label)
    if capi_clusterroles:
        for role in capi_clusterroles:
            LOG.info(f"Removing clusterrole {role.name}")
            role.delete()
    capi_clusterrolebindings = kcm_manager.api.clusterrolebindings.list_all(label_selector=capi_label)
    if capi_clusterrolebindings:
        for rolebind in capi_clusterrolebindings:
            LOG.info(f"Removing clusterrolebinding {rolebind.name}")
            rolebind.delete()
    # NOTE(va4st): sveltos resources have no labels, so it can be identified only by name
    sv_name_pattern = 'sveltos'
    all_clusterroles = kcm_manager.api.clusterrole.list_all()
    sv_clusterroles = [role for role in all_clusterroles if sv_name_pattern in role.name]
    if sv_clusterroles:
        for role in sv_clusterroles:
            LOG.info(f"Removing clusterrole {role.name}")
            role.delete()
    all_clusterrolebindings = kcm_manager.api.clusterrolebindings.list_all()
    sv_clusterrolesbindings = [binding for binding in all_clusterrolebindings if sv_name_pattern in binding.name]
    if sv_clusterrolesbindings:
        for rolebind in sv_clusterrolesbindings:
            LOG.info(f"Removing clusterrolebinding {rolebind.name}")
            rolebind.delete()

    show_step(13)
    kcm_bootstrap_manager.check.cleanup_crds()
