#!/usr/bin/env bash

set -euo pipefail

# GitHub repository information
GITHUB_REPO="k0rdent/kcm"
GITHUB_REPO_URL="https://github.com/${GITHUB_REPO}.git"

# Default values
OUTPUT="${OUTPUT:-}"
BRANCH="${BRANCH:-main}"

# Help message
usage() {
    cat <<EOF
Usage: $(basename "$0") [OPTIONS]

Generate a release.yaml file with version updates using git describe.
This script mimics the Makefile's generate-release target behavior.

OPTIONS:
    -o, --output FILE       Output file path (default: release-<version>.yaml)
    -b, --branch BRANCH     Branch to use (default: main)
    -h, --help             Show this help message

EXAMPLES:
    # Generate release.yaml with auto-detected version
    $(basename "$0")

    # Generate to a specific output file
    $(basename "$0") -o /tmp/my-release.yaml

    # Using environment variables
    OUTPUT=/tmp/release.yaml BRANCH=main $(basename "$0")

NOTES:
    - Uses shallow git clone to fetch repository from ${GITHUB_REPO}
    - Determines version using 'git describe --tags --always'
    - Applies same transformations as 'make generate-release':
      * .spec.version = VERSION
      * .metadata.name = kcm-FQDN_VERSION
      * .spec.kcm.template = kcm-FQDN_VERSION

EOF
}

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        -o|--output)
            OUTPUT="$2"
            shift 2
            ;;
        -b|--branch)
            BRANCH="$2"
            shift 2
            ;;
        -h|--help)
            usage
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            usage
            exit 1
            ;;
    esac
done

# Check if required commands are available
if ! command -v curl &> /dev/null; then
    echo "Error: curl is required but not installed"
    exit 1
fi

if ! command -v git &> /dev/null; then
    echo "Error: git is required but not installed"
    exit 1
fi

# Determine version using git describe
echo "Determining version using git describe..."
echo "Repository: ${GITHUB_REPO}"
echo "Branch: ${BRANCH}"

# Create temporary directory for git clone
TEMP_DIR=$(mktemp -d)
trap 'rm -rf "${TEMP_DIR}"' EXIT

echo "Cloning repository (blobless clone)..."
# Use --filter=blob:none for minimal clone (history and tags, but no file contents)
# This is much faster than a full clone but still allows git describe to work
if ! git clone --filter=blob:none --branch="${BRANCH}" --single-branch \
    "${GITHUB_REPO_URL}" "${TEMP_DIR}" &> /dev/null; then
    echo "Error: Failed to clone repository"
    exit 1
fi

# Run git describe in the cloned repository
GIT_VERSION=$(git -C "${TEMP_DIR}" describe --tags --always)

if [ -z "${GIT_VERSION}" ]; then
    echo "Error: Failed to determine version using git describe"
    exit 1
fi

echo "Git describe output: ${GIT_VERSION}"

VERSION="${GIT_VERSION#v}"  # Remove leading 'v' if present

echo "Computed version: ${VERSION}"

# Set output filename if not specified
if [ -z "${OUTPUT}" ]; then
    SAFE_VERSION=$(echo "${VERSION}" | sed 's/[^a-zA-Z0-9-]/-/g')
    OUTPUT="kcm-${SAFE_VERSION}.yaml"
fi

FQDN_VERSION=$(echo "${VERSION}" | sed 's/\./-/g')
echo "FQDN version: kcm-${FQDN_VERSION}"
echo "Output file: ${OUTPUT}"
echo ""

# Read release.yaml from cloned repository
RELEASE_YAML_PATH="templates/provider/kcm-templates/files/release.yaml"
RELEASE_FILE="${TEMP_DIR}/${RELEASE_YAML_PATH}"

echo "Reading release.yaml from repository..."
echo "Path: ${RELEASE_YAML_PATH}"

if [ ! -f "${RELEASE_FILE}" ]; then
    echo "Error: Could not find release.yaml at ${RELEASE_YAML_PATH}"
    exit 1
fi

RELEASE_CONTENT=$(cat "${RELEASE_FILE}")
if [ -z "${RELEASE_CONTENT}" ]; then
    echo "Error: release.yaml is empty"
    exit 1
fi

# Generate the release.yaml using sed (same transformations as Makefile)
echo "Generating release.yaml..."

# Use sed to update the three fields:
# 1. .spec.version = VERSION
# 2. .metadata.name = kcm-FQDN_VERSION
# 3. .spec.kcm.template = kcm-FQDN_VERSION (only the kcm section, not capi)

echo "${RELEASE_CONTENT}" | sed -e "/^  version:/s/version:.*/version: ${VERSION}/" \
    -e "/^  name:/s/name:.*/name: kcm-${FQDN_VERSION}/" \
    -e "/^  kcm:/,/^  [a-z]/ { /^    template:/s/template:.*/template: kcm-${FQDN_VERSION}/; }" \
    > "${OUTPUT}"

if [ -f "${OUTPUT}" ]; then
    ABSOLUTE_PATH=$(cd "$(dirname "${OUTPUT}")" && pwd)/$(basename "${OUTPUT}")
    echo ""
    echo "✓ Release file generated successfully"
    echo "  Location: ${OUTPUT}"
    echo "  Absolute path: ${ABSOLUTE_PATH}"
    echo "  Release name: $(grep 'name:' "${OUTPUT}" | head -1 | awk '{print $2}')"
    echo "  Release version: $(grep 'version:' "${OUTPUT}" | head -1 | awk '{print $2}')"
    echo "  KCM template: $(grep -A 1 'kcm:' "${OUTPUT}" | grep 'template:' | awk '{print $2}')"
else
    echo "Error: Failed to generate release.yaml"
    exit 1
fi
