# GitRepository storage for the custom cluster templates

This directory is a target for the charts reference in clustertemplate object. it contains a directories 
with charts for different cluster templates.

Table of contents:
- [Structure](#structure)
- [Creating a GitRepository and ClusterTemplate objects](#creating-a-gitrepository-and-clustertemplate-objects)
  - [Obtaining access](#obtaining-access)
  - [Creating the secret](#creating-the-secret)
  - [Creating the GitRepository object](#creating-the-gitrepository-object)
  - [Creating the ClusterTemplate object](#creating-the-clustertemplate-object)
- [Available custom cluster templates](#available-custom-cluster-templates)
  - [openstack-standalone-custom-cp (key features)](#openstack-standalone-custom-cp-key-features)
  - [openstack-hosted-custom-cp (key features)](#openstack-hosted-custom-cp-key-features)
  - [aws-hybrid-arch-standalone (key features)](#aws-hybrid-arch-standalone-custom-cp-key-features)


## Structure
The structure of this directory is as follows:

```
.
├── README.md
├── cluster-template-1
│   ├── Chart.yaml
│   ├── values.yaml
│   └── templates
│       ├── deployment.yaml
│       └── service.yaml
└── cluster-template-2
    ├── Chart.yaml
    ├── values.yaml
    └── templates
        ├── deployment.yaml
        └── service.yaml
```

## Creating a GitRepository and ClusterTemplate objects

### Obtaining access
It's possible to use PAT and https URL or ssh key and ssh URL to access the repository.
PAT should be generated with scope "repo" for private repositories and read access.
SSH key should be added to ksi repo via settings by ksi-admins or IT. key MUST not have write permisisons.
It's also possible to use plain-text auth but we don't recommend to go this way due safety reasons.

### Creating the secret
The easiest way to create the secret is to use the following command:
```bash
kubectl -n kcm-system create secret generic git-ssh --from-file=ssh-privatekey=./ksi-git --from-file=known_hosts=<(ssh-keyscan -t rsa,ecdsa,ed25519 github.com 2>/dev/null)   --type=kubernetes.io/ssh-auth
```
this will create a secret like
```yaml
apiVersion: v1
data:
  known_hosts: <b64-encoded-known-hosts>
  ssh-privatekey: <b64-encoded-private-key>
kind: Secret
metadata:
  name: git-ssh
  namespace: kcm-system
type: kubernetes.io/ssh-auth
```

or for PAT:
```bash
kubectl -n kcm-system create secret generic git-cred --from-literal=username=x-access-token --from-literal=password='<GITHUB_PAT>' --type=kubernetes.io/basic-auth
```

this will create a secret like
```yaml
apiVersion: v1
data:
  password: <b64-encoded-pat>
  username: <b64-encoded-username>
kind: Secret
metadata:
  name: git-cred
  namespace: kcm-system
type: kubernetes.io/basic-auth
```

### Creating the GitRepository object
Once the secret is created, you can create the GitRepository object. Here is an example:
```yaml
apiVersion: source.toolkit.fluxcd.io/v1
kind: GitRepository
metadata:
  labels:
    k0rdent.mirantis.com/managed: "true"
  name: custom-templates
  namespace: kcm-system
spec:
  interval: 1m
  ref:
    branch: main
  secretRef:
    name: git-ssh
  timeout: 60s
  url: ssh://git@github.com/k0rdent/ksi.git
```

It's possible to set any repo (or your personal) for testing purposes.
```yaml
apiVersion: source.toolkit.fluxcd.io/v1
kind: GitRepository
metadata:
  labels:
    k0rdent.mirantis.com/managed: "true"
  name: custom-templates
  namespace: kcm-system
spec:
  interval: 1m
  ref:
    branch: main
  secretRef:
    name: git-ssh
  timeout: 60s
  url: ssh://git@github.com/va4st/ksi.git
```

Our source controller is configured to index repos with specific label only:
```yaml
metadata:
  labels:
    k0rdent.mirantis.com/managed: "true"
```
Make sure to add it to your repo object.

### Creating the ClusterTemplate object
Once the GitRepository object is created, you can create the ClusterTemplate object. Here is an example:
```yaml
apiVersion: k0rdent.mirantis.com/v1beta1
kind: ClusterTemplate
metadata:
  annotations:
    helm.sh/resource-policy: keep
  labels:
    k0rdent.mirantis.com/component: kcm
  name: openstack-standalone-custom-cp-1-0-0
  namespace: kcm-system
spec:
  helm:
    chartSpec:
      chart: ./si_tests/templates/clustertemplates/openstack-standalone-custom-cp
      interval: 10m0s
      reconcileStrategy: ChartVersion
      sourceRef:
        kind: GitRepository
        name: custom-templates
      version: 1.0.0
```

## Available custom cluster templates

The following cluster templates are available in this repository:

| Name                           | Description                                                                        | Version |
|--------------------------------|------------------------------------------------------------------------------------|---------|
| openstack-standalone-custom-cp | OpenStack cluster with standalone control plane and extended identity capabilities | 1.0.0   |
| openstack-hosted-custom-cp     | OpenStack cluster with hosted control plane and extended identity capabilities     | 1.0.0   |

### openstack-standalone-custom-cp (key features)
Almost the same as openstack-standalone but the identity can be set to OpenStackClusterIdentity:
```yaml
    identityRef:
      cloudName: openstack
      name: ksi-openstack-auth-nugy-identity
      region: RegionOne
      type: ClusterIdentity
```

More information about OpenStackClusterIdentity can be found in [OpenStackClusterIdentity documentation](https://github.com/Mirantis/cluster-api-provider-openstack/blob/v0.12.5-mirantis.0/docs/book/src/topics/openstack-cluster-identity.md).

Complete example of ClusterDeployment using this template:
```yaml
apiVersion: k0rdent.mirantis.com/v1beta1
kind: ClusterDeployment
metadata:
  name: ksi-v4-oss140-sep29-1
  namespace: kcm-system
spec:
  config:
    authURL: https://keystone.ic-eu.ssl.mirantis.net/
    clusterAnnotations: {}
    controlPlane:
      flavor: tiny.ctl
      image:
        filter:
          name: ubuntu-24.04-server-cloudimg-amd64-20250805
    controlPlaneNumber: 1
    externalNetwork:
      filter:
        name: public
    identityRef:
      cloudName: openstack
      name: ksi-openstack-auth-nugy-identity
      region: RegionOne
      type: ClusterIdentity
    worker:
      flavor: kaas-bm.worker
      image:
        filter:
          name: ubuntu-24.04-server-cloudimg-amd64-20250805
    workersNumber: 1
  credential: ksi-openstack-auth-nugy-cred
  template: openstack-standalone-custom-cp-1-0-0
```

### openstack-hosted-custom-cp (key features)
Almost the same as openstack-hosted but the identity can be set to OpenStackClusterIdentity:
```yaml
    identityRef:
      cloudName: openstack
      name: ksi-openstack-auth-nugy-identity
      region: RegionOne
      type: ClusterIdentity
```

The template to generate a hosted control plane cluster with custom identity reference.
```yaml
apiVersion: k0rdent.mirantis.com/v1beta1
kind: ClusterDeployment
metadata:
  name: openstack-hosted
spec:
  template: openstack-hosted-custom-cp-1-0-0
  credential: openstack-credential
  config:
    workersNumber: 2
    flavor: m1.medium
    image:
      filter:
        name: ubuntu-20.04
    externalNetwork:
      filter:
        name: "public"
    identityRef:
      name: "openstack-cloud-config"
      cloudName: "openstack"
      region: "{{.spec.identityRef.region}}"
      type: ClusterIdentity

    network:
      filter:
        name: "{{.status.network.name}}"
    router:
      filter:
        name: "{{.status.router.name}}"
    subnets:
    - filter:
        name: "{{(index .status.network.subnets 0).name}}"
    ports:
    - network:
        filter:
          name: "{{.status.network.name}}"
```

This template can be filled with the following command:
```bash
kubectl -n kcm-system get openstackcluster <host_cluster> -o go-template="$(cat os-hosted-tmpl.yaml)" > os-hosted.yaml
```

For Regional API usage - host_cluster should be the regional cluster name.


### aws-hybrid-arch-standalone (key features)
This template deploys a Kubernetes cluster on AWS with a hybrid architecture, featuring a standalone control plane.
Key features include:
- Split arch types of workers (can be arm64 and amd64 workers in the same cluster)
- Workers can be set to 0 (COntrol plain will act as a worker as well)

Complete example of ClusterDeployment using this template with arm64 and amd64 workers:
```yaml
apiVersion: k0rdent.mirantis.com/v1beta1
kind: ClusterDeployment
metadata:
  name: v4-120rc3-arm-hybrid
  namespace: kcm-system
spec:
  config:
    clusterAnnotations: {}
    controlPlane:
      instanceType: c5a.large
      rootVolumeSize: 15
    controlPlaneNumber: 3
    region: eu-north-1
    worker:
      instanceType: c5.large
      rootVolumeSize: 50
    arm64Worker:
      instanceType: c5g.xlarge
    workersNumber: 1
    armWorkersNumber: 1
  credential: ksi-aws-auth-static-mnbg-cred
  template: aws-hybrid-arch-standalone-1-0-0
```
`worker.rootVolumeSize` is set for both amd64 and arm64 workers.


and withput workers only on arm64 control plane:
```yaml
apiVersion: k0rdent.mirantis.com/v1beta1
kind: ClusterDeployment
metadata:
  name: v4-120rc3-arm-mgmt-3
  namespace: kcm-system
spec:
  config:
    clusterAnnotations: {}
    controlPlane:
      instanceType: c6g.xlarge
      rootVolumeSize: 15
    controlPlaneNumber: 3
    k0s:
      arch: arm64
    region: eu-north-1
    workersNumber: 0
    armWorkersNumber: 0
  credential: ksi-aws-auth-static-eqep-cred
  template: aws-hybrid-arch-standalone-1-0-0
```

Template can be used with usual ksi [aws standalone test](../../tests/clusterdeployments/README.md#test_create_aws_clusterdeploymentpy) 
with `KCM_CLUSTER_TEMPLATE_NAME=aws-hybrid-arch-standalone-1-0-0`

Additional variables available in ksi for that template

| ENV Variable                    | Type    | Description         | Defaults                      |
|---------------------------------|---------|---------------------|-------------------------------|
| `KSI_AWS_ARM_WORKERS_NUMBER`    | `<int>` | arm64 workers count | 0                             |
| `KSI_AWS_INSTANCE_TYPE_ARM_WRK` | `<str>` | arm64 instance type | c6g.medium (template default) |