#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.


import json
import os
from importlib import resources

_boolean_states = {
    "1": True,
    "yes": True,
    "true": True,
    "on": True,
    "0": False,
    "no": False,
    "false": False,
    "off": False,
}


# In case if environment variable is set but contains an empty string,
# use the default instead of empty value
def get_var(name, default=""):
    value = os.environ.get(name) or None
    return value or default


def get_var_force_dir(name, default=''):
    """
    same as get_var but force create dir, if not exist
    """
    _result = get_var(name, default=default)
    if _result and not os.path.exists(_result):
        print(f'makedir {_result}')
        os.makedirs(_result)
    return _result


def get_var_as_bool(name, default):
    value = get_var(name, "")
    if isinstance(default, str):
        default = _boolean_states.get(default.lower(), default)
        if isinstance(default, str):
            raise Exception(
                f"default for '{name}' contains some non-binary trash: '{default}'")
    return _boolean_states.get(value.lower(), default)


def get_var_from_json(name, default="{}"):
    values = json.loads(get_var(name, default))
    return values


def get_path(name, default):
    value = get_var(name, '')
    if not value and not default:
        return ''
    return os.path.expanduser(
        value or str(resources.files(__package__) / default)
    )


# determine seed node os for future cmd preparations
def parse_seed_os_type(os_type):
    supported = ["linux", "darwin"]
    desired_os = os_type.strip().lower()
    assert desired_os in supported, "Incompatible OS type: {0}, supported {1}".format(
        desired_os, supported
    )

    return desired_os
