import requests
import time
from urllib.parse import urljoin
from typing import Optional, Dict, Any, List

from si_tests import logger
from si_tests.utils import waiters

LOG = logger.logger


class HarborAPIError(RuntimeError):
    pass


class HarborManager:
    def __init__(self, harbor_base: str, username: str, password: str, verify_tls: bool = True):
        """
        harbor_base: e.g. https://harbor.example.com
        """
        if not harbor_base.endswith("/"):
            harbor_base += "/"
        self.base = harbor_base
        # Create a configured requests.Session with basic auth, JSON headers, and TLS verify control.
        self.s = requests.Session()
        self.s.auth = (username, password)
        self.s.headers.update({"Content-Type": "application/json", "Accept": "application/json"})
        self.s.verify = verify_tls
        self._csrf_token: Optional[str] = None

    # ------------- helpers -------------
    @staticmethod
    def _ensure_ok(resp: requests.Response, expected: Optional[int] = None) -> None:
        """
        Raise a helpful error if response is not OK. Optionally enforce an expected status code.
        """
        try:
            resp.raise_for_status()
        except Exception as e:
            detail = None
            try:
                detail = resp.json()
            except Exception:
                detail = resp.text
            raise HarborAPIError(
                f"HTTP {resp.status_code} {resp.reason} at {resp.request.method} {resp.url}\n{detail}"
            ) from e
        if expected is not None and resp.status_code != expected:
            raise HarborAPIError(
                f"Unexpected status {resp.status_code} (wanted {expected}) at {resp.request.method} "
                f"{resp.url}: {resp.text}"
            )

    def _api(self, path: str) -> str:
        return urljoin(self.base, path)

    def _fetch_csrf_once(self, allow_redirects: bool = False):
        probe_headers = {"X-Harbor-CSRF-Token": "true"}
        candidates = [
            ("api/v2.0/projects", {"page": 1, "page_size": 1}),
            ("api/v2.0/systeminfo", None),
            ("api/v2.0/replication/adapters", None),
            ("api/v2.0/registries", {"page": 1, "page_size": 1}),
        ]
        for path, params in candidates:
            r = self.s.get(self._api(path), headers=probe_headers, params=params, allow_redirects=allow_redirects)
            token = (r.headers.get("X-Harbor-CSRF-Token")
                     or self.s.cookies.get("harbor_csrf")
                     or r.cookies.get("harbor_csrf"))
            if not token:
                sc = r.headers.get("Set-Cookie") or r.headers.get("set-cookie")
                if sc and "harbor_csrf=" in sc:
                    # naive parse
                    token = sc.split("harbor_csrf=", 1)[1].split(";", 1)[0]
                    self.s.cookies.set("harbor_csrf", token, path="/")
            if token:
                return token
        return None

    def _ensure_csrf(self) -> None:
        """
        Fetch and cache Harbor CSRF token & cookie for subsequent POST/PUT/DELETE.
        """
        # Always get a fresh token; clear old cookie/token
        self._csrf_token = None
        # optional: clear only the csrf cookie to avoid bloat
        try:
            self.s.cookies.clear(domain=None, path="/", name="harbor_csrf")
        except Exception:
            pass

        token = self._fetch_csrf_once(allow_redirects=False)
        if not token:
            raise HarborAPIError("Failed to obtain CSRF token: no header/cookie returned")
        self._csrf_token = token

    def _request(self, method: str, path: str, *, json=None, params=None, expected=None):
        """
        Wrapper that auto-injects CSRF for write methods and retries once on CSRF errors.
        """
        url = self._api(path)
        method_u = method.upper()
        headers = {}

        if method_u in {"POST", "PUT", "PATCH", "DELETE"}:
            self._ensure_csrf()
            headers["X-Harbor-CSRF-Token"] = self._csrf_token

        resp = self.s.request(method_u, url, json=json, params=params, headers=headers or None)
        if resp.status_code in (401, 403) and "CSRF" in resp.text and method_u in {"POST", "PUT", "PATCH", "DELETE"}:
            # Try once more with a brand-new token
            self._ensure_csrf()
            headers["X-Harbor-CSRF-Token"] = self._csrf_token
            resp = self.s.request(method_u, url, json=json, params=params, headers=headers)

        self._ensure_ok(resp, expected=expected)
        return resp

    # ------------- API methods -------------
    # 1) Create registry endpoint
    def create_registry_endpoint(
        self,
        name: str,
        url: str,
        adapter_type: str,                 # "docker-hub", "docker-registry", "harbor", "ecr", "acr", "gcr", "quay" ...
        access_key: Optional[str] = None,  # credentials for the SOURCE registry (if needed)
        access_secret: Optional[str] = None,
        insecure: bool = False,
        description: str = ""
    ) -> int:
        """
        Creates a Harbor Registry (source) endpoint and returns its numeric ID.
        For Docker Hub, use adapter_type="docker-hub" and url="https://registry-1.docker.io".
        """
        body: Dict[str, Any] = {
            "name": name,
            "url": url,
            "type": adapter_type,
            "insecure": insecure,
            "description": description,
        }
        # credential block is optional; include only if access_key is provided
        if access_key is not None:
            body["credential"] = {
                "type": "basic",
                "access_key": access_key,
                "access_secret": access_secret or ""
            }

        resp = self._request("POST", "api/v2.0/registries", json=body, expected=201)

        # Harbor usually returns 201 with Location header pointing to /registries/<id>
        loc = resp.headers.get("Location") or resp.headers.get("location")
        if loc and loc.rstrip("/").split("/")[-1].isdigit():
            return int(loc.rstrip("/").split("/")[-1])

        # Fallback: list registries and pick by name
        return self.get_registry_endpoint_id(name)

    # 2) Get registry endpoint ID (by name)
    def get_registry_endpoint_id(self, name: str) -> int:
        """
        Returns the numeric ID of a registry endpoint with the given name. Raises if not found.
        """
        resp = self._request("GET", "api/v2.0/registries", params={"page": 1, "page_size": 100})

        for r in resp.json():
            if r.get("name") == name:
                return int(r["id"])
        raise HarborAPIError(f"Registry endpoint named '{name}' not found")

    # 3) Create replication policy (manual pull of one image:tag)
    def create_registry_replication_policy(
        self,
        policy_name: str,
        src_registry_id: int,
        dest_namespace: str,  # target Harbor project (namespace)
        flattening: int,      # how many levels removed from the dest_namespace. 0 - no removes
        image_name: str,      # e.g. "library/nginx"
        image_tag: str,       # e.g. "1.27"
        override: bool = True,
        deletion: bool = False,
        enabled: bool = True
    ) -> int:
        """
        Creates a replication policy that pulls a specific image:tag into the local Harbor project.
        Returns the policy ID.
        """
        body = {
            "name": policy_name,
            "description": f"One-time pull of {image_name}:{image_tag} into {dest_namespace}",
            "src_registry": {"id": src_registry_id},
            "dest_namespace": dest_namespace,
            "dest_namespace_replace_count": flattening,
            "filters": [
                {"type": "name", "value": image_name},
                {"type": "tag",  "value": image_tag, "decoration": "matches"}
            ],
            "trigger": {"type": "manual"},
            "deletion": deletion,
            "override": override,
            "enabled": enabled
        }
        resp = self._request("POST", "api/v2.0/replication/policies", json=body, expected=201)

        # Harbor returns 201; sometimes with Location
        loc = resp.headers.get("Location") or resp.headers.get("location")
        if loc and loc.rstrip("/").split("/")[-1].isdigit():
            return int(loc.rstrip("/").split("/")[-1])

        # fallback: resolve by name
        return self.get_registry_replication_policy_id(policy_name)

    # 4) Get replication policy ID (by name)
    def get_registry_replication_policy_id(self, policy_name: str) -> int:
        """
        Returns the numeric ID of a replication policy with the given name. Raises if not found.
        """
        resp = self._request("GET", "api/v2.0/replication/policies", params={"page": 1, "page_size": 100})

        for p in resp.json():
            if p.get("name") == policy_name:
                return int(p["id"])
        raise HarborAPIError(f"Replication policy named '{policy_name}' not found")

    # 5) Execute replication policy once
    def __execute_registry_replication_policy(self, policy_id: int) -> int:
        """
        Starts a single execution for the given replication policy.
        Returns the execution ID.
        """
        resp = self._request("POST", "api/v2.0/replication/executions", json={"policy_id": policy_id}, expected=201)

        data = resp.json()
        exec_id = data.get("id")
        if exec_id is None:
            raise HarborAPIError(f"Execution created, but no ID in response: {data}")
        return int(exec_id)

    def execute_registry_replication_policy(self, policy_id: int) -> int:
        """
        Starts a single execution for the given replication policy and returns the execution ID.
        Handles Harbor variants that return 201 with an empty body and/or only a Location header.
        """
        # 1) Start the execution
        resp = self._request(
            "POST", "api/v2.0/replication/executions",
            json={"policy_id": policy_id}, expected=201
        )

        # 2) Try to parse from Location header: /replication/executions/<id>
        loc = resp.headers.get("Location") or resp.headers.get("location")
        if loc:
            tail = loc.rstrip("/").split("/")[-1]
            if tail.isdigit():
                return int(tail)

        # 3) Some Harbor builds return empty body (no JSON) - fall back to listing
        # Give Harbor a brief moment to persist the execution record
        time.sleep(0.3)

        # Prefer sorting by ID desc (fast, stable). If your Harbor ignores sort,
        # we still ask for page_size=1 which often returns newest first.
        params = {
            "policy_id": policy_id,
            "page": 1,
            "page_size": 1,
            "sort": "-id",           # also supports "-start_time" on newer Harbor
            # optionally: "trigger": "manual",  "status": "InProgress"
        }
        # This endpoint returns a list of executions
        list_resp = self._request("GET", "api/v2.0/replication/executions", params=params)
        items = list_resp.json() or []
        if items and "id" in items[0]:
            return int(items[0]["id"])

        # 4) Last resort: broaden the query in case sort/page is ignored by your build
        params.pop("sort", None)
        params["page_size"] = 10
        list_resp = self._request("GET", "api/v2.0/replication/executions", params=params)
        items = list_resp.json() or []
        if items:
            # pick the one with the max id
            exec_id = max(int(x["id"]) for x in items if "id" in x)
            return exec_id

        raise HarborAPIError("Execution created but ID not discoverable (empty body, no Location, list empty)")

    # 6) Get replication execution status
    def get_registry_replication_status(self, execution_id: int) -> Dict[str, Any]:
        """
        Returns the execution object (status, start/end times, etc) for the given execution ID.
        Useful fields: state, start_time, end_time, total, succeeded, failed, in_progress, stopped, etc.
        """
        resp = self._request("GET", f"api/v2.0/replication/executions/{execution_id}")
        if resp.text:
            return resp.json()
        else:
            return {}

    # 7) Wait replication execution status
    def wait_registry_replication_status(self, execution_id, expected_status="Succeed", timeout=3600, interval=10):

        def _check_replication_status():
            status = self.get_registry_replication_status(execution_id)
            total = status.get('total')
            if total is None:
                return False

            failed_msg = "[failed]" if status.get('failed') else ""
            in_progress_msg = "[in_progress]" if status.get('in_progress') else ""
            stopped_msg = "[stopped]" if status.get('stopped') else ""
            succeed_msg = "[succeed]" if status.get('succeed') else ""
            LOG.info(f"Replication #{status.get('id')}: {status.get('status')} "
                     f"{failed_msg}{in_progress_msg}{stopped_msg}{succeed_msg} | "
                     f"started at: {status.get('start_time')} | {status.get('succeed_msg')}")
            if status.get('status') == expected_status:
                return True
            if status.get('failed') or status.get('stopped'):
                raise Exception(f"Replication status '{status.get('status')}' , while expected '{expected_status}'")
            return False

        waiters.wait(
            _check_replication_status,
            timeout=timeout,
            interval=interval,
            timeout_msg=f"Timeout for waiting replication status #{execution_id} readiness after {timeout} sec.")
        LOG.info(f"Replication #{execution_id} is '{expected_status}'")

    # 8) Get replication task log (first task by default)
    def get_registry_replication_log(self, execution_id: int, task_index: int = 0) -> str:
        """
        Fetches the log for a specific task (by index) under the given execution.
        If there are multiple tasks, adjust task_index accordingly.
        """
        tasks_resp = self._request("GET", f"api/v2.0/replication/executions/{execution_id}/tasks")

        tasks: List[Dict[str, Any]] = tasks_resp.json()
        if not tasks:
            raise HarborAPIError(f"No tasks found for execution {execution_id}")
        if task_index < 0 or task_index >= len(tasks):
            raise HarborAPIError(f"task_index {task_index} out of range (0..{len(tasks)-1})")
        task_id = tasks[task_index]["id"]

        log_resp = self._request("GET", f"api/v2.0/replication/executions/{execution_id}/tasks/{task_id}/log")
        return log_resp.text.replace("\\n", "\n")

    # 9) Replicate the specified remote image to the Harbor registry
    def replicate_image(self, src_registry_url, src_registry_user, src_registry_password, image_name, image_tag,
                        harbor_project_name, adapter_type="docker-hub") -> str:
        # 1) Create/get registry endpoint
        try:
            reg_id = self.create_registry_endpoint(
                name=src_registry_url,  # Use registry endpoint url also as it's name
                url=src_registry_url,
                adapter_type=adapter_type,
                access_key=src_registry_user,
                access_secret=src_registry_password,
                insecure=False,  # Disable TLS verify for remote registry
                description="Docker Hub source for one-shot pull",
            )
        except HarborAPIError:
            LOG.error("Failed to create new registry endpoint, trying to get existing registry endpoint id")
            reg_id = self.get_registry_endpoint_id(src_registry_url)
        LOG.info(f"Registry ID: {reg_id}")

        # 2) Create/get replication policy
        policy_name = f"oneshot-{harbor_project_name}-{image_name.replace('/', '_')}-{image_tag}"
        try:
            pol_id = self.create_registry_replication_policy(
                policy_name=policy_name,
                src_registry_id=reg_id,
                dest_namespace=harbor_project_name,
                flattening=0,
                image_name=image_name,
                image_tag=image_tag,
                override=True,
                deletion=False,
                enabled=True,
            )
        except HarborAPIError:
            LOG.error("Failed to create new registry replication policy, "
                      "trying to get existing registry replication policy")
            pol_id = self.get_registry_replication_policy_id(policy_name)
        LOG.info(f"Policy ID: {pol_id}")

        # 3) Execute once
        exec_id = self.execute_registry_replication_policy(pol_id)
        LOG.info(f"Execution ID: {exec_id}")

        # 4) Poll status
        self.wait_registry_replication_status(exec_id, expected_status="Succeed")

        # 5) Fetch first task log
        try:
            log_text = self.get_registry_replication_log(exec_id, task_index=0)
            LOG.info(f"---- TASK LOG ----\n{log_text}")
        except HarborAPIError as e:
            LOG.info(f"No task log yet: {e}")

        return exec_id
