import re

from tabulate import tabulate

from retry import retry
from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, waiters, exceptions, k8s_utils

from typing import TYPE_CHECKING

if TYPE_CHECKING:
    from si_tests.managers.kcm_manager import ClusterDeployment

LOG = logger.logger


class ClusterDeploymentCheckManager(object):
    """Cluster deployment check manager"""

    EXCLUDED_PODS = []
    EXCLUDED_JOBS = []
    _cap_pods_logs = {}

    def __init__(self, clusterdeployment: "ClusterDeployment"):
        self._clusterdeployment: "ClusterDeployment" = clusterdeployment

    @property
    def clusterdeployment(self) -> "ClusterDeployment":
        return self._clusterdeployment

    @property
    def k8sclient(self):
        return self._clusterdeployment.k8sclient

    def collect_machines_data(self):
        machines = self.clusterdeployment.get_machines(raise_if_empty=False)
        machines_data = []
        for machine in machines:
            machine_data = machine.data
            # machine_provider = machine.capimachine_provider
            machine_provider_id = machine_data['spec'].get('providerID') or ""
            machine_status = machine_data.get("status") or {}
            machine_phase = machine_status.get("phase")

            machine_conditions = machine_status.get('conditions')
            # Here, "Ready" is not a status, but the condition name.
            machine_readiness_condition = [x for x in machine_conditions if x.get("type") == "Ready"]
            if machine_readiness_condition:
                machine_condition_message = machine_readiness_condition[0].get("message")
                machine_condition_ready = machine_readiness_condition[0].get("status")
            else:
                machine_condition_message = "NO MACHINE PROVIDER CONDITION"
                machine_condition_ready = None

            machine_provider_conditions = (machine_status.get(settings.CAPI_OPERATOR_APIVERSION, {}).
                                           get('conditions', []))
            # Here, "Ready" is not a status, but the condition name.
            machine_provider_readiness_condition = [x for x in machine_provider_conditions if x.get("type") == "Ready"]
            if machine_provider_readiness_condition:
                machine_provider_message = machine_provider_readiness_condition[0].get("message")
            else:
                machine_provider_message = "NO MACHINE PROVIDER CONDITION"

            # Collect data for status message
            machines_data.append({
                "name": machine.name,
                # "provider": machine_provider,
                "provider_id": machine_provider_id,
                "phase": machine_phase,
                "ready": machine_condition_ready,
                "message": machine_condition_message,
                "provider_message": machine_provider_message,
            })
        return machines_data

    def show_machines_conditions(self):
        # Get Machines data
        # headers = ["Machine", "Provider", "Phase", "Ready", "Message", "Provider message"]
        headers = ["Machine", "ProviderID", "Phase", "Ready", "Message", "Provider message"]
        machines_data = self.collect_machines_data()
        status_data = [[data["name"],
                        # data["provider"],
                        data["provider_id"],
                        data["phase"],
                        data["ready"],
                        data["message"],
                        data["provider_message"]]
                       for data in machines_data]
        # Show Machines status and not ready conditions
        status_msg = tabulate(status_data, tablefmt="presto", headers=headers)
        LOG.info(f"Machines status:\n{status_msg}\n")
        return machines_data

    # TODO(va4st): Extend function with other dynamic resources like machines, control plane
    #  machine sets, etc (rely on template, provider, etc)
    def _get_clusterdeployment_readiness(self, expected_condition_cld_fails=None, expected_condition_svc_fails=None,
                                         interval=None):
        """Get conditions from Cluster Deployment

        :rtype bool: True only if all conditions are true, False in other case
        """
        cluster_result = [
            self.clusterdeployment.are_conditions_ready(expected_fails=expected_condition_cld_fails,
                                                        verbose=True),
            self.clusterdeployment.are_conditions_ready(expected_fails=expected_condition_svc_fails,
                                                        verbose=True, mode='services'),
            self.clusterdeployment.are_conditions_ready(expected_fails=expected_condition_svc_fails,
                                                        verbose=True, mode='multiclusterservices')]
        # NOTE(va4st): If no machines in cluster (like in AKS by design or at all) - no needs to fill the log with
        # empty table
        if self.clusterdeployment.get_machines(raise_if_empty=False):
            self.show_machines_conditions()
        if settings.ENABLE_INTROSPECT_CAP_ERRORS:
            self.introspect_cap_errors(since_seconds=interval+15)
        return all(cluster_result)

    def check_cluster_readiness(self, timeout=settings.CHECK_CLUSTER_READINESS_TIMEOUT, interval=60,
                                expected_condition_cld_fails=None,
                                expected_condition_svc_fails=None):
        """
        Check that overall clusterdeployment is Ready
        and no unexpected deployments failed
        :param timeout: timeout for waiter
        :type timeout: int
        :param interval: interval to check status
        :type interval: int
        :param expected_condition_cld_fails: dict with conditions like
            { <condition type>: <part of condition message to match> , }
        :type expected_condition_cld_fails: Dict[str]
        :param expected_condition_svc_fails: dict with conditions like
            { <condition type>: <part of condition message to match> , }
        :type expected_condition_svc_fails: Dict[str]

        :rtype bool: bool
        """
        LOG.info("Checking readiness for clusterdeployment "
                 f"'{self.clusterdeployment.namespace}/{self.clusterdeployment.name}'")

        timeout_msg = "Timeout waiting for clusterdeployment readiness"
        self._cap_pods_logs = {}
        try:
            waiters.wait(lambda: self._get_clusterdeployment_readiness(
                expected_condition_cld_fails,
                expected_condition_svc_fails,
                interval=interval),
                timeout=timeout,
                interval=interval,
                timeout_msg=timeout_msg)
        except exceptions.TimeoutError as e:
            raise e

        LOG.info("Cluster checked")

    @staticmethod
    def _check_service_condition_reason(obj: "ClusterDeployment", service_name, service_namespace):
        """
        Check that Helm Service reason is equal expected one
        :param obj: ClusterDeployment
        :param service_name: <str> Name of the service
        :param service_namespace: <str> Namespace of the service
        :return: <bool> True/False
        """
        obj_status = obj.get_service_condition(service_name, service_namespace)
        if not obj_status:
            LOG.warning(f'Service "{service_namespace}/{service_name}" status condition not populated yet. '
                        f'Returning False as condition check result.')
            return False

        if 'reason' in obj_status:
            # Old conditions format for kcm < 1.3.0
            service_type = f'{service_namespace}.{service_name}/SveltosHelmReleaseReady'
            expected_reason = "Managing"
            status = obj_status['status']
            actual_reason = obj_status['reason']
            message = obj_status.get('message')
            log_message = (f"\n > Expected '{service_type}' reason is {expected_reason}, "
                           f"actual reason is {actual_reason}.\n"
                           f" > Status: {status}\n")
            if message:
                log_message += f" > Message: {message}"
            LOG.info(log_message)
            if not eval(status):
                LOG.error(f"'{service_type}' status condition is {status}. Message: {message}")
                raise RuntimeError(message)
            return actual_reason == expected_reason
        elif 'state' in obj_status:
            # New status.services format for kcm >= 1.3.0
            expected_state = "Deployed"
            actual_state = obj_status['state']
            message = obj_status.get('failureMessage') or ''
            log_message = (f"\n > Expected '{service_namespace} / {service_name}' state is {expected_state}, "
                           f"actual state is {actual_state}.\n")
            if message:
                log_message += f" > Message: {message}"
            LOG.info(log_message)
            return actual_state == expected_state
        else:
            raise Exception(f"Unknown service status format: {obj_status}")

    @retry(RuntimeError, delay=40, tries=3, logger=LOG)
    def wait_service_condition(self, service_name, service_namespace, timeout=1200, interval=15):
        """
        Wait clusterdeployment to reach expected Helm service status
        :param service_name: <str> Name of the service
        :param service_namespace: <str> Namespace of the service
        :param timeout: timeout to wait
        :param interval: time between checks
        :return: None
        """
        LOG.info(f"Waiting for '{service_namespace}/{service_name}' service condition readiness ...")
        waiters.wait(lambda: self._check_service_condition_reason(
                         self.clusterdeployment, service_name, service_namespace),
                     timeout=timeout,
                     interval=interval,
                     timeout_msg=f"Timeout for waiting expected '{service_name}/{service_namespace}' service readiness"
                                 f" in cluster {self.clusterdeployment.namespace}/{self.clusterdeployment.name} "
                                 f"after {timeout} sec.")

    def check_actual_expected_pods(self,
                                   expected_pods=None,
                                   exclude_pods=None):
        """Compare expected list of pods (which is fetched automatically,
           unless explicitly provided) with actual list of pods in this
           cluster. Comparison is conducted for all namespaces by default"""

        if settings.SKIP_EXPECTED_POD_CHECK:
            LOG.info("Skipping expected pods checking")
            return

        LOG.info("Checking that all pods and their replicas are in place")

        if not expected_pods:
            if exclude_pods:
                expected_pods = self.clusterdeployment.get_expected_objects(
                    exclude_pods=exclude_pods)
            else:
                expected_pods = self.clusterdeployment.expected_pods

        k8s_utils.wait_expected_pods(self.k8sclient, expected_pods=expected_pods)

    def check_k8s_pods(self, phases=('Running', 'Succeeded'),
                       target_namespaces=None,
                       timeout=settings.WAIT_PODS_READY_TIMEOUT,
                       interval=30, pods_prefix=''):
        """Wait till all expected pods for cluster are in specified
           phase and have Ready=True for all containers
        Args:
            phases: list of expected pod phases
            target_namespaces: namespace (str) or namespaces (list)
                               where pods should be checked
            timeout: timeout to wait
            interval: time between checks
        """
        if not target_namespaces:
            target_namespaces = self.clusterdeployment.get_expected_namespaces()
        self.k8sclient.pods.check_k8s_pods(
            phases=phases,
            target_namespaces=target_namespaces,
            excluded_pods=self.EXCLUDED_PODS,
            excluded_jobs=self.EXCLUDED_JOBS,
            timeout=timeout,
            interval=interval,
            pods_prefix=pods_prefix)

    def check_k8s_nodes(self, timeout=360, interval=10):
        self.k8sclient.nodes.check_k8s_nodes(timeout=timeout, interval=interval)

    def check_pvcs_cleaned_up(self, timeout=300, interval=15):
        """
        Check that all PVCs have been removed.
        """
        def pvcs_deleted():
            pvol_list = self.k8sclient.pvolumeclaims.list_all()
            return len(pvol_list) == 0

        waiters.wait(
            lambda: pvcs_deleted(),
            timeout=timeout,
            interval=interval,
            timeout_msg="Timed out waiting for PVCs to be deleted.",
        )

    def delete_cleanup_and_check_pvcs(self):
        """
        Delete all PVCs.
        """
        pvc_mgr = self.k8sclient.pvolumeclaims
        pvc_nss = []
        for pvc_ns in pvc_mgr.list_all():
            if pvc_ns.namespace not in pvc_nss:
                pvc_nss.append(pvc_ns.namespace)
        if not pvc_nss:
            LOG.info("No PVCs were found, skipping cleanup")
            return
        LOG.info(f"Deleting: {[f"{pvc.namespace}/{pvc.name}" for pvc in pvc_mgr.list_all()]}")
        for ns in pvc_nss:
            LOG.info(f"Removing PVCs in namespace: {ns}")
            pvc_mgr.delete_collection(namespace=ns)
        self.check_pvcs_cleaned_up(
            timeout=180,
            interval=10
        )

    def check_clusterdeployment_deleted(self, timeout=1800, interval=15):
        """
        Check removal process of clusterdeployment. Also verify leftovers absence after removal.
        :param timeout:
        :param interval:
        :return:
        """
        # (va4st): infracluster is mandatory here and should be already initialized.
        infracluster = self.clusterdeployment.clusterobject.infracluster

        waiters.wait(lambda: not self.clusterdeployment.present(),
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting for cluster deletion')
        LOG.info(f"Cluster {self.clusterdeployment.name} has been deleted")

        def status():
            msg = (f"Infracluster {infracluster.kind}:{infracluster.namespace}/{infracluster.name} "
                   f"is exists:{infracluster.exists()} and have readiness status:{infracluster.ready}")
            return msg
        LOG.info(f"Additionally check that infracluster "
                 f"{infracluster.resource_type}:{infracluster.namespace}/{infracluster.name} "
                 f"is removed.")
        waiters.wait(lambda: not infracluster.exists(verbose=True),
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting for cluster deletion',
                     status_msg_function=status)
        LOG.info(f"Infracluster {infracluster.namespace}/{infracluster.name} has been deleted")

    def delete_clusterdeployment_and_check(self, timeout=1800, interval=15):
        """
        Delete clusterdeployment and execute leftover check.
        :param timeout:
        :param interval:
        :return:
        """
        # (va4st): Initialize resource chain before removal
        _ = self.clusterdeployment.clusterobject.infracluster.uid
        # (va4st): Trigger async delete
        self.clusterdeployment.delete()
        LOG.info(f"Removal request sent for cluster deployment "
                 f"{self.clusterdeployment.namespace}/{self.clusterdeployment.name}")

        self.check_clusterdeployment_deleted(timeout, interval)

    def introspect_cap_errors(self, since_seconds=None):
        """Get logs from KCM cap* controllers and show the latest errors"""
        if not settings.ENABLE_INTROSPECT_CAP_ERRORS:
            # cap* controller manager logs check is not enabled
            return

        # Collect logs from all cap* pods and check for errors since last check
        cap_name_pattern = re.compile(r"cap.*-controller-manager-.*")
        pods = self.clusterdeployment._manager.api.pods.list(namespace=settings.KCM_NAMESPACE)
        cap_pods = [pod for pod in pods if cap_name_pattern.match(pod.name)]

        for cap_pod in cap_pods:
            prev_log = self._cap_pods_logs.setdefault(cap_pod.name, [])
            new_log = []
            # Concatenate multiline log to a single line
            for log_line in cap_pod.get_logs(since_seconds=since_seconds).splitlines():
                if new_log and (log_line.startswith(' ') or log_line.startswith('\t')):
                    new_log[-1] += f"\n{log_line}"
                else:
                    new_log.append(log_line)
            # Filter the error lines, and remove lines that already was collected in prev_log
            filtered_log = []
            for log_line in new_log:
                # Filter out the periodic cluster checks
                if "context deadline exceeded" in log_line:
                    continue
                # Check that this is a new 'error' message that is not collected yet in prev_log
                if log_line.startswith('E') and log_line not in prev_log:
                    filtered_log.append(log_line)

            self._cap_pods_logs[cap_pod.name].extend(filtered_log)
            if filtered_log:
                log_lines = '\n'.join(filtered_log[-4:])
                LOG.info(f"\n>>> ENABLE_INTROSPECT_CAP_ERRORS: "
                         f"Found the following errors in the pod '{cap_pod.name}' "
                         f"(4 latest errors max):\n\n{log_lines}\n")

    def wait_aks_machinepools_replicas_populated(self, timeout=300, interval=10):
        """Wait for machinepools created by aks will have in their statuses replicas count

        :param timeout:
        :param interval:
        :return:
        """
        if not self.clusterdeployment.provider == utils.Provider.aks:
            e = (f"Calling specific aks-provider function while cluster provider is "
                 f"{self.clusterdeployment.provider}")
            LOG.error(e)
            raise RuntimeError(e)

        LOG.info(f"Waiting for {timeout} seconds while system machinepool will have replicas count in status")
        waiters.wait(lambda: self.clusterdeployment.clusterobject.infracluster.systempool.replicas,
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting systempool replicas to be populated in status')
        LOG.info(f"Waiting for {timeout} seconds while user machinepool will have replicas count in status")
        waiters.wait(lambda: self.clusterdeployment.clusterobject.infracluster.userpool.replicas,
                     timeout=timeout, interval=interval,
                     timeout_msg='Timeout waiting userpoolpool replicas to be populated in status')
