import json
from exec_helpers import Subprocess

from si_tests.managers import ksi_config_manager
from si_tests import logger
from si_tests import settings
from si_tests.utils import utils

LOG = logger.logger


class AnsibleManager(object):

    SYSTEM_PACKAGES_TO_BE_INSTALLED = ('curl', 'python3-venv', 'python3-dev',
                                       'python3-apt', 'python3-pip',
                                       'libvirt-dev', 'gcc')
    ANSIBLE_PACKAGES_TO_BE_INSTALLED = ('wheel', 'jmespath', 'netaddr',
                                        'ansible', 'Jinja2', 'lxml')
    VENV = "~/ansible-venv"
    SEED_SSH_PRIV_KEY_FILENAME = "id_rsa"

    def __init__(self):
        self.runner = Subprocess()
        self.runner.logger.addHandler(logger.console)
        self.config_manager = ksi_config_manager.KSIConfigManager()

    def _setup_ssh_key_for_ansible_user(self):
        username, key = (
            self.config_manager.get_seed_username_and_private_key())
        key_file = (f"/home/{username}/.ssh/"
                    f"{self.SEED_SSH_PRIV_KEY_FILENAME}")
        if self.runner.check_call(f'[ -f {key_file} ]', verbose=True,
                                  raise_on_err=False).exit_code == 0:
            LOG.info('Private key already exists on the seed node.')
            return
        else:
            LOG.info('Defining SSH key on seed node for stack nodes '
                     'configuration using Ansible.')
            cmd = (
                f"cat << 'EOF' > {key_file}\n"
                f"{key}"
                "EOF\n"
            )
            self.runner.check_call(cmd)
            self.runner.check_call(f"chmod 600 {key_file}")
            self.runner.check_call(
                f'eval "$(ssh-agent -s)"; ssh-add {key_file}',
                verbose=True)

    def _install_required_packages(self):
        if self.runner.check_call(f'[ -d {self.VENV} ]',
                                  verbose=True,
                                  raise_on_err=False).exit_code == 0:
            LOG.info('Ansible has been already installed.')
            return

        LOG.info(f'Installing system packages: '
                 f'{self.SYSTEM_PACKAGES_TO_BE_INSTALLED}')
        self.runner.check_call(
            f'sudo apt -y install {' '.join(
                self.SYSTEM_PACKAGES_TO_BE_INSTALLED)}')

        LOG.info(f'Installing Ansible packages in virtual environment: '
                 f'{self.ANSIBLE_PACKAGES_TO_BE_INSTALLED}')
        self.runner.check_call(f'python3 -m venv --system-site-packages '
                               f'--copies {self.VENV}')
        self.runner.check_call(f'. {self.VENV}/bin/activate; '
                               f'pip3 install --upgrade pip')
        self.runner.check_call(
            f'. {self.VENV}/bin/activate; pip3 install -U {' '.join(
                self.ANSIBLE_PACKAGES_TO_BE_INSTALLED)}', verbose=True)
        self.runner.check_call(
            'mkdir -vp ~/bootstrap ~/.ansible_state', verbose=True)

    def ansible_install_and_init(self):
        """
        Create venv, install Ansible and setup SSH private key for Ansible user
        :return:
        """
        self._install_required_packages()
        self._setup_ssh_key_for_ansible_user()

    def ansible_run(self, playbook, inventory_dir, extra_vars=None):
        """
        Run Ansible playbook.

        :param playbook: Ansible playbook to be run.
        :param inventory_dir: Directory where inventory files are located.
        :param extra_vars: Extra variables to pass to Ansible.
        :return: playbook running result.
        """
        if not extra_vars:
            extra_vars = {}
        elif not isinstance(extra_vars, dict):
            raise ValueError(f"'extra_vars' could be only 'dict', "
                             f"got '{extra_vars}'")

        utils.ksi_seed_check_fuse()

        cmd = [f"-i '{inventory_dir}'"]

        if extra_vars:
            cmd.append(f"-e '{json.dumps(extra_vars)}'")

        cmd.append(playbook)

        ansible_command = ' '.join(cmd)
        ansible_execute = (
            f". {self.VENV}/bin/activate; "
            f"set -ex; "
            f"cd {settings.KSI_ANSIBLE_HOME_PATH}; "
            f"export ANSIBLE_HOME={settings.KSI_ANSIBLE_HOME_PATH}; "
            f"$(which ansible-playbook) {ansible_command}")

        LOG.info(f"Executing Ansible command: '{ansible_command}'")
        return self.runner.check_call(
            ansible_execute, raise_on_err=True, verbose=True)
