#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import functools
import logging
import os
import traceback
import requests
from requests.packages.urllib3.exceptions import InsecureRequestWarning

from si_tests import settings


# suppress iso8601 and paramiko debug logging
class NoDebugMessageFilter(logging.Filter):
    def filter(self, record):
        return not record.levelno <= logging.DEBUG


SECRET_FIELDS = set()


class SecretFilter(logging.Filter):
    """Filter to redact secret information in logs."""

    def filter(self, record):
        """Modify log message to replace secret info with asterisks."""
        log_message = record.getMessage()

        for field in SECRET_FIELDS:
            log_message = log_message.replace(field, "<***>")

        record.msg = log_message  # Override the original message
        record.args = ()  # Prevent formatting errors
        return True  # Allow logging


# Disable multiple notifications like:
# "InsecureRequestWarning: Unverified HTTPS request is being made."
requests.packages.urllib3.disable_warnings(InsecureRequestWarning)

if not os.path.exists(settings.LOGS_DIR):
    os.makedirs(settings.LOGS_DIR)

fmt_string = ('%(asctime)s - %(levelname)s %(filename)s: '
              '%(lineno)d -- %(message)s')

logging.basicConfig(level=logging.DEBUG, format=fmt_string, filename=os.path.join(settings.LOGS_DIR, settings.LOG_NAME),
                    filemode='w')

console = logging.StreamHandler()
console.setLevel(logging.INFO)
console.addFilter(SecretFilter())
formatter = logging.Formatter(fmt_string)
console.setFormatter(formatter)

logger = logging.getLogger(__name__)
logger.addHandler(console)

logging.getLogger('paramiko.transport').addFilter(NoDebugMessageFilter())
logging.getLogger('paramiko.hostkeys').addFilter(NoDebugMessageFilter())
logging.getLogger('iso8601.iso8601').addFilter(NoDebugMessageFilter())
logging.getLogger('requests').addFilter(NoDebugMessageFilter())

if not settings.LOG_REST_DEBUG:
    logging.getLogger('kubernetes.client.rest').addFilter(
        NoDebugMessageFilter())

for n in [m for m in logging.root.manager.loggerDict if
          m.startswith("heatclient")]:
    logging.getLogger(n).addFilter(NoDebugMessageFilter())

for n in [m for m in logging.root.manager.loggerDict if
          m.startswith("keystoneauth")]:
    logging.getLogger(n).addFilter(NoDebugMessageFilter())

for n in [m for m in logging.root.manager.loggerDict if
          m.startswith("urllib3")]:
    logging.getLogger(n).addFilter(NoDebugMessageFilter())


def debug(logger):
    def wrapper(func):
        @functools.wraps(func)
        def wrapped(*args, **kwargs):
            logger.debug(
                "Calling: {} with args: {} {}".format(
                    func.__name__, args, kwargs
                )
            )
            try:
                result = func(*args, **kwargs)
                logger.debug(
                    "Done: {} with result: {}".format(func.__name__, result))
            except BaseException as e:
                logger.error(
                    '{func} raised: {exc!r}\n'
                    'Traceback: {tb!s}'.format(
                        func=func.__name__, exc=e, tb=traceback.format_exc()))
                raise
            return result
        return wrapped
    return wrapper


def banner(message, sep="="):
    """Log a single-string banner"""
    message = f"{sep}{sep} {message} {sep}{sep}"
    message_len = len(message)
    logger.info(f"\n\n{sep * message_len}\n{message}\n{sep * message_len}\n")


def hide(secret):
    """Hide the specified secret string from logging"""
    SECRET_FIELDS.add(secret)


logwrap = debug(logger)
logger.banner = banner
logger.hide = hide
