#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
from tabulate import tabulate

from si_tests.clients.k8s import base
from si_tests import logger

LOG = logger.logger


class V1K0rdentManagement(base.BaseModel):
    pass


class V1K0rdentManagementList(base.BaseModelList):
    pass


class K0rdentManagement(base.K8sClusterScopedResource):
    resource_type = 'management'
    model = V1K0rdentManagement

    @property
    def ready(self):
        result = self.get_conditions(verbose=True)
        if result['not_ready']:
            return False
        else:
            return True

    @property
    def cert_secret_name(self):
        return self.data['spec'].get('core', {}).get('kcm', {}).get('config', {}).get(
            'controller', {}).get('registryCertSecret', None)

    @property
    def release(self):
        return self.data.get('spec', {}).get('release', None)

    def get_conditions(self, verbose):
        result = {
            'ready': [],
            'not_ready': []
        }
        data_status = self.data.get('status') or {}
        conditions = data_status.get('conditions', [])
        if not conditions:
            result['not_ready'].append("No conditions in helmrelease status yet")

        for condition in conditions:
            if condition['status'] in ['True', 'False']:
                ready = eval(condition['status'])
            else:
                ready = False

            condition_type = condition.get('type')
            if ready:
                result['ready'].append(condition_type)
            else:
                result['not_ready'].append(condition_type)

            if verbose:
                condition_message = condition.get('message', '')
                LOG.info(f"Condition ({condition_type}) have message ({condition_message}) and "
                         f"status ({condition['status']})")
        if verbose:
            comp_statuses = self.collect_components_statuses()
            headers = ["Component", "Ready", "Message"]
            data = [[stat['component'],
                     stat['ready'],
                     stat['message']]
                    for stat in comp_statuses]
            message = tabulate(data, tablefmt="presto", headers=headers)
            LOG.info(f"\n{message}\n")
        return result

    def collect_components_statuses(self):
        data_status = self.data.get('status') or {}
        components = data_status.get('components', {})
        components_data = []
        for component in components.keys():
            components_data.append({
                'component': component,
                'ready': True if components[component].get('success', '') else False,
                'message': components[component].get('error', '')
            })
        return components_data


class K0rdentManagementManager(base.K8sBaseManager):
    model = V1K0rdentManagementList
    resource_class = K0rdentManagement
    resource_group = 'k0rdent.mirantis.com'
    resource_version = 'v1beta1'
    resource_plural = 'managements'
