#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s import base
from si_tests import logger

LOG = logger.logger


class V1GCPManagedCluster(base.BaseModel):
    pass


class V1GCPManagedClusterList(base.BaseModelList):
    pass


class GCPManagedCluster(base.K8sNamespacedResource):
    resource_type = 'gcpmanagedcluster'
    model = V1GCPManagedCluster

    def __init__(self, manager, name=None, namespace=None, data=None, log=True):
        """
        GCPManagedCluster object functions
        Represents GCP GKE deployments for k0rdent templates
        """
        self._controlplane = None
        self._machinepool = None
        super().__init__(manager, name, namespace, data, log)

    @property
    def ready(self):
        return self.read().status['ready']

    @property
    def controlplane(self):
        if not self._controlplane:
            # NOTE(va4st): this is k0rdent-specific naming that hardcoded
            # https://github.com/k0rdent/kcm/blob/main/templates/cluster/gcp-gke/templates/_helpers.tpl#L6
            name_pattern = f"{self.name}-cp"
            self._controlplane = self._manager._cluster.gcpmanagedcontrolplanes.get(
                name=name_pattern, namespace=self.namespace)
        return self._controlplane

    @property
    def machinepool(self):
        if not self._machinepool:
            # NOTE(va4st): this is k0rdent-specific naming that hardcoded
            # https://github.com/k0rdent/kcm/blob/main/templates/cluster/gcp-gke/templates/_helpers.tpl#L10
            name_pattern = f"{self.name}-mp"
            self._machinepool = self._manager._cluster.gcpmanagedmachinepools.get(
                name=name_pattern, namespace=self.namespace)
        return self._machinepool


class GCPManagedClusterManager(base.K8sBaseManager):
    model = V1GCPManagedClusterList
    resource_class = GCPManagedCluster
    resource_group = 'infrastructure.cluster.x-k8s.io'
    resource_version = 'v1beta1'
    resource_plural = 'gcpmanagedclusters'
