#    Copyright 2025 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
from si_tests.utils import utils
from si_tests.clients.k8s import base


class V1Cluster(base.BaseModel):
    pass


class V1ClusterList(base.BaseModelList):
    pass


class Cluster(base.K8sNamespacedResource):
    resource_type = 'cluster'
    model = V1Cluster

    def __init__(self, manager, name=None, namespace=None, data=None, log=True):
        self._infracluster = None
        self._infrakind = None
        self._provider = None
        super().__init__(manager, name, namespace, data, log)

    @property
    def provider(self):
        """Return provider object for the cluster

        :return:
        """
        if not self._provider:
            self._provider = utils.Provider.get_provider_by_infrakind(self.infrakind)
        return self._provider

    @property
    def spec(self):
        return self.read().spec

    @property
    def infrakind(self):
        if not self._infrakind:
            self._infrakind = self.spec.get('infrastructureRef', {}).get('kind', '')
        return self._infrakind

    @property
    def infracluster(self):
        """Link to infrastructure cluster object based on provider

        :return:
        """
        if not self._infracluster:
            if self.provider == utils.Provider.aws:
                self._infracluster = self._manager._cluster.awsclusters.get(name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.azure:
                self._infracluster = self._manager._cluster.azureclusters.get(name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.vsphere:
                self._infracluster = self._manager._cluster.vsphereclusters.get(
                    name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.openstack:
                self._infracluster = self._manager._cluster.openstackclusters.get(
                    name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.gcp:
                self._infracluster = self._manager._cluster.gcpclusters.get(name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.gke:
                self._infracluster = self._manager._cluster.gcpmanagedclusters.get(
                    name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.eks:
                self._infracluster = self._manager._cluster.awsmanagedclusters.get(
                    name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.aks:
                self._infracluster = self._manager._cluster.azureasomanagedclusters.get(
                    name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.metal3:
                self._infracluster = self._manager._cluster.metal3clusters.get(
                    name=self.name, namespace=self.namespace)
            elif self.provider == utils.Provider.remotessh:
                self._infracluster = self._manager._cluster.remoteclusters.get(
                    name=self.name, namespace=self.namespace)
            else:
                raise NotImplementedError
        return self._infracluster


class ClusterManager(base.K8sBaseManager):
    model = V1ClusterList
    resource_class = Cluster
    resource_group = 'cluster.x-k8s.io'
    resource_version = 'v1beta1'
    resource_plural = 'clusters'
