#!/usr/bin/env python3
import optparse
import os
import yaml

from si_tests import logger, settings
from si_tests.utils import packaging_version as version


LOG = logger.logger
path = settings.KAAS_RELEASES_FOLDER
mcc_versions_releases_map = {}
files = [f for f in next(os.walk(path))[-1] if f.endswith('.yaml')]
mcc_versions = sorted([f.split('.yaml')[0] for f in files], key=version.parse)
previous_mcc = mcc_versions[-2]
previous_mcc_file = previous_mcc + '.yaml'
current_mcc = mcc_versions[-1]
current_mcc_file = current_mcc + '.yaml'


def get_versions(kaas_version, cl_type):
    with open(f"{path}/{kaas_version}.yaml", 'r') as f:
        kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)
    mcc_version_name = kaas_release_yml.get('metadata', {}).get('name')
    supported_clusterreleases = kaas_release_yml['spec']['supportedClusterReleases']
    releases = [item for item in supported_clusterreleases if item["name"].startswith(cl_type)]
    versions = sorted([v['version'] for v in releases], key=version.parse)
    major_list = []

    # Split every existing release by major version for further sorting. For example, if we have 12.7.0, 12.7.1,
    # 12.7.2 etc... and for example 15.0.1, we will generate list of majors. ["12.7", "15.0"]. It just required for
    # determining previous and latest x.y.any versions.
    for vers in versions:
        major = '.'.join(vers.split('.')[0:2])
        if major not in major_list:
            major_list.append(major)
    major_list_sorted = sorted(major_list, key=version.parse)
    oldest = major_list_sorted[0]
    latest = major_list_sorted[-1]
    previous_list = sorted([v for v in versions if v.startswith(oldest)], key=version.parse)
    latest_list = sorted([v for v in versions if v.startswith(latest)], key=version.parse)
    prev_major = previous_list[0]
    prev_major_name = next((v['name'] for v in releases if v['version'] == prev_major), '')
    major = latest_list[0]
    major_name = next((v['name'] for v in releases if v['version'] == major), '')
    previous_versions_map = {'major': prev_major, 'major_name': prev_major_name,
                             'latest_pr': '', 'latest_pr_name': '',
                             'previous_pr': '', 'previous_pr_name': '',
                             'prevprev': '', 'prevprev_name': ''}
    latest_versions_map = {'major': major, 'major_name': major_name,
                           'latest_pr': '', 'latest_pr_name': '',
                           'previous_pr': '', 'previous_pr_name': ''}

    # if we have 2 versions under the same major, then it means that we will have main major version and latest
    # patchrelease version. For example: in kaas 2.24.2 we have next releases 12.7.0, 12.7.4. In previous_list
    # we have 12.7.0 and 12.7.4. So the major is 12.7.0 and the latest patchrelease is 12.7.4
    if len(previous_list) == 2:
        previous_versions_map['prevprev'] = previous_list[-1]
        previous_versions_map['prevprev_name'] = next(
            (v['name'] for v in releases if v['version'] == previous_list[-1]), '')

    # When prevprev becomes unsupported we use just prev major for this variable. It allows us to use one job
    # not only for first PR release but for any others
    else:
        LOG.warning(f"There is no prevprev release supported. Will be used previous major {prev_major_name} "
                    f"for variable mcc_previous_{cl_type}child_prevprev")
        previous_versions_map['prevprev'] = prev_major
        previous_versions_map['prevprev_name'] = prev_major_name

    # If more than 2 versions are found under the same major version, then it means that we will have
    # previous version of patchreleases. For example, if we have 12.7.0, 12.7.1, 12.7.2, 12.7.3 and 12.7.4 then
    # 12.7.0 is major, 12.7.4 is latest patchrelease, 12.7.3 is previous patchrelease
    if len(latest_list) > 2:
        latest_versions_map['latest_pr'] = latest_list[-1]
        latest_versions_map['previous_pr'] = latest_list[-2]
        latest_versions_map['latest_pr_name'] = next(
            (v['name'] for v in releases if v['version'] == latest_list[-1]), '')
        latest_versions_map['previous_pr_name'] = next(
            (v['name'] for v in releases if v['version'] == latest_list[-2]), '')
    elif len(latest_list) == 2:
        latest_versions_map['latest_pr'] = latest_list[-1]
        latest_versions_map['latest_pr_name'] = next(
            (v['name'] for v in releases if v['version'] == latest_list[-1]), '')
    else:
        # We can't skip case when latest list contains only one version anymore, because starting from 2.27.0 we can
        # develop in scope of previous verions.
        # For example:
        # 2.27.0 contains:
        # - 24.1.5 PR
        # - 24.2.0 MAJOR
        # But 2.27.1 will contain next:
        # - 24.1.6 PR <<==== NEW
        # - 24.2.0 MAJOR <<==== The same as was before
        # So, despite the fact that 2.27.1 is PR release, we will not have 24.2.1, but will have 24.1.6 enstead
        # In this case we should rely on previous version but take max available
        latest_versions_map['latest_pr'] = previous_list[-1]
        latest_versions_map['latest_pr_name'] = next(
            (v['name'] for v in releases if v['version'] == previous_list[-1]), '')
        if len(previous_list) >= 2:
            latest_versions_map['previous_pr'] = previous_list[-2]
            latest_versions_map['previous_pr_name'] = next(
                (v['name'] for v in releases if v['version'] == previous_list[-2]), '')

    previous_versions_map = {k: v for k, v in previous_versions_map.items() if v}
    latest_versions_map = {k: v for k, v in latest_versions_map.items() if v}
    result = {f'previous_{cl_type}_versions': previous_versions_map,
              f'latest_{cl_type}_versions': latest_versions_map,
              'mcc_version_name': mcc_version_name}
    return result


if __name__ == '__main__':
    parser = optparse.OptionParser()
    parser.add_option('-p', '--placeholder',
                      action="store", dest="placeholder",
                      help="child cluster release type(mke, mosk)")
    options, args = parser.parse_args()
    if not options.placeholder:
        parser.error("Empty option. Add -p <mosk|mke> reqiried")
    cluster_type = options.placeholder
    mcc_versions_releases_map['mcc_previous'] = get_versions(previous_mcc, cl_type=cluster_type)
    LOG.info(f"Found next map of {cluster_type} releases for previous mcc version "
             f"{previous_mcc}\n{yaml.dump(mcc_versions_releases_map['mcc_previous'])}")
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_version_name',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_previous']['mcc_version_name'])
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_version_file',
            mode='w') as f:
        f.write(previous_mcc_file)
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_{cluster_type}child_prevmajor',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_previous'][f'previous_{cluster_type}_versions'].get('major_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_{cluster_type}child_prevprev',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_previous'][f'previous_{cluster_type}_versions'].get(
            'prevprev_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_{cluster_type}child_major',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_previous'][f'latest_{cluster_type}_versions'].get('major_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_{cluster_type}child_latest',
            mode='w') as f:
        f.write(
            mcc_versions_releases_map['mcc_previous'][f'latest_{cluster_type}_versions'].get('latest_pr_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_previous_{cluster_type}child_prevlatest',
            mode='w') as f:
        f.write(
            mcc_versions_releases_map['mcc_previous'][f'latest_{cluster_type}_versions'].get('previous_pr_name', ''))
    mcc_versions_releases_map['mcc_current'] = get_versions(current_mcc, cl_type=cluster_type)
    LOG.info(f"Found next map of {cluster_type} releases for current mcc version "
             f"{current_mcc}\n{yaml.dump(mcc_versions_releases_map['mcc_current'])}")
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_current_version_name',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_current']['mcc_version_name'])
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_current_version_file',
            mode='w') as f:
        f.write(current_mcc_file)
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_current_{cluster_type}child_prevmajor',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_current'][f'previous_{cluster_type}_versions'].get('major_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_current_{cluster_type}child_major',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_current'][f'latest_{cluster_type}_versions'].get('major_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_current_{cluster_type}child_latest',
            mode='w') as f:
        f.write(mcc_versions_releases_map['mcc_current'][f'latest_{cluster_type}_versions'].get('latest_pr_name', ''))
    with open(
            f'{settings.ARTIFACTS_DIR}/mcc_current_{cluster_type}child_prevlatest',
            mode='w') as f:
        f.write(
            mcc_versions_releases_map['mcc_current'][f'latest_{cluster_type}_versions'].get('previous_pr_name', ''))
    print(str(mcc_versions_releases_map))
