#!/usr/bin/env python3
import argparse
import os
import sys
import yaml
from collections import namedtuple
from tabulate import tabulate

sys.path.append(os.getcwd())
try:
    from si_tests.utils import packaging_version as version
    from si_tests import settings
    from si_tests.managers.kaas_manager import Manager
except ImportError:
    print("ImportError: Run the application from the si-tests directory or "
          "set the PYTHONPATH environment variable to directory which contains"
          " ./si_tests")
    sys.exit(1)

ReleaseUpgrade = namedtuple('ReleaseUpgrade',
                            'version,numeric_version,destination_cr_ver,upgrade_path_list,'
                            'upgrade_path_str,available_upgrades')
AvailableUpgrade = namedtuple('AvailableUpgrade',
                              'version,numeric_version')


def parse_args():
    parser = argparse.ArgumentParser(description="Release Upgrade Tool")
    parser.add_argument("--provider", choices=["openstack", "vsphere", "byo"], required=True,
                        help="Specify the provider (openstack, vsphere)")
    parser.add_argument("--output", choices=["display", "generate"], required=True,
                        help="Specify command for display upgrade path or generate pairs)")
    parser.add_argument("--upgrade-path-from-yaml", action='store_true',
                        default=settings.USE_UPGRADE_PATH_FROM_YAML,
                        help="Flag to indicate if the upgrade path should be read from a YAML file")
    return parser.parse_args()


def generate_mke_pairs(provider, output, upgrade_path_from_yaml):
    if not settings.KUBECONFIG_PATH or not os.path.isfile(
            settings.KUBECONFIG_PATH):
        print("Please set KUBECONFIG environment variable whith"
              "the path to the kubeconfig file for KaaS management cluster")
        return 11

    kaas_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    mgm_cluster = kaas_manager.get_mgmt_cluster()
    deployed_kaas_release_name = mgm_cluster.get_kaasrelease_version()
    kaas_release = kaas_manager.get_kaasrelease(deployed_kaas_release_name)
    kaas_release = kaas_release.data
    actual_kaas_version = kaas_release.get('spec', {}).get('version', '')
    supported_clusterreleases = kaas_release['spec']['supportedClusterReleases']
    releases = [item for item in supported_clusterreleases if provider in item["providers"]["supported"]]
    mke_releases = [item for item in releases if item["name"].startswith("mke")]
    mke_releases_versions = {mke_release['version']: mke_release['name'] for mke_release in mke_releases}
    mke_releases_latest_version = sorted(mke_releases_versions.keys(), key=version.parse)[-1]
    target_cluster_release_mcc = mke_releases_versions[mke_releases_latest_version]
    excluded_releases = []

    def get_release_by_cr(cr_version):
        return next((item for item in supported_clusterreleases if item["name"] == cr_version), None)

    def get_cr_by_numeric(numeric_ver):
        return next((item["name"] for item in supported_clusterreleases if item["version"] == numeric_ver), None)

    def get_upgrade_path(item, target_ver, excluded_r, rr=False, sm=False):
        nonlocal upgrade_path_res
        for a in item.get('availableUpgrades', []):
            tmp_cr_ver = get_cr_by_numeric(a['version'])
            rr = a.get("rebootRequired", False)
            sm = a.get("skipMaintenance", False)

            if tmp_cr_ver in excluded_r:
                continue

            upgrade_path_res.append(AvailableUpgrade(
                version=tmp_cr_ver,
                numeric_version=a['version']
            ))

            if tmp_cr_ver == target_ver:
                return upgrade_path_res
            else:
                release = get_release_by_cr(tmp_cr_ver)
                get_upgrade_path(release, target_ver, excluded_releases, rr=rr, sm=sm)
        return upgrade_path_res

    res_print = []
    result = []
    result_from_file = []
    upgrade_path_res = []

    def collect_path(releases, target_cr, excluded_release):
        for item in releases:
            nonlocal upgrade_path_res
            upgrade_path_res = list()
            upgrade_path = get_upgrade_path(item, target_cr, excluded_release) or []
            version = item.get("name")
            numeric_version = item.get("version")
            destination_cr_ver = target_cr
            upgrade_path_list = [x.version for x in upgrade_path]
            upgrade_path_str = " -> ".join(upgrade_path_list) if upgrade_path_list else "------"
            available_upgrades = upgrade_path
            if version not in excluded_releases:
                result.append(
                    ReleaseUpgrade(
                        version=version,
                        numeric_version=numeric_version,
                        destination_cr_ver=destination_cr_ver,
                        upgrade_path_list=upgrade_path_list,
                        upgrade_path_str=upgrade_path_str,
                        available_upgrades=available_upgrades
                    )
                )

    def collect_path_from_file(kaas_version, file_dir=settings.UPGRADE_PATHS_YAML_FILE_DIR):
        directory = os.path.dirname(file_dir)
        kaas_version = kaas_version.replace('-rc', '')
        filename = f"{kaas_version}" + ".yaml"
        actual_file_path = os.path.join(directory, filename)
        if not os.path.exists(actual_file_path):
            print(f"File with name {filename} is not existed in {directory}")
            return 1
        nonlocal result_from_file
        with open(actual_file_path, 'r') as paths_file:
            data = yaml.safe_load(paths_file)['upgrade_data']['mke']
            for i in data:
                if not i['OPTIONAL']:
                    release = i['RELEASE']
                    target_release = i['TARGET_RELEASE']
                    paths = i.get('UPGRADE_PATH')
                    for path in paths:
                        if not path['optional']:
                            upgr_path = path['versions']
                            upgrade_path_simple = " -> ".join([v['release_name'] for v in upgr_path])
                            result_from_file.append(
                                (release,
                                 target_release,
                                 ' -> '.join(f"{v['release_name']} ({v['numeric_version']})" for v in upgr_path),
                                 upgrade_path_simple))

    if not upgrade_path_from_yaml:
        collect_path(mke_releases, target_cluster_release_mcc, excluded_releases)
    else:
        collect_path_from_file(kaas_version=actual_kaas_version)

    if output == 'display':
        if not upgrade_path_from_yaml:
            for r in result:
                res_print.append((
                    r.version,
                    r.destination_cr_ver,
                    " -> ".join(["{} ({})".format(x.version, x.numeric_version) for x in r.available_upgrades])
                ))
        else:
            res_print = result_from_file

        print('RR - patchrelease, SM - skip maintenance')
        print('\n' + tabulate(res_print, tablefmt="presto", headers=["From", "To", "Path", "Path simple"]))

    elif output == 'generate':
        formatted_output = []
        if upgrade_path_from_yaml:
            for i in result_from_file:
                formatted_output.append(f"{i[0]},{i[1]},{i[3]}")
        else:
            for r in result:
                upgrade_path_simple = " -> ".join([x.version for x in r.available_upgrades])
                if upgrade_path_simple:
                    formatted_output.append(
                        f"{r.version},{r.destination_cr_ver},{upgrade_path_simple}"
                    )
                else:
                    formatted_output.append(
                        f"{r.version},{r.destination_cr_ver}"
                    )
        print('\n'.join(formatted_output))


args = parse_args()
generate_mke_pairs(args.provider, args.output, args.upgrade_path_from_yaml)
