import pytest

from si_tests import logger
from si_tests.utils import utils
from si_tests.utils import waiters

LOG = logger.logger


def test_api_conversion(show_step, tf_manager):
    """ Check conversation tfoperator api v1alpha1 -> v2

        Scenario:
            1. Check current api version
            2. Enable api conversion
            3. Check TFOperator api v2 status
            4. Check TF DaemonSets after conversation
            5. Check v1alphda1 CR objects were removed
            6. Check backup of v1alpha1 CR
            7. Run TFTests
    """
    show_step(1)
    tf_manager.detect_api_version()
    if tf_manager.apiv2:
        pytest.skip("API v1alpha1 CR wasn't found")
    tf_daemonsets = {"tf-config": [], "tf-config-db": [], "tf-control": [], "tf-webui": []}
    if tf_manager.is_analytics_enabled():
        tf_daemonsets["tf-analytics"] = []
    if not tf_manager.is_vgw_disabled:
        tf_daemonsets["tf-vrouter-vgw"] = []
    for key in tf_daemonsets.keys():
        ds = tf_manager.get_k8s_resource("daemonsets", key)
        tf_daemonsets[key].append(ds.data)
    # Add tf-vrouters daemonsets (include dpdk and vrouters with custom specs)
    for ds in tf_manager.get_vrouter_daemonsets():
        tf_daemonsets[ds.name] = [ds.data]

    show_step(2)
    job = tf_manager.get_api_conversion_job(read=True)
    res_version = job.metadata.resource_version

    tf_helmbundle = tf_manager.get_tf_helmbundle()
    data = tf_helmbundle.data
    data["spec"]["releases"][0]["values"]["operator"] = {"convertToV2": True}
    tf_helmbundle.patch({"spec": data["spec"]})

    LOG.info(f"Wait conversion job is updated. Current resource version: {res_version}")
    waiters.wait(lambda: tf_manager.get_api_conversion_job(read=True).metadata.resource_version != res_version,
                 timeout=180)

    api_conversation_job = tf_manager.wait_api_conversation_job()
    api_conversation_job.wait_succeded(timeout=300)

    show_step(3)
    tf_manager.detect_api_version()
    assert tf_manager.apiv2, "TF Operator still uses v1alpha1 CR"
    tf_manager.wait_tfoperator_healthy(timeout=1200)

    show_step(4)
    # Get updated daemonsets
    for ds in tf_daemonsets.keys():
        tf_daemonsets[ds].append(tf_manager.get_k8s_resource("daemonsets", ds).data)

    for key in tf_daemonsets.keys():
        LOG.info(f"Compare DaemonSet {key} after conversation:")
        res_ver1 = tf_daemonsets[key][0]['metadata']['resource_version']
        res_ver2 = tf_daemonsets[key][1]['metadata']['resource_version']
        LOG.info(f"Resource version: {tf_daemonsets[key][0]['metadata']['resource_version']} "
                 f"-> {tf_daemonsets[key][1]['metadata']['resource_version']}")
        assert res_ver1 != res_ver2, f"DaemonSet {key} wasn't updated"
        containers_v1 = tf_daemonsets[key][0]['spec']['template']['spec']['containers']
        containers_v2 = tf_daemonsets[key][1]['spec']['template']['spec']['containers']
        LOG.debug(f"Container specs:\n"
                  f"apiv1:\n{containers_v1}\napiv2:\n{containers_v2}")
        assert len(containers_v1) == len(containers_v2),\
            "Pods has different amount of containers"
        for c_v1 in containers_v1:
            name = c_v1['name']
            # Api v2 controller creates DPDK DaemonSet containers with 'dpdk' suffix
            if 'dpdk' in key:
                if name == "provisioner":
                    name = f"dpdk-{name}"
            LOG.info(f"Compare container {name}")
            c_v2 = [c_v2 for c_v2 in containers_v2 if c_v2['name'] == name][0]
            if c_v1['env'] != c_v2['env']:
                LOG.warning(f"Env parameters are different:\n"
                            f"apiv1:\n{c_v1['env']}\napiv2:\n{c_v2['env']}")
            assert c_v1['image'] == c_v2['image'],\
                (f"Container {name} difference in env parameters:\n"
                 f"apiv1:\n{c_v1['env']}\napiv2:\n{c_v2['env']}")

    show_step(5)
    # Switch manager to api v1 and try to get v1alpha1 resources
    tf_manager.apiv2 = False
    assert not utils.is_k8s_res_exist(tf_manager.tfoperator())
    assert not utils.is_k8s_res_exist(tf_manager.tfconfig())
    assert not utils.is_k8s_res_exist(tf_manager.tfcontrol())
    assert not utils.is_k8s_res_exist(tf_manager.tfanalytics())
    assert not utils.is_k8s_res_exist(tf_manager.tfvrouter())
    assert not utils.is_k8s_res_exist(tf_manager.tfwebui())
    assert not utils.is_k8s_res_exist(tf_manager.tfimageprecaching())
    assert not utils.is_k8s_res_exist(tf_manager.tftool())
    assert not utils.is_k8s_res_exist(tf_manager.tftest())
    assert not utils.is_k8s_res_exist(tf_manager.tfdbbackup())
    assert not utils.is_k8s_res_exist(tf_manager.tfdbrestore())
    tf_manager.apiv2 = True

    show_step(6)
    configmap = tf_manager.get_confmap_apiv1_backup()
    assert utils.is_k8s_res_exist(configmap)

    show_step(7)
    assert tf_manager.run_pytest() == 'Succeeded'
