#    Copyright 2024 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

import pytest
import re
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Machine
from si_tests.utils import utils, waiters, exceptions

LOG = logger.logger
ovn_controller_pattern = settings.OVN_CONTROLLER_PATTERN_NAME


def patch_daemonset(cluster, pattername):
    dss = cluster.k8sclient.daemonsets.list(
        name_prefix=pattername, namespace="openstack")
    rnd_str = utils.gen_random_string(8)
    for ds in dss:
        annotations = ds.data.get('spec', {}).get('template', {}).get('metadata', {}).get('annotations', {})
        LOG.info(f"Annotations before patch for daemonset {ds.name}:\n{yaml.dump(annotations)}")
        ds.patch(body={"spec": {"template": {"metadata": {
            "annotations": {"si-team/restart-policy-checker-" + rnd_str: ""}}}}})
    dss = cluster.k8sclient.daemonsets.list(name_prefix=pattername, namespace="openstack")
    for ds in dss:
        annotations = ds.data.get('spec', {}).get('template', {}).get('metadata', {}).get('annotations', {})
        LOG.info(f"Annotations after patch for daemonset {ds.name}:\n{yaml.dump(annotations)}")


@pytest.mark.usefixtures("collect_downtime_statistics")
@pytest.mark.usefixtures('post_action_stop_mm_mode')
def test_ovn_controller_restart_after_maintenance(kaas_manager, show_step):
    """Check that openstack controller works correctly with openvswitch-ovn-controller pods restart policy.
       It should restart openvswitch-ovn-controller after removing node maintenance

    Scenario:
            1. Check cluster readiness, get one openvswitch-ovn-controller pod, node and machine name
            2. Initiate cluster maintenance and selected machine maintenance
            3. Patch annotations in ovn daemonset(to change generation) because this is a reboot criteria
            4. Stop machine maintenance and check that pod is restarted by openstack-controller
            5. Stop cluster maintenance and check that we have same pod name
            6. Exit from mm mode (post action)
    """

    managed_ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)

    show_step(1)
    cluster.check.check_k8s_pods()
    cluster.check.check_k8s_pods(target_namespaces="openstack")
    cluster.check.check_cluster_readiness()
    ovn_pods = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                if re.match(ovn_controller_pattern, pod.name)]
    assert ovn_pods, ("No openvswitch-ovn-controller pods found. "
                      "Check Openstack deploy config.")
    LOG.info(f"List of openvswitch-ovn-controller pods: {ovn_pods}")
    machines: list[Machine] = cluster.get_machines()
    show_step(2)
    cluster.cluster_maintenance_start()

    # Check all openvswitch-ovn-controller pods
    for selected_pod in ovn_pods:
        pod_name = selected_pod.name
        pod_node_name = selected_pod.node_name
        target_machine = None
        for machine in machines:
            k8s_node_name = machine.get_k8s_node_name()
            if k8s_node_name == pod_node_name:
                LOG.info(f"Pod {pod_name} found on K8s node name : {k8s_node_name}")
                target_machine = machine
                break
        if not target_machine:
            raise Exception(f"Selected pod: {pod_name} wasn't scaled to any machine in cluster, "
                            f"node_name field from pod: {pod_node_name}")

        target_machine.machine_maintenance_start()

        show_step(3)
        patch_daemonset(cluster, ovn_controller_pattern)

        # Check that selected pod still exists
        try:
            LOG.info(f"Make sure that the pod {pod_name} has not restarted")
            waiters.wait(lambda: selected_pod not in cluster.k8sclient.pods.list(namespace='openstack'),
                         timeout=30, interval=5)
            raise AssertionError(f"Pod {pod_name} has restarted, but shouldn't")
        except exceptions.TimeoutError:
            pass
        LOG.info(f"Selected pod: {pod_name} exists as expected")

        show_step(4)
        target_machine.machine_maintenance_stop()
        cluster.check.check_k8s_pods()
        cluster.check.check_k8s_pods(target_namespaces="openstack")

        # Get list of new ooc pods
        new_ovn_pods = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                        if re.match(ovn_controller_pattern, pod.name)]
        LOG.info(f"List of openvswitch-ovn-controller pods: {new_ovn_pods}")

        assert selected_pod not in new_ovn_pods, (f"The initial pod with name: {pod_name} still "
                                                  f"exists in cluster after nodemaintenancerequest was deleted")

    show_step(5)
    # Get list of new ovn pods before clustermaintenancerequest deletion
    pods_before_clmr = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                        if re.match(ovn_controller_pattern, pod.name)]

    cluster.cluster_maintenance_stop()
    cluster.check.check_k8s_pods()
    cluster.check.check_k8s_pods(target_namespaces="openstack")
    cluster.check.check_cluster_readiness()

    # Check that we have same pods after clustermaintenancerequest deletion
    pods_after_clmr = [pod for pod in cluster.k8sclient.pods.list(namespace='openstack')
                       if re.match(ovn_controller_pattern, pod.name)]
    assert pods_before_clmr == pods_after_clmr, \
        (f"Pods before: {pods_before_clmr} are not same as after "
         f"clustermaintenancerequest deletion: {pods_after_clmr}")
