#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import base64
from datetime import datetime
import socket

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import certs, utils, waiters

LOG = logger.logger


def test_set_mke_custom_cert_mgmt_regional(kaas_manager, show_step):
    """Add custom cert to the regional cluster.

    Scenario:
        1. Collecting env data
        2. Init bootstrap using SI_CONFIG
        3. Apply MKE TLS settings
        4. Wait for kubeconfig to be updated
        5. Check clusters
        6. Update cluster kubeconfig

    """

    def cluster_contains_ca(kaas_manager, cluster, ca):
        ns = kaas_manager.get_namespace(cluster.namespace)
        cluster = ns.get_cluster(cluster.name)

        provider_status = cluster.data['status']['providerStatus']
        api_server_cert = provider_status['apiServerCertificate']
        api_server_cert_decoded = base64.b64decode(api_server_cert).decode("utf-8")

        if ca in api_server_cert_decoded:
            return True
        else:
            return False

    show_step(1)
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(2)
    bootstrap = bootstrap_manager.BootstrapManager.get_si_config_bootstrap_manager()

    show_step(3)
    app = "mke"
    ca_dir_suffix = datetime.now().strftime('%d-%-H-%-M')  # 17-7-37
    ca_path = f'{app}-{ca_dir_suffix}'
    cert_dir = cluster.name

    host = cluster.get_load_balancer_address()
    ips = None
    dns_names = None
    if utils.is_valid_ip(host):
        ips = [host]
    else:
        dns_names = [host]

    cluster_data = cluster.get_lcm_cluster(name=settings.TARGET_CLUSTER, namespace=settings.TARGET_NAMESPACE).read()
    if cluster.provider.name != 'byo':
        control_nodes = cluster_data.spec['machineTypes']['control']
        deploy_params = [x for x in control_nodes if x['name'] == 'deploy'][0]['params']

        # Note
        # ucp_kube_apiserver_port used on providers with lb on nodes
        # kube_apiserver_port used on providers with external lb

        port = deploy_params.get('ucp_kube_apiserver_port') or deploy_params.get('kube_apiserver_port')
        assert port, "Kube API port not found in LCMCluster task 'deploy' for 'control' nodes"
    else:
        port = 443

    cert_pem, key_pem, ca_pem = bootstrap.generate_cert(ips=ips, dns_names=dns_names,
                                                        ca_path=ca_path, cert_dir=cert_dir)

    # Update KUBECONFIG to use IP address with disabled TLS verify,
    # to not lose connection to the cluster during certificate update
    if cluster.is_management:
        lb_address = cluster.get_load_balancer_address()
        try:
            _ip = socket.gethostbyname(lb_address)
        except socket.gaierror as e:
            raise Exception(f"Unable to resolve IP address from loadBalancerIP '{lb_address}':\n{e}")
        kubeconfig_path = certs.update_seed_kubeconfig(bootstrap=bootstrap, cluster=cluster,
                                                       skip_tls_verify=True, new_server_url=f"https://{_ip}:443")
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        ns = kaas_manager.get_namespace(cluster.namespace)
        cluster = ns.get_cluster(cluster.name)

    cert_manager = certs.CertManager(kaas_manager=kaas_manager)
    cert_manager.apply_cert_for_app(app=app, cluster=cluster, hostname=host,
                                    cert_pem=cert_pem, key_pem=key_pem, ca_pem=ca_pem)

    if cluster.is_management:
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
    ns = kaas_manager.get_namespace(cluster.namespace)
    cluster = ns.get_cluster(cluster.name)

    show_step(4)
    waiters.wait(
        lambda: cluster_contains_ca(kaas_manager=kaas_manager, cluster=cluster, ca=ca_pem),
        interval=10, timeout=300,
        timeout_msg="Kubeconfig was not updated with new cert data.")

    cluster.check.check_cert_equal(port, cert=cert_pem)

    show_step(5)
    target_clusters = kaas_manager.get_child_clusters_in_region(cluster.region_name)
    target_clusters.append(cluster)

    for target_cluster in target_clusters:
        LOG.info(f"Checking cluster {target_cluster.name} certificate")
        target_cluster.check.validate_agent_kubeconfig(ca=ca_pem)
        target_cluster.check.validate_helm_controller_secret(ca=ca_pem)
        LOG.info(f"Checking cluster {target_cluster.name} readiness")
        target_cluster.check.check_cluster_readiness()

    show_step(6)
    # Update KUBECONFIG with new CA and actual loadBalancerHost
    kubeconfig_path = certs.update_seed_kubeconfig(
        bootstrap=bootstrap, cluster=cluster, new_ca_pem=ca_pem,
        new_server_url=f"https://{cluster.get_load_balancer_address()}:443")

    cluster.store_k8s_artifacts()
