import pytest

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, utils

LOG = logger.logger


def get_updated_machines_lists(child_cluster, target_distro):
    machines = child_cluster.get_machines()
    # Get distribution versions from machines hosts and convert it to distribution id
    perhostdata = child_cluster.get_nodes_kernel_and_tarfs_versions()
    expected_versions_map_current = child_cluster.get_expected_kernel_version()
    os_version_to_id = {}
    for item in expected_versions_map_current['allowedDistributions']:
        if not item.get('notgreenfield'):
            os_version_to_id[item['os_version']] = item['id']

    for m in machines:
        # Default distribution version to "---", because RHEL distribution codes are undefined yet
        perhostdata[m.name]['id'] = os_version_to_id.get(perhostdata[m.name]['os_version'], "---")

    # Check if the cluster machines are ready for distribution upgrade
    wrong_distro_machines = [f"Machine {m.name} distribution name in spec: '{m.get_distribution()}' , "
                             f"but on the node: '{perhostdata[m.name]['os_version']}'" for m in machines
                             if m.get_distribution() != perhostdata[m.name]['id']]
    assert not wrong_distro_machines, '\n'.join(wrong_distro_machines)

    updated_machines = [m for m in machines if target_distro == perhostdata[m.name]['id']]
    not_updated_machines = [m for m in machines if target_distro != perhostdata[m.name]['id']]

    return updated_machines, not_updated_machines


def test_postponed_distribution_upgrade(kaas_manager):
    """Do not upgrade distribution now, just set the postponeDistributionUpdate flag

    Result will be checked in the 'test_update_child_clusterrelease'
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    child_ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = child_ns.get_cluster(cluster_name)

    target_release_name = child_cluster.clusterrelease_version
    target_distro = kaas_manager.get_latest_available_clusterrelease_distro(target_release_name)
    updated_machines, not_updated_machines = get_updated_machines_lists(child_cluster, target_distro)

    if not not_updated_machines:
        msg = (f"Looks like all Machines in the cluster '{namespace_name}/{cluster_name}' "
               f"already upgraded to the distribution '{target_distro}', skipping inplace upgrade")
        LOG.warning(msg)
        pytest.skip(msg)

    # Enable postponed inplace upgrade
    LOG.info(f"==== Enable postponed distribution upgrade to the distribution '{target_distro}' "
             f"for cluster '{namespace_name}/{cluster_name}'")
    child_cluster.set_postpone_distribution_upgrade(enabled=True)
    for machine in not_updated_machines:
        LOG.info(f"Setting {machine.name} distribution to {target_distro}")
        machine.set_distribution(distribution=target_distro)
    LOG.info("Done.")


@pytest.mark.usefixtures("introspect_child_target_objects")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures('mcc_loadtest_kibana')
@pytest.mark.usefixtures('mcc_loadtest_alertmanager')
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
def test_inplace_distribution_upgrade(kaas_manager):
    """Upgrade distro to the latest available version if possible, and check the result"""
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    child_ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = child_ns.get_cluster(cluster_name)

    target_release_name = child_cluster.clusterrelease_version
    target_distro = kaas_manager.get_latest_available_clusterrelease_distro(target_release_name)
    updated_machines, not_updated_machines = child_cluster.get_machines_by_distribution(target_distro)

    if not not_updated_machines:
        msg = (f"Looks like all Machines in the cluster '{namespace_name}/{cluster_name}' "
               f"already upgraded to the distribution '{target_distro}', skipping inplace upgrade")
        LOG.warning(msg)
        pytest.skip(msg)

    # Get uptimes count before distribution upgrade
    uptimes = child_cluster.get_machines_uptime()

    # Run inplace distribution upgrade
    LOG.info(f"==== Run distribution upgrade to the distribution '{target_distro}' "
             f"for cluster '{namespace_name}/{cluster_name}'")

    # Specific annotation is required for 2.30.x release to allow ubuntu/noble distribution in child cluster.
    # Later ubuntu/noble distro is planned to be allowed unconditionally.
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    mgmt_version = mgmt_cluster.spec['providerSpec']['value']['kaas']['release']
    if target_distro == "ubuntu/noble" and "2-30" in mgmt_version:
        if settings.KAAS_BM_CI_ENABLE_NOBLE_DISTRIBUTION:
            LOG.info(f"Adding allow {target_distro} annotation to {child_cluster.name}")
            child_cluster.add_cluster_annotation("kaas.mirantis.com/allow-ubuntu-noble-distribution", "true")
        else:
            if 'mke' in child_cluster.clusterrelease_version:
                pytest.skip(f"Upgrade to {target_distro} is skipped")
            else:
                LOG.warning(f"Upgrade to {target_distro} is started without allowing annotation")

    for machine in not_updated_machines:
        LOG.info(f"Setting {machine.name} distribution to {target_distro}")
        machine.set_distribution(distribution=target_distro)

    if child_cluster.is_postpone_distribution_upgrade_enabled:
        LOG.info("Disable 'postponeDistributionUpdate' flag to perform inplace distribution upgrade")
        child_cluster.set_postpone_distribution_upgrade(enabled=False)

    # Wait until distribution upgrade is started
    LOG.info("Wait until distribution upgrade is started and Cluster status becomes not ready")
    waiters.wait(lambda: child_cluster.cluster_status.lower() != 'ready',
                 timeout=1200, interval=10)

    # Wait until distribution upgrade is completed
    child_cluster.check.wait_distribution_upgrade_finished(
        timeout=settings.KAAS_CHILD_CLUSTER_UPDATE_TIMEOUT, interval=120)
    # WR for PRODX-46686

    if settings.OPENSTACK_DEPLOY_DPDK:
        machines = child_ns.get_cluster(cluster_name).get_machines()
        LOG.info("Install DPDK driver on compute")
        lkey, lvalue = settings.OPENSTACK_NODES_WITH_HUGEPAGES_LABEL.split('=')
        for machine in machines:
            node_labels = machine.get_k8s_node().read().metadata.labels
            for k, v in node_labels.items():
                if k == lkey and v == lvalue:
                    LOG.info("Get DPDK driver package")
                    distrib_codename = machine.get_distribution().split('/')[-1]
                    openstack_dpdk_driver_package = utils.get_dpdk_driver_package(distrib_codename)
                    LOG.info(f"Install DPDK driver {openstack_dpdk_driver_package}")
                    machine.run_cmd(f"sudo apt install -y {openstack_dpdk_driver_package}", timeout=600)
                    break

    # Check the Cluster objects
    child_cluster.check.check_machines_status()
    child_cluster.check.check_cluster_nodes()
    child_cluster.check.check_k8s_nodes()
    child_cluster.check.check_k8s_pods()
    child_cluster.check.check_cluster_readiness()
    child_cluster.check.check_diagnostic_cluster_status()

    # Get uptimes count after distribution upgrade
    uptimes_after = child_cluster.get_machines_uptime()

    # Check that distros were changed on the hosts
    child_cluster.check.check_inplace_distribution_upgrade_completed(target_distro)

    # Check runtime
    if settings.DESIRED_RUNTIME:
        child_cluster.check.compare_machines_runtime_with_desired(updated_machines)

    # Check that each Machine got at least one reboot if distribution was upgraded
    reboots_msg = ""
    updated_machines_names = [m.name for m in updated_machines]
    for k, v in uptimes.items():
        # It is possible to have 1-second difference between uptime before and after upgrade
        # due to rounding float numbers in calculations.
        time_delta = abs((uptimes_after[k] - v).total_seconds())
        LOG.debug(f"Time difference between uptime before and after upgrade = {time_delta}")
        if k in updated_machines_names:
            # Machine was already upgraded before the test, and should not be rebooted
            if time_delta > 1:
                reboots_msg += (
                    f"Machine '{k}' distribution was already upgraded before test, but machine uptime has been changed:"
                    f"  uptime before='{v}' , uptime after='{uptimes_after[k]}' (must be unchanged).\n")
        else:
            # Machine should be upgraded and rebooted
            if time_delta < 2:
                reboots_msg += (
                    f"Machine '{k}' distribution just upgraded, but machine uptime is still the same: "
                    f"  uptime before='{v}' , uptime after='{uptimes_after[k]}'\n")
    assert not reboots_msg, (
        f"At least 1 reboot is expected during distribution upgrade\n"
        f"{reboots_msg}")
