import pytest
import time

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters

LOG = logger.logger

cluster_name = settings.TARGET_CLUSTER
namespace_name = settings.TARGET_NAMESPACE
rook_ns = settings.ROOK_CEPH_NS


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures("check_ceph_keyrings")
def test_ceph_filesystem(kaas_manager, show_step):
    """Test ceph filesystem availability on Ceph cluster.
    Scenario:
        1. Gather current ceph data
        2. Check ceph health is OK before proceed
        3. Patch ceph cluster with cephFS specification
        4. Check cephFS deployed and cluster health is OK
        5. Check ceph pvc with RWX access mode
        6. Remove cephFS from Ceph cluster
        7. Check cephFS removed and cluster health is OK
    """
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_crd = cluster.get_miracephcluster()
    else:
        ceph_crd = cluster.get_cephcluster()
    k8sclient = cluster.k8sclient

    if 'mos' in cluster.clusterrelease_version:
        pytest.skip('Skip test for mosk-* clusters')

    storage_nodes = [node_name for node_name in cluster.get_ceph_storage_nodes()]
    min_replicas = min(len(storage_nodes), 3)

    show_step(1)
    cephfs_name = f'test-cephfs-{time.time_ns()}'
    active_count = 1
    # get first available device class from ceph cluster pools
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_pools = ceph_crd.data.get('spec', {}).get('pools', [])
    else:
        ceph_pools = ceph_crd.data.get('spec', {}).get('cephClusterSpec', {}).get('pools', [])
    if not ceph_pools:
        pytest.skip('There are no pools available therefore'
                    ' could not get device class for CephFS')
    device_class = ceph_pools[0].get('deviceClass', '')
    assert device_class, 'deviceClass is empty'

    if cluster.workaround.skip_kaascephcluster_usage():
        cephFs = ceph_crd.data.get('spec', {}).get('sharedFilesystem', {})
    else:
        cephFs = ceph_crd.data.get('spec', {}).get('cephClusterSpec', {}).get('sharedFilesystem', {})
    if cephFs:
        pytest.skip('Ceph Filesystem already deployed on Ceph cluster therefore'
                    ' could not verify the desired scenario')

    if cluster.workaround.skip_kaascephcluster_usage():
        nodes = ceph_crd.data['spec']['nodes']
        for node in nodes:
            if 'mon' in node.get('roles', []) and 'mds' not in node.get('roles', []):
                LOG.info(f"!!! adding mds role to {node['name']} node")
                node['roles'].append('mds')
    else:
        nodes = ceph_crd.data['spec']['cephClusterSpec']['nodes']
        for node_name in nodes:
            if 'mon' in nodes[node_name].get('roles', []) and 'mds' not in nodes[node_name].get('roles', []):
                LOG.info(f"!!! adding mds role to {node_name} node")
                nodes[node_name]['roles'].append('mds')

    spec = {
        'nodes': nodes,
        'sharedFilesystem': {
            'cephFS': [
                {
                    'name': cephfs_name,
                    'metadataPool': {
                        'deviceClass': device_class,
                        'replicated': {
                            'size': min_replicas,
                        },
                    },
                    'dataPools': [
                        {
                            'name': 'pool-1',
                            'deviceClass': device_class,
                            'replicated': {
                                'size': min_replicas,
                            },
                            'allowVolumeExpansion': True,
                        },
                        {
                            'name': 'pool-2',
                            'deviceClass': device_class,
                            'replicated': {
                                'size': min_replicas,
                            },
                            'allowVolumeExpansion': True,
                        },
                    ],
                    'metadataServer': {
                        'activeCount': active_count,
                        'activeStandby': False,
                    },
                },
            ],
        },
    }

    show_step(2)
    if cluster.workaround.skip_kaascephcluster_usage():
        health_info = cluster.check.get_miracephhealths_health_status()
    else:
        health_info = cluster.check.get_ceph_health_status()
    assert health_info == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                       f'Current ceph health status: {health_info}'

    show_step(3)
    if cluster.workaround.skip_kaascephcluster_usage():
        patch_data = {'spec': spec}
        cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
    else:
        patch_data = {'spec': {'cephClusterSpec': spec}}
        mgmt_cluster.patch_ceph_data(data=patch_data, crd=ceph_crd)
    cluster.check.check_k8s_pods(target_namespaces=rook_ns)

    show_step(4)
    # If  deployments.wait_ready() return success, it means that all pods started successfully
    LOG.info("Check that deployments in status ready")
    mds_deployments = [r for r in k8sclient.deployments.list(namespace=rook_ns, name_prefix='rook-ceph-mds')
                       if 'canary' not in r.name]
    waiters.wait(
        lambda: cluster.check.check_deployments_ready(deployments_list=mds_deployments),
        timeout=1200, interval=120)

    LOG.info('Check Ceph Filesystem is Ready to use')

    def wait_for_cephfs_status_ready():
        if cluster.workaround.skip_kaascephcluster_usage():
            mc = cluster.get_miracephhealth()
            cephfs_status = mc.data.get('status', {}).get('fullClusterStatus', {}).get(
                'sharedFilesystemStatus', {}).get('cephFsStatus', {}).get(cephfs_name, {})
            return (cephfs_status.get('present', False) and
                    cephfs_status.get('status', '') in ('Ready', 'Connected'))
        else:
            kcc = cluster.get_cephcluster()
            cephfs_status = kcc.data.get('status', {}).get('fullClusterInfo', {}).get(
                'sharedFilesystemStatus', {}).get('cephFsStatus', {}).get(cephfs_name, {})
            return (cephfs_status.get('present', False) and
                    cephfs_status.get('status', '') in ('Ready', 'Connected'))

    waiters.wait(
        lambda: wait_for_cephfs_status_ready(),
        timeout=1200, interval=120)
    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status(timeout=1200)

    ceph_tools_pod = cluster.get_ceph_tool_pod()
    cmd_endpoint = ['/bin/sh', '-c', f'ceph fs subvolumegroup create {cephfs_name} csi']
    ceph_tools_pod.exec(cmd_endpoint)

    show_step(5)
    LOG.info(f'Check Ceph Filesystem PVC for {cephfs_name}-pool-1 data pool')
    cluster.check.check_ceph_pvc(cephfs_enabled=True,
                                 sc_name=f'{cephfs_name}-pool-1',
                                 device_class=device_class)
    LOG.info(f'Check Ceph Filesystem PVC for {cephfs_name}-pool-2 data pool')
    cluster.check.check_ceph_pvc(cephfs_enabled=True,
                                 sc_name=f'{cephfs_name}-pool-2',
                                 device_class=device_class)

    show_step(6)
    LOG.info("Remove Ceph Filesystem from Ceph cluster")
    if cluster.workaround.skip_kaascephcluster_usage():
        ceph_crd = cluster.get_miracephcluster()
        patch_remove = {'spec': {'sharedFilesystem': None}}
        cluster.patch_ceph_data(data=patch_remove, crd=ceph_crd)
    else:
        ceph_crd = cluster.get_cephcluster()
        patch_remove = {'spec': {'cephClusterSpec': {'sharedFilesystem': None}}}
        mgmt_cluster.patch_ceph_data(data=patch_remove, crd=ceph_crd)
    cluster.check.check_k8s_pods(target_namespaces=rook_ns)

    show_step(7)
    LOG.info('Check Ceph Filesystem is removed')

    def check_cephfs_status_removed():
        if cluster.workaround.skip_kaascephcluster_usage():
            mc = cluster.get_miracephhealth()
            cephfs_status = mc.data.get('status', {}).get('fullClusterStatus', {}).get(
                'sharedFilesystemStatus', {}).get('cephFsStatus', {}).get(cephfs_name, {})
            return not cephfs_status
        else:
            kcc = cluster.get_cephcluster()
            cephfs_status = kcc.data.get('status', {}).get('fullClusterInfo', {}).get(
                'sharedFilesystemStatus', {}).get('cephFsStatus', {}).get(cephfs_name, {})
            return not cephfs_status

    waiters.wait(
        lambda: check_cephfs_status_removed(),
        timeout=1200, interval=120)

    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status(timeout=1200)
