import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import waiters
from si_tests.utils.utils import Provider

LOG = logger.logger


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures("introspect_child_deploy_objects")
def test_add_nodes(kaas_manager, show_step, _):
    """Add nodes to the child cluster."""
    # Collecting env data
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE

    # step 001 - Get namespace
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    region = kaas_manager.get_mgmt_cluster().region_name
    facility = cluster.spec['providerSpec']['value']['facility']
    stacklight_label = {"key": "stacklight", "value": "enabled"}
    provider = Provider.equinixmetalv2

    LOG.info("Negative checks for nodeLabels")
    worker_machine = cluster.get_machines(machine_type="worker")[0]
    try:
        non_existing = {"key": "nonexiting", "value": "enabled"}
        worker_machine.add_labels([non_existing])
    except Exception as e:
        LOG.info(e)
        if e.status != 400:
            raise e
    else:
        raise Exception("Unexpected success: 'nonexiting' label was assigned")

    assert non_existing not in worker_machine.nodeLabels, \
        "Not allowed label was successfully set"

    # check duplicates
    # PRODX-8361
    """
    try:
        worker_machine.add_labels([stacklight_label])
    except Exception as e:
        LOG.info(e)
        message = json.loads(e.body)['message']
        if e.status != 400 or 'duplicate' not in message:
            raise e
    else:
        raise Exception("Unexpected success: duplicated label was assigned")

    assert worker_machine.nodeLabels.count(stacklight_label) == 1, \
        "Duplicated label was assigned: {}".format(worker_machine.nodeLabels)
    """

    try:
        non_existing = {"key": "nonexiting", "value": "enabled"}
        cluster.create_equinix_machine(
            node_type="node",
            os=settings.KAAS_EQUINIX_OS,
            billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
            machine_type=settings.KAAS_EQUINIX_MACHINE_TYPE,
            region=region,
            provider_name=provider.provider_name,
            facility=facility,
            node_labels=[non_existing])
    except Exception as e:
        LOG.info(e)
        if e.status != 400:
            raise e
    else:
        raise Exception("Unexpected success: machine was created "
                        "with 'nonexiting' label")

    LOG.info("Adding master machine")
    # step 003 - Add 1 master node
    master_node1 = cluster.create_equinix_machine(
        node_type="master",
        os=settings.KAAS_EQUINIX_OS,
        billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
        machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_MASTER,
        region=region,
        provider_name=provider.provider_name,
        facility=facility)
    LOG.info('Master machine {}'.format(master_node1))
    master_node2 = cluster.create_equinix_machine(
        node_type="master",
        os=settings.KAAS_EQUINIX_OS,
        billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
        machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_MASTER,
        region=region,
        provider_name=provider.provider_name,
        facility=facility)
    LOG.info('Master machine {}'.format(master_node2))

    LOG.info("Adding worker machine")
    # step 004 - Add 1 slave node
    slave_node = cluster.create_equinix_machine(
        node_type="node",
        os=settings.KAAS_EQUINIX_OS,
        billing_cycle=settings.KAAS_EQUINIX_BILLING_CYCLE,
        machine_type=settings.KAAS_EQUINIX_CHILD_MACHINE_TYPE_WORKER,
        region=region,
        provider_name=provider.provider_name,
        facility=facility)
    LOG.info('Added worker machine {}'.format(slave_node))

    for m_node in [master_node1, master_node2]:
        # Waiting for master node become Ready
        cluster.check.wait_machine_status_by_name(machine_name=m_node.name,
                                                  timeout=5800,
                                                  expected_status='Ready')

    # Waiting for slave node become Ready
    cluster.check.wait_machine_status_by_name(machine_name=slave_node.name,
                                              expected_status='Ready')

    # Waiting for machines are Ready
    cluster.check.check_machines_status(timeout=1800)
    cluster.check.check_cluster_nodes()
    cluster.check.check_k8s_nodes()
    cluster.check.check_actual_expected_ceph_pods(
        scale_machines_names_list=[master_node1.name, master_node2.name, slave_node.name])
    # Refresh expected objects
    cluster._refresh_expected_objects()

    new_machine = [x for x in cluster.get_machines(machine_type="worker")
                   if not x.nodeLabels][0]

    LOG.info("Assigning 'stacklight' label to worker machine")
    try:
        new_machine.add_labels([stacklight_label])
    except Exception as e:
        LOG.info(e)

    assert stacklight_label in new_machine.nodeLabels, \
        "Label was not set"
    waiters.wait(lambda: new_machine.check_k8s_nodes_has_labels(),
                 timeout=300, interval=10)

    assert stacklight_label in worker_machine.nodeLabels, \
        "Label 'stacklight' was not found on "\
        "machine {}".format(worker_machine.name)
    # TODO (tleontovich): Uncomment when PRODX-28063 has been done
    # LOG.info("Removing 'stacklight' label from previous worker machine")
    # worker_machine.remove_labels([stacklight_label])
    # assert stacklight_label not in worker_machine.nodeLabels, \
    #     "Label was not removed"
    #
    # waiters.wait(lambda: worker_machine.check_k8s_nodes_has_labels(),
    #              timeout=300, interval=10)

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    # Check/wait for correct docker service replicas in cluster
    ucp_worker_agent_name = cluster.check.get_ucp_worker_agent_name()
    cluster.check.check_actual_expected_docker_services(
        changed_after_upd={'ucp-worker-agent-x': ucp_worker_agent_name})
    cluster.check.check_k8s_pods()
    cluster.check.check_actual_expected_pods()
    try:
        health_info = cluster.check.get_ceph_health_detail()
        assert health_info['status'] == "HEALTH_OK", f'Health is not OK. Will not proceed. ' \
                                                     f'Current ceph health status: {health_info}'
    except AssertionError:
        cluster.check.wait_ceph_health_status('HEALTH_OK')
    cluster.check.check_cluster_readiness()
    cluster.check.check_ceph_pvc(base_image_repo='127.0.0.1:44301')
    cluster.provider_resources.save_artifact()
