import random
import string

from si_tests import logger
from si_tests.lcm.openstack_lcm.policies import Policy
from si_tests.utils.utils import verify


LOG = logger.logger


def test_crud_operations_user_policy(openstack_client_manager, policy_user):
    """Test CRUD operations for user policy
       https://mirantis.testrail.com/index.php?/cases/view/4963686

       Parameters required for test execution:
         - KUBECONFIG

       Setup environment:
       Create user with role member
    """
    user_name, password = policy_user
    http_exception = "HTTP 403"
    os_service = 'keystone'
    os_pod = 'keystone-api'
    policy = Policy(os_service, os_pod)
    letters = string.ascii_lowercase
    name_postfix = ''.join(random.choice(letters) for i in range(5))

    def _delete_resources():
        user_list = openstack_client_manager.user.list([])
        for user in user_list:
            if name_postfix in user.get('Name'):
                openstack_client_manager.user.delete([user.get('ID')])

    try:
        get_policy_file_before_update = ['/bin/sh', '-c',
                                         r' grep -irn "identity:create_user:\|identity:list_users:"'
                                         ' /etc/keystone/policy.d/02-custom.yaml']
        policy.check_policy_absent_before_update(get_policy_file_before_update)

        LOG.info("Step 1: Checking that policies are absent. Only admin can create user and get user list")
        step = 1

        user = openstack_client_manager.user.create([f"test_policy_no_admin_user_{name_postfix}"], combined_output=True)
        verify(http_exception not in user["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. Admin can create a new user")

        user_list = openstack_client_manager.user.list([], combined_output=True)
        verify(http_exception not in user_list["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. Admin can obtain the user list")

        error_output = openstack_client_manager.user.create(
            [f"test_policy_no_user_user_{name_postfix}", f"--os-username={user_name}", f"--os-password={password}"],
            combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. User cannot create a new user")

        error_output = openstack_client_manager.user.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. User cannot get the user list")

        LOG.info("Step 2: Adding new rules to OSDPL file")
        step = 2

        policy_add_role = {'identity:create_user': '!', 'identity:list_users': '!'}
        policy.update_policy_osdpl(policy_add_role, step)

        LOG.info("Step 3: Checking that the policies were added to Keystone API pod")
        step = 3

        check_policy_file = ['/bin/sh', '-c', ' cat /etc/keystone/policy.d/02-custom.yaml']
        policy.wait_policy_updated_on_pod(check_policy_file, policy_add_role, step)

        LOG.info("Step 4: The policies were added. Checking that admin and user cannot create a new user and cannot"
                 " obtain the user list")
        step = 4

        error_output = openstack_client_manager.user.create([f"test_policy_added_admin_user_{name_postfix}"],
                                                            combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone to create a new user. Admin cannot create a new user as"
               f" well")

        error_output = openstack_client_manager.user.list([], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone to get the user list. Admin cannot obtain the user list as"
               f" well")

        error_output = openstack_client_manager.user.create(
            [f"test_policy_added_user_user_{name_postfix}", f"--os-username={user_name}", f"--os-password={password}"],
            combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone to create a new user. User cannot create a new user as well")

        error_output = openstack_client_manager.user.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies disallow everyone to get the user list. User cannot obtain the user list as"
               f" well")

        LOG.info("Step 5: Updating policies. Checking that the policies were added to Keystone API pod")
        step = 5

        policy_update_role = {'identity:create_user': '@', 'identity:list_users': '@'}
        policy.update_policy_osdpl(policy_update_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, policy_update_role, step)

        LOG.info("Step 6: The policies were updated. Checking that admin and user can create a new user and obtain the"
                 " user list")
        step = 6

        user = openstack_client_manager.user.create([f"test_policy_updated_admin_user_{name_postfix}"],
                                                    combined_output=True)
        verify(http_exception not in user["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow everyone to create a new user, admin can create a new user as well")

        user_list = openstack_client_manager.user.list([], combined_output=True)
        verify(http_exception not in user_list["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow everyone to get the user list, admin can obtain the user list as well")

        user = openstack_client_manager.user.create(
            [f"test_policy_updated_user_user_{name_postfix}", f"--os-username={user_name}",
             f"--os-password={password}"], combined_output=True)
        verify(http_exception not in user["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow everyone to create a new user, user can create a new user as well")

        user_list = openstack_client_manager.user.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception not in user_list["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies allow everyone to get the user list, user can obtain the user list as well")

    finally:
        LOG.info("Step 7: Deleting policies. Checking that the policies were deleted and"
                 "\nthe result is the same as in the first step")
        step = 7

        policy_not_role = None
        policy.update_policy_osdpl(policy_not_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, {}, step)

        user = openstack_client_manager.user.create([f"test_policy_deleted_admin_user_{name_postfix}"],
                                                    combined_output=True)
        verify(http_exception not in user["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. Admin can create a new user")

        user_list = openstack_client_manager.user.list([], combined_output=True)
        verify(http_exception not in user_list["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. Admin can obtain the user list")

        error_output = openstack_client_manager.user.create(
            [f"test_policy_deleted_user_user_{name_postfix}", f"--os-username={user_name}",
             f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. User cannot create a new user")

        error_output = openstack_client_manager.user.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Policies are not required for the test",
               f"[Step {step}]: Policies are absent. User cannot get the user list")

        LOG.info("Deleting created resources")

        _delete_resources()
