import pytest

from si_tests import settings
from si_tests import logger
from si_tests.utils import update_release_names
from kubernetes.client.rest import ApiException

LOG = logger.logger


update_release_names = list(update_release_names.generate_update_release_names())
is_update_test_failed = False


@pytest.fixture(scope='function', params=update_release_names,
                ids=[f"RELEASE={x}" for x in update_release_names])
def update_release_name(request):
    global is_update_test_failed
    # Check if the previous update steps failed
    if is_update_test_failed:
        msg = (f"Skip updating clusterrelease to {request.param} because "
               f"previous update step failed")
        LOG.info(msg)
        pytest.skip(msg)

    yield request.param

    # Check the result of the current step
    test_passed = (hasattr(request.node, 'rep_call') and
                   request.node.rep_call.passed)
    if not test_passed:
        is_update_test_failed = True


def test_update_child_clusterrelease_negative_with_disabled_machine(kaas_manager, update_release_name, _):
    """Negative check for Child cluster upgrade attempt with a disabled Machine in the cluster

    Requirements:
    - Child cluster must already have a 'disabled' Machine which was disabled during previous upgrade
    - Disabled Machine should not be an SL Machine unless SL services are in quorum. Same for any other
      services that may block the Child cluster upgrade.

    Expected scenario where this test should be used:
    1. Deploy Child cluster with previous major version (<N-1>.0.0)
    2. Upgrade Child cluster to the latest avaliable patch version of the previous major (<N-1>.<latest>.0)
       using the day2operation upgrade test, where a Machine is disabled during upgrade
    3. Upgrade MCC release to the latest version
    4. >>>>> Run this negative test, trying to upgrade Child cluster to the latest major version (<N>.0.0)
    5. ... Other scenario tests can be run: Enable 'disabled' Machine with the day2operation test
                                            and upgrade the cluster to (<N>.0.0)
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE

    ns = kaas_manager.get_namespace(namespace_name)
    child_cluster = ns.get_cluster(cluster_name)
    cr_before = child_cluster.clusterrelease_version

    if update_release_name == cr_before:
        msg = (f"Requested {update_release_name} is the same as current "
               f"clusterrelease version {cr_before}, skipping update")
        LOG.info(msg)
        pytest.skip(msg)

    LOG.info("Cluster release before update {0}".format(cr_before))
    LOG.banner(f"Updating child cluster {cluster_name} to the clusterrelease {update_release_name}")

    with pytest.raises(ApiException) as exc_info:
        ##########################
        #  Start cluster update  #
        ##########################
        LOG.banner("Start cluster upgrade with outdated disabled Machine", sep='#')
        child_cluster.update_cluster(update_release_name)

        pytest.fail(f"Expected result with 'Bad Request', but Cluster '{child_cluster.name}' "
                    f"accepted upgrade requrest from ClusterRelease version '{cr_before}' to '{update_release_name}'")

    LOG.info(f"Got the following response: {exc_info.value.body}")

    assert exc_info.value.status == 400 and exc_info.value.reason == 'Bad Request', (
        "Expected reason 'Bad Request' when Cluster is updating to new version "
        "while a 'disabled' Machine has outdated release version")

    expected_message = (f"cannot apply upgrade with major version upgrade, cluster "
                        f"{child_cluster.namespace}/{child_cluster.name} has outdated disabled machines")
    assert expected_message in exc_info.value.body, (
        f"Expected message in body: '{expected_message}', but got: '{exc_info.value.body}'")
