# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import random

import pytest

from si_tests import logger
from si_tests.managers.clustercheck_mos_manager import ClusterCheckMosManager

LOG = logger.logger


class TestTFHA(object):
    # Workload details
    lb_url = None

    @staticmethod
    def _set_cls_attr(request, stack_outputs, attr):
        for output in stack_outputs:
            if output['output_key'] == attr:
                setattr(request.cls, attr, output['output_value'])

    @pytest.fixture(scope='class', autouse=True)
    def setup_cls(self, request, openstack_client_manager, tf_manager):
        LOG.info("Create heat stack with loadbalancer before Cassandra cluster tests")
        stack = ClusterCheckMosManager.created_stack_tf_lb(request, openstack_client_manager)
        self._set_cls_attr(request, stack['outputs'], 'lb_url')
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @staticmethod
    def get_restarts_number(pods):
        # Collect restart number for containers
        restarts = {}
        for pod in pods:
            amount = sum([status['restart_count'] for status in
                          pod.data['status']['container_statuses']])
            restarts[pod.data['metadata']['name']] = amount
        return restarts

    def check_restarts_number(self, pods, restarts):
        errors = []
        current_restarts = self.get_restarts_number(pods)
        for pod in pods:
            name = pod.data['metadata']['name']
            if restarts[name] != current_restarts[name]:
                LOG.warning(f"Compare restarts for pod {name}: {restarts[name]} "
                            f"-> {current_restarts[name]}")
                errors.append(f"Restarts number for pod {name} was increased: from {restarts[name]} to "
                              f"{current_restarts[name]}")
            else:
                LOG.info(f"Compare restarts for pod {name}: {restarts[name]} "
                         f"-> {current_restarts[name]}")
        assert not errors, f"Restarts number of checked pods were changed:{chr(10).join(errors)}"

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_ha_tf_control(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Checked TF API availability during the restart of the tf-control pod.

        Scenario:
            1. Delete tf-control pod
            2. Wait tf-control and tf-operator status become ready
            3. Check tf-api readiness
            4. Check vRouter and other tf-control pods aren't restarted
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_tf_control_pods()

        # Choose pod for restart
        pod = random.choice(pods)
        # Remove the restarted pod from the list and add vRouter pods to track restart counters.
        pods.remove(pod)
        vrouter_pods = tf_manager.get_vrouter_pods()
        pods.extend(vrouter_pods)
        restart_numbers = self.get_restarts_number(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_tfcontrol_status(status="NotReady", timeout=60)
        tf_manager.wait_tfcontrol_status(status="Ready")
        tf_manager.wait_tfoperator_healthy(timeout=180)

        show_step(3)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(4)
        self.check_restarts_number(pods, restart_numbers)

        show_step(5)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_ha_tf_config(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Checked TF API availability during the restart of the tf-config pod.

        Scenario:
            1. Delete tf-config pod
            2. Wait tf-config and tf-operator status become ready
            3. Check tf-api readiness
            4. Check vRouter and other tf-config pods aren't restarted
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_tf_config_pods()

        # Choose pod for restart
        pod = random.choice(pods)
        # Remove the restarted pod from the list and add vRouter pods to track restart counters.
        pods.remove(pod)
        vrouter_pods = tf_manager.get_vrouter_pods()
        pods.extend(vrouter_pods)
        restart_numbers = self.get_restarts_number(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_tfconfig_status(status="NotReady", timeout=60)
        tf_manager.wait_tfconfig_status(status="Ready")
        tf_manager.wait_tfoperator_healthy(timeout=180)

        show_step(3)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(4)
        self.check_restarts_number(pods, restart_numbers)

        show_step(5)
        assert ClusterCheckMosManager.is_lb_functional(openstack_client_manager, 2, self.lb_url)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_ha_tf_rabbitmq(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Checked TF API availability during the restart of the tf-rabbitmq pod.

        Scenario:
            1. Delete tf-rabbitmq pod
            2. Wait tf-rabbitmq stateful set becomes ready
            3. Check tf-api readiness
            4. Check vRouter and other tf-rabbitmq pods aren't restarted
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_rabbitmq_pods()

        # Choose pod for restart
        pod = random.choice(pods)
        # Remove the restarted pod from the list and add vRouter pods to track restart counters.
        pods.remove(pod)
        vrouter_pods = tf_manager.get_vrouter_pods()
        pods.extend(vrouter_pods)
        restart_numbers = self.get_restarts_number(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_sts_rabbitmq_ready()
        tf_manager.wait_tfoperator_healthy(timeout=180)

        show_step(3)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(4)
        self.check_restarts_number(pods, restart_numbers)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_ha_tf_redis(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Checked TF API availability during the restart of the tf-redis pod.

        Scenario:
            1. Delete tf-redis pod
            2. Wait tf-redis stateful set becomes ready
            3. Check tf-api readiness
            4. Check vRouter and other tf-redis pods aren't restarted
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_redis_pods()

        # Choose pod for restart
        pod = random.choice(pods)
        # Remove the restarted pod from the list and add vRouter pods to track restart counters.
        pods.remove(pod)
        vrouter_pods = tf_manager.get_vrouter_pods()
        pods.extend(vrouter_pods)
        restart_numbers = self.get_restarts_number(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_sts_redis_ready()
        tf_manager.wait_tfoperator_healthy(timeout=180)

        show_step(3)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(4)
        self.check_restarts_number(pods, restart_numbers)

    @pytest.mark.usefixtures('mos_tf_api_loadtest')
    def test_ha_tf_zookeeper(self, show_step, tf_manager, os_manager, openstack_client_manager):
        """ Checked TF API availability during the restart of the tf-zookeeper pod.

        Scenario:
            1. Delete tf-zookeeper pod
            2. Wait tf-zookeeper stateful set becomes ready
            3. Check tf-api readiness
            4. Check vRouter and other tf-zookeeper pods aren't restarted
            5. Check OpenStack workload (TF LB) functionality of created heat stack (Setup class)
        """

        show_step(1)
        pods = tf_manager.get_zookeeper_pods()

        # Choose pod for restart
        pod = random.choice(pods)
        # Remove the restarted pod from the list and add vRouter pods to track restart counters.
        pods.remove(pod)
        vrouter_pods = tf_manager.get_vrouter_pods()
        pods.extend(vrouter_pods)
        restart_numbers = self.get_restarts_number(pods)
        LOG.info(f"Delete pod {pod.data['metadata']['name']}")
        pod.delete()

        show_step(2)
        tf_manager.wait_sts_zookeeper_ready()
        tf_manager.wait_tfoperator_healthy(timeout=180)

        show_step(3)
        ClusterCheckMosManager.check_contrail_api_readiness(openstack_client_manager)

        show_step(4)
        self.check_restarts_number(pods, restart_numbers)
