from si_tests import settings, logger
from si_tests.managers import bootstrap_manager
from si_tests.managers import openstack_manager
import pytest

LOG = logger.logger


def test_get_cluster_logs(kaas_manager, target_clusters_logs):
    """
    Get cluster logs, collected by bootstrap script
    """
    if not target_clusters_logs:
        msg = "No target cluster namespace/name provided, skipping log collection"
        LOG.warning(msg)
        pytest.skip(msg)

    cluster_namespace, cluster_name = target_clusters_logs.split("/")
    ns = kaas_manager.get_namespace(cluster_namespace)
    cluster = ns.get_cluster(cluster_name)
    collect_mosk_sos_report_logs = False

    if not cluster.is_existed():
        msg = ("Namespace {0} or cluster {1} doesn't "
               "exist".format(cluster_namespace, cluster_name))
        LOG.warning(msg)
        pytest.skip(msg)

    seed_ip = (settings.SEED_STANDALONE_EXTERNAL_IP or
               kaas_manager.si_config.data.get('run_on_remote', {}).get('SEED_STANDALONE_EXTERNAL_IP'))
    if not seed_ip:
        raise Exception("seed_ip has not been detected")

    bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
    try:
        remote = bootstrap.remote_seed()
    except Exception as e:
        LOG.error(e)
        LOG.error("Cannot connect to seed node. Fallback to "
                  "old collect log mechanism (accessing nodes directly)")
        cluster.private_key = settings.KAAS_PRIVATE_KEY_FILE_PATH
        cluster.key_file_var = 'KAAS_PRIVATE_KEY_FILE_PATH'
        cluster.download_kubernetes_logs(raise_on_error=True)
        cluster.download_system_logs(raise_on_error=True)
        return None

    bootstrap_version = kaas_manager.get_kaasrelease_bootstrap_version()
    kaas_binary_version = bootstrap.get_kaas_binary_version()
    LOG.info(f"\nManagement cluster bootstrap version: '{bootstrap_version}'"
             f"\nBootstrap binary version: '{kaas_binary_version}'")
    if bootstrap_version != kaas_binary_version:
        if settings.SEED_STANDALONE_EXTERNAL_IP:
            LOG.info("Bootstrap binary doen's match the Management cluster version, updating the binary")
            bootstrap.update_bootstrap_version(settings.SEED_STANDALONE_EXTERNAL_IP)
        else:
            LOG.warning("Bootstrap binary doen's match the Management cluster version, "
                        "but won't be updated because SEED_STANDALONE_EXTERNAL_IP is not set")

    # Specify sane defaults
    management_kubeconfig_path = ""
    kubeconfig_path = "kubeconfig" \
        if cluster.is_management else "kubeconfig-{}".format(cluster_name)
    # quick hot-fix to be able to pass exact path at seed node, for kubeconfig
    # Must be used only under si_tests/tests/deployment/test_collect_logs.py
    # Required only for stand-alone test_collect_logs run, when bootstrap schme uniq and not in
    # ["~/kubeconfig", "~/bootstrap/dev/kubeconfig"] schemas, used in predicted tests logic below
    if settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH:
        LOG.warning(f"Looking for {settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH} at seed node")
        if remote.isfile(settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH):
            LOG.info(f"Fixed {settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH} exists at seed node, "
                     f"using it as 'management_kubeconfig_path'")
            management_kubeconfig_path = settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH
            if cluster.is_management:
                LOG.info(f"Fixed {settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH} exists at seed node, "
                         f"using it as 'kubeconfig_path'")
                kubeconfig_path = settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH
        else:
            LOG.warning(f"kubeconfig on {settings.KAAS_MGMT_KUBECONFIG_ON_SEED_PATH} not found, "
                        f"trying to use the mgmt kubeconfig {kubeconfig_path}")

    default_key_path = bootstrap.get_bootstrap_ssh_keyfile_path(remote)
    keyfile_path = "id_rsa_{}".format(cluster_name) \
        if settings.KAAS_PRIVATE_KEY_FILE_PATH else default_key_path

    # Optional: BM provider specific case
    if remote.isdir('./bootstrap/dev'):
        bootstrap_dir = 'bootstrap/dev'
        # bm mgmt - kubeconfig in '~/bootstrap/dev'
        # bm region, mgmt-other - kubeconfig in '~/'
        # bm region, mgmt-bm - kubeconfig in '~/bootstrap/dev'
        # bm child - kubeconfig in '~/bootstrap/dev' or we'll copy it there
        # we rely on either of these 2 folders and kubeconfig format:
        # "kubeconfig-{}".format(cluster_name) for non-mgmt cluster
        # kubeconfig - for mgmt
        kubeconfig_path = kubeconfig_path if remote.isfile(kubeconfig_path) \
            else f'{bootstrap_dir}/{kubeconfig_path}'
        keyfile_path = keyfile_path if remote.isfile(keyfile_path) else \
            f'{bootstrap_dir}/{keyfile_path}'
    else:
        bootstrap_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR

    # Optional: Clean seed regional bootstrap specific case
    artifacts_home = settings.SEED_NODE_MCC_ARTIFACTS_BASE_DIR
    if cluster.is_regional and \
            remote.isdir(f"{artifacts_home}/default/{cluster_name}"):
        regional_artifacts = bootstrap.get_regional_artifacts(cluster_name)
        keyfile_path = regional_artifacts["keyfile_path"]
        kubeconfig_path = regional_artifacts["kubeconfig_path"]

    # Optional: Child cluster typical case
    # Always override child kubeconfig file if it is available from secrets
    if not cluster.is_management and not cluster.is_regional:
        LOG.info("Check if child cluster kubeconfig secret exists")
        if cluster.kubeconfig_secret_exists():
            LOG.info("Fetch child cluster kubeconfig secret")
            all_secrets = cluster.get_parent_client().secrets.list(
                namespace=cluster_namespace)
            # We've checked that kubeconfig secret exists earlier
            kubeconfig_secret = [
                x for x in all_secrets
                if x.name == "{}-kubeconfig".format(cluster_name)
            ][0].read()
            LOG.info("Check child cluster kubeconfig in secret is not empty")
            if kubeconfig_secret.data and \
                    'admin.conf' in kubeconfig_secret.data and \
                    kubeconfig_secret.data['admin.conf']:
                _, child_kubeconfig = cluster.get_kubeconfig_from_secret()
                remote.check_call("rm {}|true".format(kubeconfig_path))
                remote.execute("cat << EOF >> {0} \n{1}EOF"
                               .format(kubeconfig_path, child_kubeconfig))
                # Check requirements for collecting logs via osctl sos report
                if cluster.clusterrelease_version.startswith(
                        settings.MOSK_RELEASE_PREFIX):
                    collect_mosk_sos_report_logs = True
            else:
                LOG.debug(f"Child cluster kubeconfig taken from secrets is empty:\n{kubeconfig_secret}")
                if not remote.isfile(kubeconfig_path):
                    kubeconfig_path = ""
                    LOG.warning("Child cluster kubeconfig was not found by file or secret. Continue..")
                else:
                    LOG.warning(f'Reusing the existing kubeconfig: {kubeconfig_path}')
        else:
            if not remote.isfile(kubeconfig_path):
                LOG.warning("Child cluster kubeconfig was not found. Continue..")
                kubeconfig_path = ""
            else:
                LOG.warning(f'Reusing the existing kubeconfig: {kubeconfig_path}')

    if settings.KAAS_PRIVATE_KEY_FILE_PATH:
        LOG.info("Use provided private key")
        with open(settings.KAAS_PRIVATE_KEY_FILE_PATH, 'r') as f:
            id_rsa = f.read()
        remote.check_call("rm {}|true".format(keyfile_path))
        LOG.info("Place private key to seed node to '{}'".format(keyfile_path))
        remote.execute("cat << EOF >> {0} \n{1}EOF"
                       .format(keyfile_path, id_rsa))
        remote.execute("chmod 400 {}".format(keyfile_path))

    LOG.info("Collect cluster logs ({})".format(cluster_name))
    res = bootstrap.step_collect_logs(remote,
                                      cluster_name,
                                      cluster_namespace,
                                      kubeconfig_path,
                                      keyfile_path,
                                      bootstrap_dir,
                                      management_kubeconfig_path=management_kubeconfig_path)

    assert res.exit_code == 0, \
        "Error occurred during collecting logs {}".format(res)

    if collect_mosk_sos_report_logs:
        LOG.info("Collect child cluster logs ({}) via osctl sos report".format(cluster_name))
        child_kubeconfig_name = "kubeconfig-{}".format(cluster_name)
        child_kubeconfig_path = f'{settings.ARTIFACTS_DIR}/{child_kubeconfig_name}'
        LOG.info("Save child cluster kubeconfig to %s", child_kubeconfig_path)
        with open(child_kubeconfig_path, 'w') as f:
            f.write(child_kubeconfig)
        os_manager = openstack_manager.OpenStackManager(kubeconfig=child_kubeconfig_path)
        os_manager.collect_sos_report_logs()
