import re
import pytest
import yaml

from si_tests import logger
from si_tests import settings
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import waiters

LOG = logger.logger


def check_cache_warmup_pods_exist(cluster):
    cache_warmup_name = 'mcc-cache-warmup'
    pods = [pod for pod in cluster.k8sclient.pods.list(namespace='kaas') if
            pod.name.startswith(cache_warmup_name)]
    return pods


class TestCacheWarmup():
    def test_cache_warmup(self, kaas_manager: Manager, show_step):
        """Warmup regional cache before child cluster(s) deployment

         Scenario:
             1. Check init state cluster
             2. Create Cache Warmup request
             3. Wait for job start
             4. Check images and version from cache_warmup pod
             5. Verify warmup regional status for cluster
             6. Verify cluster warmup condition
         """
        # TARGET_CLUSTER has to be either management or regional
        cluster_name = settings.TARGET_CLUSTER
        ns = kaas_manager.get_namespace(namespace='default')
        cluster = ns.get_cluster(cluster_name)

        show_step(1)
        LOG.info(
            f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
        cluster.check.check_machines_status()
        cluster.check.check_cluster_readiness()
        cluster.check.check_k8s_nodes()

        mke_creleases = kaas_manager.si_config.version_generators.gen_child_cluster_release_versions(
            'mke', active_mcc_release_only=True)
        mosk_creleases = kaas_manager.si_config.version_generators.gen_child_cluster_release_versions(
            'mosk', active_mcc_release_only=True)
        mke_supported_warmup = [release for release in mke_creleases]
        mosk_supported_warmup = [release for release in mosk_creleases]
        releases_for_warm_up = mke_supported_warmup + mosk_supported_warmup

        LOG.info(f"Generated releases to warm-up {releases_for_warm_up}")

        cluster_releases = settings.KAAS_CACHE_WARMUP_CLUSTER_RELEASES.split(',') if \
            settings.KAAS_CACHE_WARMUP_CLUSTER_RELEASES else releases_for_warm_up

        openstack_releases = settings.KAAS_CACHE_WARMUP_OPENSTACK_RELEASES.split(',') if \
            settings.KAAS_CACHE_WARMUP_OPENSTACK_RELEASES else []

        if not cluster_releases and not openstack_releases:
            pytest.skip("KAAS_CACHE_WARMUP_OPENSTACK_RELEASES "
                        "and KAAS_CACHE_WARMUP_CLUSTER_RELEASES are empty"
                        "so nothing to warm up")
        show_step(2)

        cwr = cluster.create_cache_warmup_request(cluster_name, cluster_releases, openstack_releases,
                                                  settings.KAAS_CACHE_WARMUP_OPENSTACK_ONLY,
                                                  clients_per_endpoint=settings.KAAS_CACHE_WARMUP_CLIENTS_PER_EP)

        show_step(3)
        # Mgmt or Regional cluster resides in default namespace only
        cluster.wait_warmup_job_creation(namespace='kaas')

        show_step(4)

        waiters.wait(lambda: bool(check_cache_warmup_pods_exist(cluster)),
                     timeout_msg="after creation warmup job warmup"
                                 " pods mcc-cache-warmup are missing")

        # Get logs from pod

        pod = check_cache_warmup_pods_exist(cluster)[0]
        LOG.info("Wait for pod to be Completed...")
        pod.wait_phase(phases=['Succeeded', 'Failed'], timeout=9000)
        pod_logs = pod.get_logs()

        # parse images and version from pod
        pattern = r'for image ([^}]+) loaded'
        matches = re.findall(pattern, pod_logs)

        images_set = {}
        for match in matches:
            image_name, image_version = match.split(':')
            if image_name in images_set:
                if image_version not in images_set[image_name]:
                    images_set[image_name].append(image_version)
            else:
                images_set[image_name] = [image_version]

        logs_yaml = yaml.dump(images_set)
        with open(
                f'{settings.ARTIFACTS_DIR}/cache_images_{cluster.name}.yaml',
                mode='w') as f:
            f.write(logs_yaml)
        LOG.info(f"Loaded images: \n{logs_yaml}")

        show_step(5)
        cluster.wait_for_cluster_warmup_success(cwr.creation_timestamp,
                                                timeout=settings.KAAS_CACHE_WARMUP_READINESS_TIMEOUT)

        show_step(6)
        cluster.wait_for_cluster_condition('CacheWarmup')
