#    Copyright 2020 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.utils import waiters
from si_tests.managers.kaas_manager import Manager


LOG = logger.logger


class TestProvisionEquinixMetalV2MgmCluster():

    @pytest.mark.usefixtures("introspect_standalone_seed_management_deploy_objects")
    def test_bootstrap_kaas_on_equinix_private(self, show_step):
        """Bootstrap MGMT MCC EquinixmetalV2

        Scenario:
            1. Verify seed node accessibility
            2. Prepare seed node to run MCC bootstrap on Equinix
            3. Configure equinix network setup
            4. Patch cluster yaml templates
            5. Generate Keycloak user credentials and store them in si-config.yaml before rendering templates
            6. Check Equinix bootstrap templates for management cluster
            7. Bootstrap MCC at the seed node
            8. Create default MCC users, if requested
            9. Check versions
            10. Store machine artifacts
            11. Update SI_CONFIG with SI_SCENARIO
            12. Update SI_CONFIG with equinixmetalv2 network configuration
            13. Wait for all pods to be Running and Ready
            14. Check that all expected pods exist
            15. Download bootstrap artifacts
        """

        show_step(1)
        assert waiters.icmp_ping(settings.SEED_STANDALONE_EXTERNAL_IP), (
            f"EM2 seed node for management cluster is not accessible by IP {settings.SEED_STANDALONE_EXTERNAL_IP}")
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=settings.SEED_STANDALONE_EXTERNAL_IP)

        # skipping seed node creation, already managed by equinix infra
        show_step(2)
        bootstrap.create_seed_bash_completion()
        bootstrap.step_003_prepare_standalone_seed_node()

        show_step(3)
        management_network_config, region_network_configs, child_network_configs = \
            bootstrap.equinixmetalv2_configure_networking()

        show_step(4)
        bootstrap.equinixmetalv2_patch_templates(settings.CLUSTER_NAMESPACE,
                                                 settings.CLUSTER_NAME,
                                                 management_network_config,
                                                 settings.KAAS_MANAGEMENT_CLUSTER_NODES,
                                                 'templates/equinixmetalv2')
        # step 005
        show_step(5)
        bootstrap.add_credentials_data_to_si_config()

        if settings.SKIP_BOOTSTRAP_TEMPLATES_CHECK:
            LOG.warning("Skip bootstrap templates check due "
                        "SKIP_BOOTSTRAP_TEMPLATES_CHECK flag is set")
            if not settings.KAAS_CUSTOM_LMA_LOGGING_ENABLED:
                bootstrap.disable_lma_logging()
        else:
            show_step(5)
            bootstrap.check_templates()

        show_step(6)
        bootstrap.step_004_deploy_kaas_cluster(extra_config={"network_config": management_network_config})

        if settings.CREATE_DEFAULT_MCC_USERS_VIA_CLI:
            show_step(7)
            bootstrap.step_004_create_default_mcc_users()

        if settings.MCC_SCHEDULED_UPGRADES:
            LOG.info("Update mcc-upgrade crd for feature upgrade scheduling")
            bootstrap.step_006_update_mcc_upgrade_crd()

        show_step(8)
        bootstrap.step_005_check_versions()

        show_step(9)
        kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig"
        if not os.path.isfile(kubeconfig_path):
            raise Exception(f"KUBECONFIG {kubeconfig_path} not found in"
                            f" artifacts!")
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        # store private Ips for equinix mgmt nodes
        mgmt_cluster = kaas_manager.get_mgmt_cluster()
        mgmt_cluster.store_machines_artifacts(public_ip=False)

        show_step(10)
        kaas_manager.si_config.create_si_scenario()

        show_step(11)
        region = kaas_manager.get_mgmt_cluster().region_name
        kaas_manager.si_config.create_equinixmetalv2_network_config(region, management_network_config,
                                                                    region_network_configs, child_network_configs)

        show_step(12)
        bootstrap.step_006_wait_for_pods()

        show_step(13)
        bootstrap.step_006_postdeployment_checks()

        show_step(14)
        bootstrap.step_007_download_bootstrap_artifacts()
