import pytest
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.utils import utils

LOG = logger.logger


@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                               .format(settings.KAAS_CHILD_CLUSTER_NAME)])
@pytest.mark.usefixtures("store_cluster_description")
@pytest.mark.usefixtures('introspect_child_deploy_objects')
def test_kaas_create_aws_child_k8s_lma(kaas_manager, proxy_manager, show_step, _):
    """Deploy AWS child cluster on top of management cluster.

    Scenario:
        1. Collecting env data
        2. Creating namespace
        3. Creating public key
        4. Creating aws credential
        5. Set external proxy
        6. Configure bastion (optional)
        7. Create child cluster and machines with SL label
        8. Check cluster readiness
        9. Pod checks (optional)
        10. Check offline isolation (optional)
        11. Check bastion VM (optional)

    """

    # Collecting env data
    show_step(1)
    cluster_name = settings.KAAS_CHILD_CLUSTER_NAME
    namespace_name = settings.KAAS_NAMESPACE
    release_name = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME
    LOG.info("Available releases: {}".format(
        kaas_manager.get_clusterrelease_names()))
    assert release_name in kaas_manager.get_clusterrelease_names()
    child_dns = settings.KAAS_CHILD_CLUSTER_DNS
    aws_region = settings.AWS_DEFAULT_REGION
    aws_az = settings.AWS_DEFAULT_AZ
    ami = settings.KAAS_CHILD_CLUSTER_MACHINE_IMAGE_NAME
    instance_type = settings.KAAS_CHILD_CLUSTER_MACHINE_FLAVOR_NAME
    master_instance_type = settings.KAAS_CHILD_CLUSTER_MASTER_FLAVOR_NAME \
        if settings.KAAS_CHILD_CLUSTER_MASTER_FLAVOR_NAME else instance_type
    master_disk_type = settings.KAAS_CHILD_CLUSTER_MASTER_AWS_DISK_TYPE
    master_nodes_count = int(settings.KAAS_CHILD_CLUSTER_MASTER_NODES_COUNT)
    slave_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_NODES_COUNT)
    sl_instance_type = settings.KAAS_CHILD_CLUSTER_SLAVE_SL_FLAVOR_NAME or instance_type
    sl_disk_type = settings.KAAS_CHILD_CLUSTER_SLAVE_SL_AWS_DISK_TYPE
    sl_nodes_count = int(settings.KAAS_CHILD_CLUSTER_SLAVE_SL_NODES_COUNT)
    custom_indexes = settings.KAAS_CHILD_CLUSTER_SET_CUSTOM_INDEXES
    indexes = utils.generate_upgrade_indexes()
    stacklight_labels = [{'key': 'stacklight', 'value': 'enabled'}]
    lma_extra_options = yaml.safe_load(settings.KAAS_CHILD_LMA_EXTRA_OPTIONS) \
        if settings.KAAS_CHILD_LMA_EXTRA_OPTIONS else None
    rnd_string = utils.gen_random_string(6)
    region = kaas_manager.get_mgmt_cluster().region_name

    # Creating namespace
    show_step(2)
    LOG.info("Namespace name - %s", namespace_name)
    ns = kaas_manager.get_or_create_namespace(namespace_name)

    # Creating public key
    show_step(3)
    public_key_name = "{cluster_name}-{rnd_string}-key".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Public key name - {public_key_name}".format(
        public_key_name=public_key_name))
    with open(settings.KAAS_CHILD_CLUSTER_PUBLIC_KEY_FILE) as pub_key:
        pub_key_content = pub_key.read()
    ns.create_publickey(public_key_name, pub_key_content)

    # Get aws bootstrap credentials from management cluster
    aws_key_id, aws_secret_access_key = kaas_manager.get_aws_credential(
        "cloud-config", "default")

    # Creating aws credential
    show_step(4)
    credential_name = "{cluster_name}-{rnd_string}-cred".format(
        cluster_name=cluster_name, rnd_string=rnd_string)
    LOG.info("Credential name - {credential_name}".format(
        credential_name=credential_name))
    ns.create_aws_credential(
        credential_name, aws_key_id, aws_secret_access_key, region)

    helm_releases = []
    for hrelease in settings.KAAS_CHILD_CLUSTER_EXTRA_HELM_RELEASES:
        helm_releases.append({"name": hrelease})

    show_step(5)
    if ((settings.KAAS_EXTERNAL_PROXY_ACCESS_STR or
         settings.KAAS_INSTALL_FAKE_PROXY) and
            not settings.KAAS_OFFLINE_DEPLOYMENT):
        LOG.error("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT "
                  "is False.Using Proxy on online deployments makes no sense.")
        raise RuntimeError("Proxy variables is set but KAAS_OFFLINE_DEPLOYMENT"
                           " is False")

    if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
        LOG.info('Proxy mode is enabled. However, external network will not '
                 'be restricted on child cluster nodes. Please consider limit '
                 'internet connection on your own. (Native internet isolation '
                 'is not available for AWS inside si-tests now)')
        # TODO(vryzhenkin): Find out how to correctly set up internet isolation
        # TODO: for aws
        proxy_access_str = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    elif settings.KAAS_INSTALL_FAKE_PROXY:
        proxy_manager.deploy_proxy_sts()
        proxy_manager.deploy_proxy_svc()
        proxy_access_str = proxy_manager.generate_proxy_string()
        proxy_manager.check_proxy_conn()
        proxy_object = ns.create_proxyobject(
            name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
            region=region,
            proxy_str=proxy_access_str)
        proxy_object_name = proxy_object.data['metadata']['name']
    else:
        proxy_object_name = None

    show_step(6)
    if settings.KAAS_BASTION_CONFIG:
        bastion = settings.KAAS_AWS_BASTION_CONFIG
        LOG.info("Use custom bastion config:\n{0}".format(yaml.dump(bastion)))
    else:
        LOG.info("Deploy child cluster without additional bastion config")
        bastion = None

    show_step(7)
    cluster = ns.create_cluster(
        cluster_name,
        release_name,
        credential_name,
        provider="aws",
        region=region,
        aws_region=aws_region,
        aws_az=aws_az,
        aws_priv_subnet_cidr="10.0.0.0/24",
        aws_pub_subnet_cidr="10.0.1.0/24",
        services_cidr="10.96.0.0/16",
        pods_cidr="192.168.0.0/16",
        nodes_cidr="10.10.10.0/24",
        dns=child_dns,
        extra_helm_releases=helm_releases,
        lma_enabled=settings.KAAS_CHILD_CLUSTER_DEPLOY_LMA,
        lma_extra_options=lma_extra_options,
        public_key_name=public_key_name,
        proxy_name=proxy_object_name,
        secure_overlay=settings.KAAS_CHILD_CLUSTER_SECURE_OVERLAY_ENABLED,
        bastion=bastion)

    if custom_indexes:
        LOG.info("Custom indexes is enabled. Random index will be set during machine creation")
    for node in range(master_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_aws_machine(
            node_type="master",
            instance_type=master_instance_type,
            ami=ami,
            root_device_size=120,
            region=region,
            root_device_type=master_disk_type,
            upgrade_index=machine_index
        )

    if sl_nodes_count > 0:
        LOG.info("Adding separate Stacklight nodes to the deployment")
        for node in range(sl_nodes_count):
            machine_index = indexes.__next__() if custom_indexes else None
            cluster.create_aws_machine(
                node_type="node",
                instance_type=sl_instance_type,
                ami=ami,
                root_device_size=200,
                root_device_type=sl_disk_type,
                region=region,
                node_labels=stacklight_labels,
                upgrade_index=machine_index
            )

    for node in range(slave_nodes_count):
        machine_index = indexes.__next__() if custom_indexes else None
        cluster.create_aws_machine(
            node_type="node",
            instance_type=instance_type,
            ami=ami,
            root_device_size=120,
            region=region,
            upgrade_index=machine_index
        )

    show_step(8)
    # Waiting for machines are Ready and helmbundles are deployed
    cluster.check.check_machines_status(timeout=settings.CHECK_MACHINE_STATUS_TIMEOUT)
    cluster.check.check_cluster_nodes()
    cluster.check.check_cluster_readiness()
    cluster.check.check_helmbundles()

    # Collecting artifacts
    cluster.store_k8s_artifacts()

    cluster.check.check_k8s_nodes()

    show_step(9)
    if settings.RUN_POD_CHECKS:
        # Check/wait for correct docker service replicas in cluster
        cluster.check.check_actual_expected_docker_services()
        cluster.check.check_k8s_pods()
        cluster.check.check_actual_expected_pods(timeout=1600,
                                                 check_all_nss=True)
    cluster.provider_resources.save_artifact()

    show_step(10)
    if settings.KAAS_OFFLINE_DEPLOYMENT:
        cluster.check.check_offline_isolation(
            settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
    cluster.check.check_overlay_encryption_functionality()

    show_step(11)
    if bastion:
        LOG.info(f"New bastion IP = {cluster.bastion_ip} for cluster {cluster.name}")
        LOG.info("Check ssh to KaaS machine vie bastion VM")
        machines = cluster.get_machines()
        target_machine = machines[0]
        cluster.check.check_connection_via_bastion(target_machine)
        LOG.info("Check bastion parameters VM in AWS")
        cluster_status = cluster.data.get('status') or {}
        instance_id = cluster_status.get('providerStatus', {}).get('bastion', {}).get('id')
        cluster.check.check_aws_vm(instance_id=instance_id,
                                   instance_type=bastion['instanceType'],
                                   ami_id=bastion['amiId'])
