import os

from functools import cached_property
from kubernetes.client.rest import ApiException

from si_tests.clients.k8s import K8sCluster
from si_tests.settings import KUBECONFIG_PATH
from si_tests.utils import utils


class OsCtlManager:
    def __init__(self, kubeconfig=KUBECONFIG_PATH):
        self.kubeconfig = kubeconfig
        self.os_helm_system_namespace = 'osh-system'
        self.client = None

    @cached_property
    def api(self) -> K8sCluster:
        """
        :rtype: cluster.K8sCluster
        """
        if not os.path.isfile(self.kubeconfig):
            raise FileNotFoundError(
                "kubeconfig file={} not found!".format(self.kubeconfig)
            )
        return K8sCluster(kubeconfig=self.kubeconfig)

    @property
    def oc_name(self):
        if utils.is_rockoon_used(self.api):
            return "rockoon"
        return "openstack-controller"

    def reload_client(self):
        app_instance_label = "openstack-operator"
        app_name_label = "openstack-operator"
        os_ctl_pod_name = self.oc_name

        if utils.is_rockoon_used(self.api):
            app_name_label = "rockoon"
        for pod in self.api.pods.list_starts_with(
            os_ctl_pod_name, self.os_helm_system_namespace
        ):
            pod_obj = pod.read()
            if (
                pod_obj.metadata.labels.get("app.kubernetes.io/instance")
                == app_instance_label
                and pod_obj.metadata.labels.get("app.kubernetes.io/name")
                == app_name_label
            ):
                self.client = pod
                break

    def exec(self, cmd, timeout=120, raise_on_error=True, verbose=False):
        """
        Executes osctl command on openstack-controller pod
        """
        if self.client is None:
            self.reload_client()
        command = ["/bin/sh", "-c", cmd]
        try:
            return self.client.run_exec(
                command, container="osdpl", timeout=timeout, raise_on_error=raise_on_error, verbose=verbose
            )
        except ApiException as e:
            if "Not Found" in e.reason:
                self.reload_client()
                return self.client.run_exec(
                    command, container="osdpl", timeout=timeout, raise_on_error=raise_on_error, verbose=verbose
                )
            raise e
