import exec_helpers
import yaml
import json

from kubernetes.client import ApiException
from websocket import WebSocketException

from si_tests import settings
from si_tests import logger
from si_tests.utils import templates as utils

LOG = logger.logger


class K8SConformanceManager(object):
    """Run k8s tests using conformance image"""

    def __init__(self, cluster):
        """kubectl_client: k8s client to KaaS management cluster"""
        self._cluster = cluster
        self._kubectl_client = cluster.k8sclient

    def run_tests(self, name=settings.K8S_CONFORMANCE_POD_NAME,
                  secret_name='kubeconfig',
                  conformance_version=settings.K8S_CONFORMANCE_IMAGE_VERSION,
                  byo_cluster=False):
        """Run conformance tests

        :param name: str, pod name to run the tests
        :param secret_name: str, secret name with kubeconfig for child cluster
        """
        ns = settings.K8S_CONFORMANCE_NAMESPACE
#        cdn_region = settings.CDN_REGION
#        artifacts_region_map = settings.ARTIFACTS_REGION_MAPPING
#        k8s_image_base = artifacts_region_map[cdn_region]["images_base_url"]
        base_image_repo = self._cluster.determine_mcp_docker_registry()
        k8s_image = f"{base_image_repo}/lcm/kubernetes/k8s-conformance"

        options = {
            'KUBECONFIG_SECRET_NAME': secret_name,
            'K8S_CONFORMANCE_SA': name,
            'K8S_CONFORMANCE_CLUSTER_DOMAIN':
                settings.K8S_CONFORMANCE_CLUSTER_DOMAIN,
            'K8S_CONFORMANCE_IMAGE_URL': k8s_image + ':v' + conformance_version,
            'K8S_CONFORMANCE_USER_DEFINED_SKIP_REGEX':
                settings.K8S_CONFORMANCE_USER_DEFINED_SKIP_REGEX
        }
        options['DISABLE_OFFLINE_LOGIC'] = True if byo_cluster \
            else settings.K8S_CONFORMANCE_DISABLE_OFFLINE_LOGIC

        templates = utils.render_template(
            settings.K8S_CONFORMANCE_POD_YAML, options)
        LOG.debug(templates)
        json_body = json.dumps(yaml.load(templates, Loader=yaml.SafeLoader))
        pod = self._kubectl_client.pods.create(name=name,
                                               namespace=ns,
                                               body=json.loads(json_body))
        pod.wait_phase('Running', timeout=1800, interval=10)
        return pod

    def wait_test_results(
            self, pod, timeout=settings.K8S_CONFORMANCE_RESULTS_WAIT_TIMEOUT):
        pod.wait_test(filepath='./test_*', timeout=timeout)

    def check_test_results(self, pod):
        """Check if the file ./test_passed presents in the pod

        Return True if present, False otherwise
        """
        command = ['/bin/sh', '-c', 'test -f ./test_passed && echo "OK"']
        return pod.exec(command) == "OK\n"

    def download_test_results(self, pod, cluster_type):

        try:
            pod.cp_from_pod(source_dir=settings.K8S_CONFORMANCE_REPORTS_DIR)
        except (WebSocketException, ApiException):
            LOG.error("Connection exception occurred. Retry copy from pod")
            pod.cp_from_pod(source_dir=settings.K8S_CONFORMANCE_REPORTS_DIR)

        exec_helpers.Subprocess().execute(
            "env", verbose=True)

        conformance_reports = exec_helpers.Subprocess().execute(
            "ls -1 artifacts/junit_conformance*").stdout
        serial_report = exec_helpers.Subprocess().execute(
            "ls -1 artifacts/junit_storage*").stdout
        storage_reports = exec_helpers.Subprocess().execute(
            "ls -1 artifacts/junit_storage*").stdout

        if conformance_reports:
            exec_helpers.Subprocess().check_call(
                "junitparser merge {0}/junit_conformance*"
                " {0}/k8s_conformance_{1}_report.xml && "
                "rm {0}/junit_conformance*"
                .format(settings.ARTIFACTS_DIR,
                        cluster_type))

        if serial_report:
            exec_helpers.Subprocess().check_call(
                "mv {0}/junit_serial01.xml {0}/junit_serial01_{1}_report.xml"
                .format(settings.ARTIFACTS_DIR,
                        cluster_type))

        if storage_reports:
            exec_helpers.Subprocess().check_call(
                "junitparser merge {0}/junit_storage*"
                " {0}/k8s_storage_{1}_report.xml && "
                "rm {0}/junit_storage*"
                .format(settings.ARTIFACTS_DIR,
                        cluster_type))
