import os
import yaml

from si_tests import settings
from si_tests import logger
from si_tests.managers import bootstrap_manager
from si_tests.managers.kaas_manager import Manager
from si_tests.utils import utils, waiters


LOG = logger.logger


class IntrospectBootstrap(object):
    def __init__(self, openstack_client=None, JIRA="INTROSPECT-2021", seed_ip=None):
        self.kubeconfig_path = None
        self.bootstrap_manager = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        self.openstack_client = openstack_client
        self.__remote = None
        self.PREF = f"=== {JIRA}: "
        self.kind_container_name = "clusterapi-control-plane"

        self.bastion_ip = None
        self.private_key_path = None
        self.mgmt_ips = None

    @property
    def remote(self):
        if self.__remote:
            return self.__remote

        LOG.debug(f"{self.PREF} Check if the seed node alive")
        try:
            if self.bootstrap_manager.get_seed_ip():
                remote = self.bootstrap_manager.remote_seed()
                remote.check_call('hostname -s')
            else:
                raise Exception("Seed node IP address not found")
        except Exception as e:
            LOG.error(f"{self.PREF} Seed node is not available via SSH: {e}")
            raise
        self.__remote = remote
        return self.__remote

    @property
    def r_base(self):
        remote_base = '.'
        if self.remote.isfile("bootstrap/dev/container-cloud"):
            remote_base = 'bootstrap/dev'
        return remote_base

    @property
    def is_kind_running(self):
        return self.bootstrap_manager.is_kind_running(kind_container_name=self.kind_container_name)

    def get_kubeconfig_path(self):
        kubeconfig_path = "{0}/kubeconfig".format(self.r_base)
        assert self.remote.isfile(kubeconfig_path), (
            f"{self.PREF} Kubeconfig not found on {kubeconfig_path}")
        return kubeconfig_path

    def get_private_key_path(self):
        key_path = self.bootstrap_manager.get_bootstrap_ssh_keyfile_path(
                self.remote)
        assert self.remote.isfile(key_path), (
            f"{self.PREF} SSH private key not found on {key_path}")
        return key_path

    def cmd_run_in_kind(self, cmd):
        res = self.remote.check_call(
            "docker exec clusterapi-control-plane /bin/bash -c \""
            "export KUBECONFIG=/etc/kubernetes/admin.conf && " + cmd + " \"")
        return res

    def cmd_run_in_mgmt(self, cmd):
        kubeconfig_path = self.get_kubeconfig_path()
        res = self.remote.check_call(
            f"PATH=${{PATH}}:~/kaas-bootstrap/bin:"
            f"/home/ubuntu/bootstrap/dev/bin;"
            f"bash -c \"export KUBECONFIG={kubeconfig_path} && " + cmd + " \"")
        return res

    def get_mgmt_ips(self):
        """Return list of IP addresses of lcmmachines"""

        cmd = ("kubectl get lcmmachines -o=jsonpath='{.items[*].status."
               "addresses[?(@.type==\\\"InternalIP\\\")].address}'")
        mgmt_ips = []

        if self.is_kind_running:
            # Try to get lcmmachines from kind cluster first
            res = self.cmd_run_in_kind(cmd)
            mgmt_ips = res.stdout_str.split()

        if not mgmt_ips:
            # Get lcmmachines from management nodes
            res = self.cmd_run_in_mgmt(cmd)
            mgmt_ips = res.stdout_str.split()

        assert len(mgmt_ips) > 0, (
            "LCMMachines were not found on kind/mgmt clusters")
        return mgmt_ips

    def get_bastion_ip(self):
        """Return the bastion IP address"""
        cmd = ("kubectl get cluster -o=jsonpath="
               "'{.items[*].status.providerStatus.bastion.publicIP}'")
        bastion_ip = ""

        if self.is_kind_running:
            # Try to get lcmmachines from kind cluster first
            res = self.cmd_run_in_kind(cmd)
            bastion_ip = res.stdout_str

        if not bastion_ip:
            # Get lcmmachines from management nodes
            res = self.cmd_run_in_mgmt(cmd)
            bastion_ip = res.stdout_str

        assert bastion_ip, (
            "Bastion IP was not found on kind/mgmt clusters")
        return bastion_ip

    def get_ssh_cmd(self, mgmt_ip, private_key_path, bastion_ip=""):
        if bastion_ip:
            return (
                f"ssh -i {private_key_path} mcc-user@{mgmt_ip} "
                f"-oStrictHostKeyChecking=no -o proxycommand='ssh -W %h:%p "
                f"-i {private_key_path} mcc-user@{bastion_ip} "
                f"-oStrictHostKeyChecking=no' ")
        else:
            return (
                f"ssh -i {private_key_path} mcc-user@{mgmt_ip} "
                f"-oStrictHostKeyChecking=no ")

    def get_openstack_instance_details(self, ssh_cmd, mgmt_ip=""):
        """Get instance details from OpenStack"""
        details = {}

        LOG.debug(f"{self.PREF} Check the mgmt node availability)")
        try:
            res = self.remote.check_call(f"{ssh_cmd} 'hostname -s'")
            hostname = res.stdout_str
            details['hostname'] = hostname
        except Exception as e:
            LOG.error(f"{self.PREF} MGMT node '{mgmt_ip}' is not available"
                      f" via SSH: {e}")
            return details

        LOG.debug(f"{self.PREF} Get the mgmt node hypervisor name")
        try:
            server = self.openstack_client.get_server_by_name(
                hostname, verbose=False)._info
            hypervisor_hostname = server["OS-EXT-SRV-ATTR:hypervisor_hostname"]
            instance_name = server["OS-EXT-SRV-ATTR:instance_name"]
            instance_id = server["id"]
            details['os'] = {
                'hypervisor_hostname': hypervisor_hostname,
                'instance_name': instance_name,
                'instance_id': instance_id,
            }
        except Exception as e:
            LOG.error(f"{self.PREF} Unable to retrive the MGMT node "
                      f"'{hostname}' details from OpenStack: {e}")

        return details

    def init(self):
        if not self.bastion_ip:
            self.bastion_ip = self.get_bastion_ip()
        if not self.private_key_path:
            self.private_key_path = self.get_private_key_path()
        if not self.mgmt_ips:
            self.mgmt_ips = self.get_mgmt_ips()

    def introspect_management_deploy_events(self):
        """Collect events after Management/Region cluster deploy, including KinD cluster

        1. Define the name of the kind container
        2. Check if the KinD container is running and it's kind kubeconfig exists
        3. Try to read the events from KinD cluster, using "timeout" directive
        4. Try to read the output from kubectl into yaml
        5. Check if the Cluster kubeconfig exists
        6. Try to read the events from deployed Cluster, using "timeout" directive
        7. Try to read the output from kubectl into yaml
        """

        from kubernetes import client
        from kubernetes.client.models.v1_event_list import V1EventList
        api_client = client.ApiClient()

        def _deserialize_custom_object_list(data):
            result_list = api_client._ApiClient__deserialize(data, V1EventList)
            return result_list

        kind_kubeconfig_path = f"{self.r_base}/.kube/kind-config-clusterapi"
        if self.is_kind_running and self.remote.isfile(kind_kubeconfig_path):
            # Try to get events from kind cluster
            cmd = "timeout 30 kubectl get events --all-namespaces -o yaml"
            res = self.cmd_run_in_kind(cmd)
            kind_events_yaml = yaml.safe_load(res.stdout_str)
            kind_events = _deserialize_custom_object_list(kind_events_yaml)

            msg = "Events with non-Normal resulting status from the KinD cluster (only the latest 10 events):"
            prefix = (f"\n\n"
                      f"\n{'=' * len(msg)}"
                      f"\n{msg}"
                      f"\n{'=' * len(msg)}")
            parsed_kind_events = utils.parse_events(kind_events.items,
                                                    read_event_data=False,
                                                    filtered_events=True)
            events_msg = utils.create_events_msg(parsed_kind_events, prefix=prefix)
            LOG.info(events_msg)
            # Write all KinD events into artifacts
            all_kind_events = utils.parse_events(kind_events.items,
                                                 read_event_data=False,
                                                 filtered_events=False)
            all_events_msg = utils.create_events_msg(all_kind_events)
            with open(f'{settings.ARTIFACTS_DIR}/events_from_kind_cluster.txt', mode='w') as f:
                f.write(all_events_msg)

        cluster_kubeconfig_path = os.path.join(self.r_base, "kubeconfig")
        if self.remote.isfile(cluster_kubeconfig_path):
            # Get events from management nodes
            cmd = "timeout 30 kubectl get events --all-namespaces -o yaml"
            res = self.cmd_run_in_mgmt(cmd)
            cluster_events_yaml = yaml.safe_load(res.stdout_str)
            cluster_events = _deserialize_custom_object_list(cluster_events_yaml)

            msg = "Events with non-Normal resulting status from the Cluster (only the latest 10 events):"
            prefix = (f"\n\n"
                      f"\n{'=' * len(msg)}"
                      f"\n{msg}"
                      f"\n{'=' * len(msg)}")
            parsed_cluster_events = utils.parse_events(cluster_events.items,
                                                       read_event_data=False,
                                                       filtered_events=True)
            events_msg = utils.create_events_msg(parsed_cluster_events, prefix=prefix)
            LOG.info(events_msg)
            # Write all Cluster events into artifacts
            all_cluster_events = utils.parse_events(cluster_events.items,
                                                    read_event_data=False,
                                                    filtered_events=False)
            all_events_msg = utils.create_events_msg(all_cluster_events)
            with open(f'{settings.ARTIFACTS_DIR}/events_from_deployed_cluster.txt', mode='w') as f:
                f.write(all_events_msg)

    def _get_object_events(self, object_name, object_namespace, object_kind, k8s_client):
        events = k8s_client.events.list_starts_with(object_name, namespace=object_namespace)
        parsed_events = utils.parse_events(events, filtered_events=False)
        prefix = f"Events for {object_kind} '{object_namespace}/{object_name}':"
        events_msg = utils.create_events_msg(parsed_events, prefix=prefix, header="\n")
        return events_msg

    def introspect_management_deploy_objects(self, cluster_name, cluster_namespace, seed_ip):
        """Inspect Cluster deploy status for a Management cluster

        1. Get KaaS Manager from:
        - Management cluster if exists and contains Cluster object
        - KinD cluster if exists
        - None otherwise

        2. Get Region cluster client from:
        - Management cluster if exists and contains pods 'lcm-lcm-controller'
        - KinD cluster if exists
        - None otherwise

        3. Get Target client from:
        - Management cluster if exists
        - None otherwise
        """
        # For management cluster
        kind_manager = None
        kaas_manager = None
        region_k8s_client = None
        target_k8s_client = None
        mgmt_kubeconfig_path = None

        if not waiters.icmp_ping(seed_ip, verbose=False):
            LOG.error(f"Seed node is not accessible via IP {seed_ip}, skipping clusters introspection")
            return
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)

        # Check if the KinD cluster is running
        kind_kubeconfig_remote_path = f"{self.r_base}/.kube/kind-config-clusterapi"
        LOG.debug(f"kind_kubeconfig_remote_path: {kind_kubeconfig_remote_path}")
        is_kind_running = bootstrap.is_kind_running(kind_kubeconfig_remote_path=kind_kubeconfig_remote_path)
        if is_kind_running:
            run_on_remote = settings.RUN_ON_REMOTE
            kind_kubeconfig_path = bootstrap.expose_kind_cluster(
                run_on_remote=run_on_remote, kind_kubeconfig_remote_path=kind_kubeconfig_remote_path)
            kind_manager = Manager(kubeconfig=kind_kubeconfig_path)
            LOG.debug(f"KinD kubeconfig found in {kind_kubeconfig_path}")

        # Find the Management cluster kubeconfig: in KinD cluster or on the seed node
        if kind_manager is not None:
            # get mgmt kubeconfig if exists
            secret_name = f"{cluster_name}-kubeconfig"
            if kind_manager.api.secrets.present(secret_name, namespace=cluster_namespace):
                mgmt_kubeconfig_data = kind_manager.get_secret_data(secret_name, cluster_namespace, 'admin.conf')
                if mgmt_kubeconfig_data is not None:
                    mgmt_kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig_from_kind"
                    LOG.info(f"Found management kubeconfig in the KinD secret '{secret_name}', "
                             f"saving to {mgmt_kubeconfig_path}")
                    with open(mgmt_kubeconfig_path, "w") as f:
                        f.write(mgmt_kubeconfig_data)
                else:
                    LOG.debug(f"Secret {cluster_name}-kubeconfig found in KinD cluster,"
                              f" but contains no 'admin.conf' data yet")
            else:
                LOG.debug(f"Secret {cluster_name}-kubeconfig not found in KinD cluster")
        else:
            mgmt_kubeconfig_path, kubeconfig_remote_path = bootstrap.download_remote_kubeconfig(
                kubeconfig_remote_name="kubeconfig", kubeconfig_local_name="management_kubeconfig")
            if mgmt_kubeconfig_path:
                LOG.info(f"Found management kubeconfig '{kubeconfig_remote_path}' on seed node, "
                         f"saved to '{mgmt_kubeconfig_path}'")
            else:
                LOG.debug("KinD is not running, Management cluster kubeconfig not found on seed node")

        # 1. Get KaaS Manager
        if mgmt_kubeconfig_path:
            _kaas_manager = Manager(kubeconfig=mgmt_kubeconfig_path)
            # If Management cluster contains Cluster object, then use the cluster as a KaaS object storage
            if _kaas_manager.api.kaas_clusters.available:
                if _kaas_manager.api.kaas_clusters.present(name=cluster_name, namespace=cluster_namespace):
                    LOG.info("Using Management cluster as a KaaS object storage")
                    kaas_manager = _kaas_manager
                else:
                    LOG.debug(f"'kaas_clusters' found in k8s API on Management cluster,"
                              f" but there is no Cluster object '{cluster_namespace}/{cluster_name}'")
            else:
                LOG.debug("'kaas_clusters' not found in k8s API on Management cluster")
        if kaas_manager is None and is_kind_running:
            # Use KinD cluster as a KaaS object storage
            # TODO(ddmitriev): for bootstrapv2, need to add extra checks for the KinD bootstrap cluster conditions

            # Ensure that KinD cluster has /clusters api-resource
            if kind_manager.api.kaas_clusters.available:
                clusters = kind_manager.api.kaas_clusters.list(namespace=cluster_namespace)
                cluster_names = [cluster.name for cluster in clusters]
                LOG.info(f"Using KinD cluster as a KaaS object storage, found clusters: {cluster_names}")
                kaas_manager = kind_manager
            else:
                LOG.debug("'kaas_clusters' not found in k8s API on KinD cluster")
        if kaas_manager is None:
            LOG.error("Cluster for KaaS object storage not found")

        # 2. Get Regional k8s client
        if kaas_manager and kaas_manager != kind_manager:
            if kaas_manager.api.pods.list_starts_with("lcm-lcm-controller", namespace='kaas'):
                LOG.info("*** Found 'lcm-lcm-controller' pods on Management cluster, assuming it's the LCM cluster")
                LOG.info("Using Management cluster as a Regional k8s client")
                region_k8s_client = kaas_manager.api
            else:
                LOG.info("*** Pods 'lcm-lcm-controller' on Management cluster NOT FOUND")
        if region_k8s_client is None and is_kind_running:
            LOG.info("Using KinD cluster as a Regional k8s client")
            region_k8s_client = kind_manager.api
        if region_k8s_client is None:
            LOG.error("Cluster for Regional k8s client not found")

        # 3. Get Target k8s client
        if mgmt_kubeconfig_path:
            LOG.info("Using Management cluster as a Target k8s client")
            _kaas_manager = Manager(kubeconfig=mgmt_kubeconfig_path)
            target_k8s_client = _kaas_manager.api
        else:
            LOG.error("Cluster for Target k8s client not found")

        self._introspect_cluster_deploy_objects(cluster_name,
                                                cluster_namespace,
                                                kaas_manager,
                                                region_k8s_client,
                                                target_k8s_client)

    def introspect_regional_deploy_objects(self, cluster_name, cluster_namespace, kaas_manager, seed_ip):
        """Inspect Cluster deploy status for a Regional cluster

        1. Get KaaS Manager from:
        - Management cluster

        2. Get Regional cluster client from:
        - Regional cluster if exists and contains pods 'lcm-lcm-controller'
        - KinD cluster if exists
        - None otherwise

        3. Get Target client from:
        - Regional cluster if exists
        - None otherwise
        """
        # For management cluster
        kind_manager = None
        region_k8s_client = None
        target_k8s_client = None
        regional_kubeconfig_path = None

        if not waiters.icmp_ping(seed_ip, verbose=False):
            LOG.error(f"Seed node is not accessible via IP {seed_ip}, skipping clusters introspection")
            return
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)

        # Check if the KinD cluster is running
        kind_kubeconfig_remote_path = f"{self.r_base}/.kube/kind-config-clusterapi"
        LOG.debug(f"kind_kubeconfig_remote_path: {kind_kubeconfig_remote_path}")
        is_kind_running = bootstrap.is_kind_running(kind_kubeconfig_remote_path=kind_kubeconfig_remote_path)
        if is_kind_running:
            si_config = kaas_manager.si_config.data
            run_on_remote = settings.RUN_ON_REMOTE or si_config.get('run_on_remote', {}).get('RUN_ON_REMOTE', False)
            kind_kubeconfig_path = bootstrap.expose_kind_cluster(
                run_on_remote=run_on_remote, kind_kubeconfig_remote_path=kind_kubeconfig_remote_path)
            kind_manager = Manager(kubeconfig=kind_kubeconfig_path)
            LOG.debug(f"KinD kubeconfig found in {kind_kubeconfig_path}")

        # Find the Management cluster kubeconfig: in KinD cluster or on the seed node
        if kind_manager is not None:
            # get regional kubeconfig if exists
            secret_name = f"{cluster_name}-kubeconfig"
            if kind_manager.api.secrets.present(secret_name, namespace=cluster_namespace):
                regional_kubeconfig_data = kind_manager.get_secret_data(secret_name, cluster_namespace, 'admin.conf')
                if regional_kubeconfig_data is not None:
                    regional_kubeconfig_path = f"{settings.ARTIFACTS_DIR}/regional_kubeconfig_from_kind"
                    LOG.info(f"Found regional kubeconfig in the KinD secret '{secret_name}', "
                             f"saving to '{regional_kubeconfig_path}'")
                    with open(regional_kubeconfig_path, "w") as f:
                        f.write(regional_kubeconfig_data)
                else:
                    LOG.debug(f"Secret {cluster_name}-kubeconfig found in KinD cluster,"
                              f" but contains no 'admin.conf' data yet")
            else:
                LOG.debug(f"Secret {cluster_name}-kubeconfig not found in KinD cluster")
        else:
            regional_kubeconfig_path, kubeconfig_remote_path = bootstrap.download_remote_kubeconfig(
                kubeconfig_remote_name=f"kubeconfig-{cluster_name}", kubeconfig_local_name="regional_kubeconfig")
            if regional_kubeconfig_path:
                LOG.info(f"Found regional kubeconfig '{kubeconfig_remote_path}' on seed node, "
                         f"saved to '{regional_kubeconfig_path}'")
            else:
                LOG.debug("KinD is not running, Region cluster kubeconfig not found on seed node")

        # 1. Log KaaS Manager
        LOG.info("Using Management cluster as a KaaS object storage")

        # 2. Get Regional k8s client
        if regional_kubeconfig_path:
            _region_manager = Manager(kubeconfig=regional_kubeconfig_path)
            if _region_manager.api.pods.list_starts_with("lcm-lcm-controller", namespace='kaas'):
                LOG.info("*** Found 'lcm-lcm-controller' pods on Regional cluster, assuming it's the LCM cluster")
                LOG.info("Using Regional cluster as a Regional k8s client")
                region_k8s_client = _region_manager.api
            else:
                LOG.info("*** Pods 'lcm-lcm-controller' on Regional cluster NOT FOUND")
        if region_k8s_client is None and is_kind_running:
            LOG.info("Using KinD cluster as a Regional k8s client")
            region_k8s_client = kind_manager.api
        if region_k8s_client is None:
            LOG.error("Cluster for Regional k8s client not found")

        # 3. Get Target k8s client
        if regional_kubeconfig_path:
            LOG.info("Using Regional cluster as a Target k8s client")
            _region_manager = Manager(kubeconfig=regional_kubeconfig_path)
            target_k8s_client = _region_manager.api
        else:
            LOG.error("Cluster for Target k8s client not found")

        self._introspect_cluster_deploy_objects(cluster_name,
                                                cluster_namespace,
                                                kaas_manager,
                                                region_k8s_client,
                                                target_k8s_client)

    def introspect_child_deploy_objects(self, cluster_name, cluster_namespace, kaas_manager):
        """Inspect Cluster deploy status for a Child cluster

        1. Get KaaS Manager from:
        - Management cluster

        2. Get Regional cluster client from:
        - Regional cluster, if Child cluster object created
        - None otherwise

        3. Get Target client from:
        - Regional cluster if exists
        - None otherwise
        """
        region_k8s_client = None
        target_k8s_client = None

        # 1. Log KaaS Manager
        LOG.info("Using Management cluster as a KaaS object storage")

        # 2. Get Regional k8s client
        ns = kaas_manager.get_namespace(cluster_namespace)
        child_cluster = ns.get_cluster(cluster_name)
        if child_cluster.is_existed():
            region_k8s_client = child_cluster.get_parent_client()
            LOG.info(f"Regional k8s client found for the Child cluster '{cluster_namespace}/{cluster_name}'")
        else:
            LOG.error("Regional cluster not found because Child cluster was not created")

        # 3. Get Target k8s client
        if region_k8s_client is not None:
            try:
                target_k8s_client = child_cluster.k8sclient
                LOG.info(f"Target k8s client found for the child cluster '{cluster_namespace}/{cluster_name}'")
            except Exception as e:
                LOG.error(f"Failed to get k8s client for the child cluster: {e}")
                target_k8s_client = None
        else:
            LOG.error("Target cluster not found because Child cluster was not created")

        self._introspect_cluster_deploy_objects(cluster_name,
                                                cluster_namespace,
                                                kaas_manager,
                                                region_k8s_client,
                                                target_k8s_client)

    def introspect_management_upgrade_objects(self, kaas_manager):
        """Check all objects status from Management and Regional clusters after upgrade"""
        # 1. Check Management cluster
        mgmt_cluster = kaas_manager.get_mgmt_cluster()
        region_k8s_client = kaas_manager.api
        target_k8s_client = region_k8s_client
        self._introspect_cluster_deploy_objects(mgmt_cluster.name, mgmt_cluster.namespace,
                                                kaas_manager, region_k8s_client, target_k8s_client)
        # 2. Check all Regional clusters
        regional_clusters = kaas_manager.get_regional_clusters()
        for regional_cluster in regional_clusters:
            region_k8s_client = regional_cluster.k8sclient
            target_k8s_client = region_k8s_client
            self._introspect_cluster_deploy_objects(regional_cluster.name, regional_cluster.namespace,
                                                    kaas_manager, region_k8s_client, target_k8s_client)

    def _introspect_cluster_deploy_objects(self, cluster_name, cluster_namespace,
                                           kaas_manager, region_k8s_client, target_k8s_client):
        """Check all the objects status related to the Cluster creation process

        Input data:
        - cluster_name and cluster_namespace: metadata of the Cluster object to check
        - kaas_manager: Manager of the user-created KaaS objects (Cluster, Machine, etc)
                        Management cluster, if deployed; or KinD cluster if Management is in the middle of deploy.
                        *None* if nor KinD neither Management clusters contain the "Cluster" object to check.
        - region_k8s_client: K8S API client to the cluster where LCM objects are created (LCMCluster, LCMMachine, etc).
                             KinD cluster, if a Management or Regional cluster deploy is in progress;
                             or Management|Regional if deploy of Management|Regional MKE cluster is completed;
                             or Management|Regional in case if Child cluster is deploying.
                             *None* if no KinD|Region|Management cluster is available for the Cluster object.
        - target_k8s_client: K8S API of the deployed MKE cluster.
                             *None* if no "kubeconfig" object found for the Cluster in parent's cluster secrets
                             or on the seed node.

        Expected phases:

        1. Error while preparing the deploy or running the KinD:
            Expected objects status:
            - kaas_manager is None (Management KinD is absent or Management cluster kubeconfig is missing)
            Objects to check:
            - Nothing to check

        2. Error while creating Cluster, Machine or other user-defined objects:
            Expected objects status:
            - kaas_manager exists, but missing the Cluster object, Machine objects or other user-defined objects
            Objects to check:
            - Admission controller logs can be checked

        3. Error while creating the LCMCluster and LCMMachine objects (lcm-controller errors):
            Expected objects status:
            - Cluster and Machine objects were created
            Objects to check:
            - The Cluster-related LCMCluster and LCMMachine objects presense
            - lcm-controller logs (?)

        4. Error while provision the Cluster nodes (interactions with a cloud provider):
            Expected objects status:
            - region_k8s_client exists (KinD or Regional cluster is running)
            - in kaas_manager, present Management cluster object, Machine objects and other user-defined objects
            - in region_k8s_client, present LCMCluster, LCMMachines and other provisioning-related objects
            - [BM,EM2] in region_k8s_client, present BareMetalHosts, KaasCephCluster and other provider-related objects
            Objects to check:
            - [BM, EM2] BareMetalHosts status, if exists; dnsmasq/ironic/tftpd logs related to the host's MAC/IP
            - Provider controller events related to the LCMMachine that doen't have status
            - Network connectivity from the seed node to the Management cluster nodes and bastion node (for details)

        5. Error while deploying the Cluster nodes (LCM tasks):
            Expected objects status:
            - region_k8s_client exists (KinD or Regional cluster is running)
            - in kaas_manager, Machine objects have non-empty status, where "ProviderInstance" condition is 'true'
            Objects to check:
            - Machines status conditions, if exist
            - "Swarm" condition in the Machines object status
            - LCMMachines status fields from "stateItemStatuses", if exist
            - Logs from LCM tasks on the nodes (filtered by the failed tasks)

        6. Error while deploying KaaS services on the Cluster (helmbundle tasks):
            Expected objects status:
            - kaas_manager exists (Management cluster is accessible via KUBECONFIG) how to check that MKE is running?
            - in region_k8s_client, LCMMachines have not in empty nor Provision status
              (? better to check the Cluster status)
            Objects to check:
            - Cluster conditions status
            - Helmbundle charts status
            - Deployments/StatefulSets/ReplicaSets status related to helmbundle charts (or even all available?)
            - Events related to the objects where replicas don't match the expected value
        """
        # If errors found on some step - set this flag to True to skip further checks on next steps
        failure_flag = False

        # The list object to pass into checkers and collect ordered messages with important check results
        messages = []
        messages.append(f"Inspecting NAMESPACE='{cluster_namespace}'  CLUSTER='{cluster_name}'")

        # 1. Check the Management cluster (or bootstrap cluster in KinD) presense
        if kaas_manager is None:
            messages.append("kaas_manager is empty, NOTHING TO CHECK")
            failure_flag = True

        # 2. Check the Cluster objects presense on the Management cluster
        if not failure_flag:
            failure_flag = self._check_user_defined_objects(cluster_name, cluster_namespace, kaas_manager, messages)

        # 3. Check the LCMCluster and LCMMachine objects presense on the region cluster
        if not failure_flag:
            # Check the Region cluster presense
            if region_k8s_client is None:
                messages.append("region_k8s_client is empty, cannot check the provider objects status")
                failure_flag = True
            else:
                failure_flag = self._check_lcm_related_objects(cluster_name,
                                                               cluster_namespace,
                                                               kaas_manager,
                                                               region_k8s_client,
                                                               messages)

        # 4. Check the Machines provisioning status
        if not failure_flag:
            provision_failure_flag = self._check_provisioning_machines(cluster_name,
                                                                       cluster_namespace,
                                                                       kaas_manager,
                                                                       region_k8s_client,
                                                                       messages)
        # 5. Check the LCMMachines deploying status which may have some LCM tasks even for not provisioned Machines
        if not failure_flag:
            lcm_failure_flag = self._check_lcm_machines(cluster_name,
                                                        cluster_namespace,
                                                        kaas_manager,
                                                        region_k8s_client,
                                                        messages)
            failure_flag = provision_failure_flag or lcm_failure_flag

        # 6. Check the Helmbundles and Cluster deploy conditions
        if not failure_flag:
            failure_flag = self._check_deploy_conditions(cluster_name,
                                                         cluster_namespace,
                                                         kaas_manager,
                                                         region_k8s_client,
                                                         messages)
        # 7. Check cluster stages
        if not failure_flag:
            messages.append(f"Inspecting cluster '{cluster_name} stages")
            failure_flag = self._check_deploy_stages(cluster_name,
                                                     cluster_namespace,
                                                     kaas_manager,
                                                     messages)

        # Finnally, note about success if no steps were failed
        if not failure_flag:
            messages.append(f"Cluster '{cluster_name}' objects: READY")

        # Print the 'messages' content and finish the introspection
        if messages:
            output = '\n'
            for message in messages:
                output += f"{self.PREF} {message}\n"
            LOG.warning(output)
        else:
            LOG.warning(f"{self.PREF} No deviations found from the expected behaviour. "
                        f"Please improve the introspection code if something actually went wrong.")
        return

    def _check_admission_controller_logs(self, kind, kaas_manager, messages):
        pods = [pod for pod in kaas_manager.api.pods.list_starts_with("admission-controller", namespace='kaas')]
        if not pods:
            messages.append("No 'admission-controller' pods found on the current kaas_manager")
        filtered_logs = []
        for pod in pods:
            pod_logs = pod.get_logs()
            for log_str in pod_logs.splitlines():
                if log_str.startswith("E") and kind in log_str:
                    filtered_logs.append(f"> {pod.name}: {log_str}")

        if filtered_logs:
            # Sort the logs by the time
            filtered_logs.sort(key=lambda x: x[5:])
            # Log only the latest 10 errors from the admission controller logs
            messages.extend(filtered_logs[-10:])
        else:
            messages.append(f"No error found for '{kind}' in the 'admission-controller' logs")

    def _get_bastion_lcmmachine(self, cluster_name, cluster_namespace, region_k8s_client):
        """Return a bastion lcmmachine for the current Cluster if exists, or None"""
        lcmm = region_k8s_client.kaas_lcmmachines.list_all()
        all_lcmmachines = [x for x in lcmm if x.namespace == cluster_namespace]
        for lcmmachine in all_lcmmachines:
            lcmspec = lcmmachine.data.get('spec', {})
            if cluster_name == lcmspec.get('clusterName', '') and 'bastion' == lcmspec.get('type'):
                return lcmmachine
        return None

    def _check_user_defined_objects(self, cluster_name, cluster_namespace, kaas_manager, messages):
        """Ensure that user-defined objects are present in the Management cluster"""

        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = ns.get_cluster(cluster_name)
        # Check if Cluster object exists
        if not cluster.is_existed():
            messages.append(f"Cluster object '{cluster_name}' not found")
            self._check_admission_controller_logs("Cluster", kaas_manager, messages)
            return False

        messages.append(f"Cluster version: '{cluster.clusterrelease_version}'")

        # Check if Machine objects created
        machines_created = True
        machine_types = ["control"]
        if cluster.is_child:
            machine_types.append("worker")
        for machine_type in machine_types:
            machines = cluster.get_machines(machine_type=machine_type, raise_if_empty=False)
            if not machines:
                messages.append(f"Machine objects for '{machine_type}' nodes not found "
                                f"in the Cluster '{cluster_name}'")
                self._check_admission_controller_logs("Machine", kaas_manager, messages)
                machines_created = False
        if not machines_created:
            messages.append("Machines NOT CREATED")
            return True

        # TODO(ddmitriev): Check for user-defined provider-related Credentials and Licences objects
        # TODO(ddmitriev): Check for user-defined BM-related objects:
        #                  bmh, bmhp, ipam subnets and l2templates, kaascephcluster

        # False means that no errors found, all user-defined objects are found in the Management cluster
        # (regardless of their status)
        return False

    def _check_lcm_related_objects(self, cluster_name, cluster_namespace,
                                   kaas_manager, region_k8s_client, messages):
        """Check that exist all LCM objects related to the Cluster and Machine objects"""

        if not region_k8s_client.kaas_lcmclusters.present(cluster_name, cluster_namespace):
            messages.append(f"LCMCluster '{cluster_name}' NOT FOUND")
            return True

        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = ns.get_cluster(cluster_name)
        machines = cluster.get_machines()

        lcmcluster = region_k8s_client.kaas_lcmclusters.get(cluster_name, cluster_namespace).data
        lcmcluster_spec = lcmcluster.get('spec', {})
        lcmcluster_status = lcmcluster.get('status') or {}
        requestedNodes = lcmcluster_status.get('requestedNodes')
        readyNodes = lcmcluster_status.get('readyNodes')
        machineTypes = lcmcluster_spec.get('machineTypes')
        machinetypeitems = 'machineType items: '
        if machineTypes:
            for machinetype, items in machineTypes.items():
                machinetypeitems += f"{machinetype}: {len(items)}  "
        else:
            machinetypeitems += "-"
        messages.append(f"Found LCMCluster '{cluster_name}':"
                        f"  requestedNodes={requestedNodes}  readyNodes={readyNodes}"
                        f"  Machines={len(machines)}  {machinetypeitems}")

        if requestedNodes != len(machines):
            messages.append(f"LCMCluster '{cluster_name}'"
                            f" status.requestedNodes DON'T MATCH the Machines count")
            # TODO(ddmitriev): inspect the lcm-controller logs, why LCMCluster has incorrect requestedNodes in status
            return True

        missing_lcm_machines = []
        for machine in machines:
            if not region_k8s_client.kaas_lcmmachines.present(machine.name, cluster_namespace):
                missing_lcm_machines.append(f"LCMMachine '{machine.name}' NOT FOUND")
                continue
        if missing_lcm_machines:
            messages.extend(missing_lcm_machines)
            return True

        return False

    def _check_provisioning_machines(self, cluster_name, cluster_namespace,
                                     kaas_manager, region_k8s_client, messages):
        """Ensure that provider instances were provisioned successfuly

        Expected:
        - LoadBalancer condition in Cluster object is 'true'
        - (optional) Bastion condition in Cluster object is 'true' and has working LCMMachine for bastion
        - (BM/EM2) BareMetalHosts status.operationalStatus == "OK"
        - Machines status providerInstanceState.ready == True
        - Machines condition for "ProviderInstance" is 'true'
        - LCMMachines spec.stateItems is not None, meaning that lcm-agent was able to report it's status from the nodes
          and lcm controller prepared the lcm tasks to execute
        """
        failure_flag = False
        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = ns.get_cluster(cluster_name)

        cluster_status = cluster.data.get('status') or {}
        cluster_provider_status = cluster_status.get('providerStatus', {})
        cluster_conditions = {condition['type']: condition
                              for condition in cluster_provider_status.get('conditions', [])}
        # Check for Cluster LoadBalancer condition
        if 'LoadBalancer' in cluster_conditions:
            lb_condition = cluster_conditions['LoadBalancer']
            lb_condition_ready = lb_condition.get('ready')
            lb_condition_message = lb_condition.get('message')
            messages.append(f"LoadBalancer condition: ready='{lb_condition_ready}'  "
                            f"message='{lb_condition_message}'")
        else:
            loadbalancerhost = cluster_provider_status.get('loadBalancerHost')
            loadbalancerstatus = cluster_provider_status.get('loadBalancerStatus')
            messages.append(f"LoadBalancer '{loadbalancerhost}' condition NOT FOUND, "
                            f"current status: '{loadbalancerstatus}'")
            failure_flag = True

        # Check for the bastion node:
        bastion_lcmmachine = self._get_bastion_lcmmachine(cluster_name, cluster_namespace, region_k8s_client)
        if 'Bastion' in cluster_conditions or 'bastion' in cluster_provider_status or bastion_lcmmachine:
            # Looks like the Cluster uses a Bastion node
            bastion_condition = cluster_conditions.get('Bastion', {})
            bastion_condition_ready = bastion_condition.get('ready')
            bastion_condition_message = bastion_condition.get('message')

            cluster_bastion_status = cluster_provider_status.get('bastion', {})
            bastion_lcm_managed = cluster_bastion_status.get('lcmManaged', False)
            # Different providers have different key name
            bastion_public_ip = cluster_bastion_status.get('publicIp') or cluster_bastion_status.get('publicIP')

            # Show Bastion cluster condition
            messages.append(f"Bastion condition: ready='{bastion_condition_ready}'  publicIp='{bastion_public_ip}'  "
                            f"lcmManaged='{bastion_lcm_managed}'  message='{bastion_condition_message}'")

            if bastion_lcm_managed:
                if bastion_lcmmachine:
                    bastion_lcmmachine_status = bastion_lcmmachine.data.get('status') or {}
                    bastion_lcm_state = bastion_lcmmachine_status.get('state')
                    bastion_lcm_message = (f"Found LCMMachine {bastion_lcmmachine.name}:  "
                                           f"status.state='{bastion_lcm_state}'")
                else:
                    bastion_lcm_message = ("LCMMachine for bastion NOT FOUND")
                # Show Bastion LCMMachine status
                messages.append(bastion_lcm_message)

            if bastion_public_ip:
                if not waiters.tcp_ping(bastion_public_ip, 22, timeout=60):
                    # Show Bastion availability
                    messages.append(f"Bastion SSH on publicIp={bastion_public_ip} IS NOT ACCESSIBLE")

        machines = cluster.get_machines(raise_if_empty=False)

        # Check for BareMetalHosts provisioning status
        region_name = cluster.region_name
        if kaas_manager.api.kaas_baremetalhosts.available:
            machines_bmh = {bmh.read(cached=True).spec.get('consumerRef', {}).get('name'): bmh.read(cached=True)
                            for bmh in kaas_manager.api.kaas_baremetalhosts.list(namespace=cluster_namespace)
                            if bmh.read(cached=True).metadata.labels.get('kaas.mirantis.com/region', '') == region_name}
        else:
            machines_bmh = {}

        if machines_bmh:
            for machine in machines:
                # Check only BareMetalHosts that are consumed by the machines for the <cluster_name>
                if machine.name in machines_bmh:
                    # Check for BareMetalHost status
                    bmh_status = machines_bmh[machine.name].status or {}
                    error_message = bmh_status.get('errorMessage', '')
                    error_count = bmh_status.get('errorCount')
                    operational_status = bmh_status.get('operationalStatus')
                    powered_on = bmh_status.get('poweredOn')
                    provisioning_state = bmh_status.get('provisioning', {}).get('state')
                    bmh_name = machines_bmh[machine.name].metadata.name
                    if error_message:
                        error_message = f"({error_count} times)error_message='{error_message}'"
                    messages.append(f"Found BareMetalHost '{bmh_name}' for Machine '{machine.name}':"
                                    f"  operationalStatus='{operational_status}'  poweredOn='{powered_on}'"
                                    f"  provisioning_state='{provisioning_state}'"
                                    f"  {error_message}")
                    # See all possible operational statuses for BareMetalHost object in kaas/core repo:
                    # "type OperationalStatus string" in the module
                    # vendor/github.com/metal3-io/baremetal-operator/apis/metal3.io/v1alpha1/baremetalhost_types.go
                    if operational_status != "OK":
                        failure_flag = True
                else:
                    messages.append(f"BareMetalHost for Machine '{machine.name}' NOT FOUND")
                    failure_flag = True

        # Check for Machines provisioning status
        not_provisioned_machines = []
        for machine in machines:
            machine_data = machine.data
            machine_data_status = machine_data.get('status') or {}
            provider_status = machine_data_status.get('providerStatus')
            if provider_status is not None:
                machine_status = provider_status.get('status')
                provider_instance_ready = provider_status.get('providerInstanceState', {}).get('ready')
                provider_instance_state = provider_status.get('providerInstanceState', {}).get('state')
                machine_conditions = []
                for condition in provider_status.get('conditions', []):
                    condition_type = condition.get('type')
                    if utils.is_reboot_condition(condition_type):
                        LOG.info(f"Skipping status.providerStatus.conditions.{condition_type} "
                                 f"for machine '{machine.name}': "
                                 f"ready={condition.get('ready')}, message={condition.get('message')}")
                        continue
                    machine_conditions.append(condition)

                ready_conditions = sum([condition.get('ready', 0) for condition in machine_conditions])
                messages.append(f"Found Machine '{machine.name}':  status={machine_status}"
                                f"  providerInstanceState: ready={provider_instance_ready}"
                                f",state='{provider_instance_state}'"
                                f"  ready_conditions={ready_conditions}/{len(machine_conditions)}")
                if cluster.provider == utils.Provider.byo:
                    # BYO provider should not have any providerInstanceState, ignore any status from it
                    pass
                elif not provider_instance_ready:
                    not_provisioned_machines.append(machine)
            else:
                messages.append(f"Machine '{machine.name}' providerStatus NOT FOUND")
                not_provisioned_machines.append(machine)

        if not_provisioned_machines:
            # Show the Events for not_provisioned_machines
            for machine in not_provisioned_machines:
                events_msg = self._get_object_events(machine.name, cluster_namespace, "Machine", region_k8s_client)
                messages.append(events_msg)

            # TODO(ddmitriev): check for provider controller logs related to not_provisioned_machines.
            #                  For BM/EM2 - also check for provision services logs
            messages.append("Some Machines were NOT PROVISIONED")
            failure_flag = True

        # Check for LCMMachines lcm-agent availability from the Machines after provisioning
        not_updated_lcmmachines = []
        for machine in machines:
            lcmmachine_data = region_k8s_client.kaas_lcmmachines.get(machine.name, cluster_namespace).data
            lcmmachine_status = lcmmachine_data.get('status')
            if lcmmachine_status is None:
                messages.append(f"LCMMachine '{machine.name}' 'status' IS EMPTY")
                not_updated_lcmmachines.append(machine)
                continue
            lcmmachine_state = lcmmachine_status.get('state')
            # From kaas/core:
            # // The agent didn't set the address yet.
            # LCMMachineStateUninitialized LCMMachineState = "Uninitialized"
            if not lcmmachine_state or lcmmachine_state == "Uninitialized":
                messages.append(f"LCMMachine '{machine.name}'"
                                f" lcm-agent didn't set the address yet.  state={lcmmachine_state}")
                not_updated_lcmmachines.append(machine)
                continue
        if not_updated_lcmmachines:
            # TODO(ddmitriev): check for the network connetivity to the provider instance IP addresses
            #                  If network connectivity exists - check the lcm-agent service on the node,
            #                  certificate, logs, etc
            messages.append("Some Machines agents NOT REPORTED")
            failure_flag = True

        return failure_flag

    def _check_lcm_machines(self, cluster_name, cluster_namespace,
                            kaas_manager, region_k8s_client, messages):
        """Ensure that provider instances were provisioned successfuly

        Expected:
        - LCMMachines status.addresses list is non-empty (?)
        - LCMMachines len(spec.stateItems) equals len(status.stateItemStatuses)
        - LCMMachines status.stateItemStatuses don't contain errors
        - LCMMachines status.state is 'Ready'
        - Machines status.providerStatus.status is 'Ready'
        - LCMCluster requestedNodes == readyNodes
        """

        # // The agent did set the IP address, but there aren't any
        # // state items set for this machine.
        # LCMMachineStatePending LCMMachineState = "Pending"

        # // The agent did return an IP address, and the state items
        # // are only set for Prepare phase.
        # LCMMachineStatePrepare LCMMachineState = "Prepare"

        # // The agent did return an IP address, and the state items are
        # // set for the phases up to and including the Deploy phase
        # // (also possibly for Reconfigure phase, but this isn't
        # // checked), but Prepare and/or Deploy phases didn't finish
        # // yet.
        # LCMMachineStateDeploy LCMMachineState = "Deploy"

        # // The agent did return an IP address, all of the state items
        # // for the machine are set and fully processed by the agent, but
        # // the desired UCP version doesn't match the actual one.
        # LCMMachineStateUpgrade LCMMachineState = "Upgrade"

        # // The agent did return an IP address, and the state items are
        # // set for the phases up to and including the Reconfigure phase,
        # // and the Prepare and Deploy phases are fully processed,
        # // but Reconfigure phase isn't processed yet.
        # LCMMachineStateReconfigure LCMMachineState = "Reconfigure"

        # // The agent did return an IP address, all of the state items
        # // for the machine are set and fully processed by the agent.
        # LCMMachineStateReady LCMMachineState = "Ready"

        # // The agent is going to proceed with a node reboot
        # LCMMachineStateReboot LCMMachineState = "Reboot"

        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = ns.get_cluster(cluster_name)
        machines = cluster.get_machines(raise_if_empty=False)

        not_ready_machines = []
        not_ready_lcmmachines = []
        failed_lcm_tasks = []
        for machine in machines:
            lcmmachine = region_k8s_client.kaas_lcmmachines.get(machine.name, cluster_namespace).data
            lcmmachine_spec = lcmmachine.get('spec', {})
            lcmmachine_status = lcmmachine.get('status') or {}
            lcmmachine_state = lcmmachine_status.get('state')

            if cluster.provider == utils.Provider.byo:
                # BYO provider don't have 'stateItems' in LCMMachines, nothing to check
                messages.append(f"Found LCMMachine '{machine.name}':  status.state={lcmmachine_state}")
            else:
                # Check spec.stateItems and status.stateItemStatuses
                stateItems = lcmmachine_spec.get('stateItems')
                if stateItems is None:
                    messages.append(f"LCMMachine '{machine.name}' spec.stateItems is None."
                                    f"  state={lcmmachine_state}")
                    not_ready_lcmmachines.append(machine)
                    # TODO(ddmitriev): check for lcm-controller logs for the machine
                    continue
                num_stateItems = len(stateItems)

                stateItemStatuses = lcmmachine_status.get('stateItemStatuses')
                if stateItemStatuses is None:
                    messages.append(f"LCMMachine '{machine.name}' status.stateItemStatuses is None."
                                    f"  state={lcmmachine_state}  spec.stateItems={num_stateItems}")
                    not_ready_lcmmachines.append(machine)
                    # TODO(ddmitriev): check for lcm-agent logs on the machine, was it able to report the status?
                    continue
                num_stateItemStatuses = len(stateItemStatuses.keys())

                messages.append(f"Found LCMMachine '{machine.name}':  status.state={lcmmachine_state}  "
                                f"spec.stateItems={num_stateItems}  status.stateItemStatuses={num_stateItemStatuses}")

                failed_tasks = []
                for name, data in stateItemStatuses.items():
                    exitCode = data.get('exitCode')
                    attempt = data.get('attempt')
                    startedAt = data.get('startedAt')
                    finishedAt = data.get('finishedAt')
                    message = data.get('message', '')
                    if exitCode != 0:
                        failed_tasks.append(f"LCMMachine '{machine.name}' task '{name}' ERROR:"
                                            f"  exitCode={exitCode}  attempt={attempt}"
                                            f"  from:{startedAt}  to:{finishedAt}")
                        filtered_message_lines = utils.filter_ansible_log(message)
                        for fline in filtered_message_lines:
                            failed_tasks.append(f"  >> {fline}")

                if failed_tasks:
                    not_ready_lcmmachines.append(machine)
                    failed_lcm_tasks.extend(failed_tasks)
                    # TODO(ddmitriev): check for lcm-agent ansible tasks log on the machine
                    continue

                if num_stateItems != num_stateItemStatuses:
                    not_ready_lcmmachines.append(machine)
                    # TODO(ddmitriev): check for lcm-agent logs on the machine, was it able to report the status?
                    continue

            if lcmmachine_state != "Ready":
                not_ready_lcmmachines.append(machine)
                # TODO(ddmitriev): check lcm-controller logs about the LCMMachine phases
                continue

            machine_data = machine.data
            machine_data_status = machine_data.get('status') or {}
            provider_status = machine_data_status.get('providerStatus') or {}
            machine_status = provider_status.get('status')
            if machine_status is None or machine_status != "Ready":
                messages.append(f"Machine '{machine.name}' has UNEXPECTED"
                                f" status.providerStatus.status='{machine_status}'")
                not_ready_machines.append(machine)
                # TODO(ddmitriev): check provider controller logs about the Machine providerStatus phases
                continue

        if failed_lcm_tasks:
            # Add group of messages related to the failed tasks, at the end of the messages list
            messages.extend(failed_lcm_tasks)

        if not_ready_lcmmachines:
            messages.append("LCMMachines NOT READY")
            # TODO(ddmitriev): Inspect events of PODs that are in non-Running state, in the following priority:
            # 'Terminating': Something is stuck in the cluster
            # 'Init': Some containers cannot start because of external conditions
            # 'Pending': Some containers are waiting for pre-conditions (NodeAffinity, image downloading, ...)
            # 'CrashLoopBack': Something wrong inside the container
            return True

        if not_ready_machines:
            messages.append("Machines NOT READY")
            # TODO(ddmitriev): Inspect Machines events from the Management cluster
            return True

        lcmcluster = region_k8s_client.kaas_lcmclusters.get(cluster_name, cluster_namespace).data
        lcmcluster_status = lcmcluster.get('status') or {}
        requestedNodes = lcmcluster_status.get('requestedNodes')
        readyNodes = lcmcluster_status.get('readyNodes')
        mke_components_state = lcmcluster_status.get('mke', {}).get('mkeComponentsState', {})  # <<< missing in 2.23.5
        mke_components_ready = mke_components_state.get('ready', False)

        if mke_components_state:
            # LCMCluster status.mkeComponentsState is added in 2.24
            messages.append(
                f"MKE component state message: '{mke_components_state.get('message', '')}'   "
                f"ready: '{mke_components_ready}'   "
                f"notReadyComponents: '{mke_components_state.get('notReadyComponents', '')}'")
            lcm_machines_flag = not mke_components_ready
        else:
            # Backward compatibility for releases <2.24
            messages.append("MKE component state is missing in the LCMCluster, skipping check")
            lcm_machines_flag = False

        if readyNodes != requestedNodes:
            messages.append(f"LCMCluster '{cluster_name}' has NOT READY nodes:"
                            f"  requestedNodes={requestedNodes}  readyNodes={readyNodes}")
            # TODO(ddmitriev): check lcm-cluster logs for LCMCluster update errors,
            # because all nodes should be ready at this moment
            # Also, inspect the lcm-cluster events from the regional cluster
            # Also, check for such events from LcmClusterState:
            # 3m1s   Warning   Reconcile Error lcmclusterstate/cd-cluster-rqyf-mke-7-9-0-3-4-9-mke-node9vvcs
            #                  ceph csi-driver is not evacuated yet, waiting...
            lcm_machines_flag = True

        return lcm_machines_flag

    def _check_helmbundle(self, helmbundle_name, helmbundle_namespace, region_k8s_client, messages):
        if not region_k8s_client.kaas_helmbundles.present(helmbundle_name, helmbundle_namespace):
            messages.append(f"HelmBundle '{helmbundle_namespace}/{helmbundle_name}' NOT FOUND")
            return True

        helmbundle = region_k8s_client.kaas_helmbundles.get(helmbundle_name, helmbundle_namespace).data
        helmbundle_status = helmbundle.get('status') or {}
        helmbundle_charts_status = helmbundle_status.get('releaseStatuses', {})
        helmbundle_charts_num = len(helmbundle_charts_status.keys())

        failed_charts = []
        helmbundle_charts_ready = 0
        helmbundle_charts_deployed = 0
        for name, data in helmbundle_charts_status.items():
            if data.get('ready') is True:
                helmbundle_charts_ready += 1
            status = str(data.get('status'))

            # Values for 'status' field from lcm-controller source code:

            # // StatusUnknown indicates that a release is in an uncertain state.
            # StatusUnknown Status = "unknown"
            # // StatusDeployed indicates that the release has been pushed to Kubernetes.
            # StatusDeployed Status = "deployed"
            # // StatusUninstalled indicates that a release has been uninstalled from Kubernetes.
            # StatusUninstalled Status = "uninstalled"
            # // StatusSuperseded indicates that this release object is outdated and a newer one exists.
            # StatusSuperseded Status = "superseded"
            # // StatusFailed indicates that the release was not successfully deployed.
            # StatusFailed Status = "failed"
            # // StatusUninstalling indicates that a uninstall operation is underway.
            # StatusUninstalling Status = "uninstalling"
            # // StatusPendingInstall indicates that an install operation is underway.
            # StatusPendingInstall Status = "pending-install"
            # // StatusPendingUpgrade indicates that an upgrade operation is underway.
            # StatusPendingUpgrade Status = "pending-upgrade"
            # // StatusPendingRollback indicates that an rollback operation is underway.
            # StatusPendingRollback Status = "pending-rollback"
            if status == 'deployed':
                helmbundle_charts_deployed += 1
            else:
                failed_charts.append(f"Chart '{name}' status='{status}', message:")
                message = str(data.get('message')) + "\n"
                # Use yaml.dump() to format the message into multiline object
                formatted_message = [f"  >> {line}" for line in yaml.dump(message).splitlines()]
                failed_charts.extend(formatted_message)

        messages.append(f"Found HelmBundle '{helmbundle_namespace}/{helmbundle_name}':"
                        f"  ready charts={helmbundle_charts_ready}/{helmbundle_charts_num}"
                        f"  deployed charts={helmbundle_charts_deployed}/{helmbundle_charts_num}")
        if failed_charts:
            messages.extend(failed_charts)
            messages.append(f"Helmbundle '{helmbundle_namespace}/{helmbundle_name}' charts NOT READY")
            return True
        # No errors found in the HelmBundle
        return False

    def _check_cluster_conditions(self, cluster_name, cluster_namespace, kaas_manager, messages):
        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster_data = ns.get_cluster(cluster_name).data
        cluster_status = cluster_data.get('status') or {}
        conditions = cluster_status.get('providerStatus', {}).get('conditions', [])
        failed_conditions = []
        for condition in conditions:
            ready = condition.get('ready')
            condition_type = condition.get('type')
            message = condition.get('message')
            if not ready:
                if utils.is_reboot_condition(condition_type):
                    LOG.info(f"Skipping status.providerStatus.conditions.{condition_type} "
                             f"for cluster '{cluster_name}': "
                             f"ready={ready}, message={message}")
                    continue
                failed_conditions.append(f"Cluster '{cluster_name}' "
                                         f"status.providerStatus.conditions.{condition_type} ERROR: {message}")
        if failed_conditions:
            messages.extend(failed_conditions)
            messages.append("Cluster conditions NOT READY")
            return True
        # No errors found in the Cluster status.conditions
        return False

    def _check_deploy_conditions(self, cluster_name, cluster_namespace,
                                 kaas_manager, region_k8s_client, messages):
        """Check for Helmbundles status and Cluster conditions"""
        # Check Cluster HelmBundle
        # result: boolean flag which reflects the errors
        failure_flag = self._check_helmbundle(cluster_name, cluster_namespace, region_k8s_client, messages)
        # Check Stacklight HelmBundle if enabled
        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = ns.get_cluster(cluster_name)
        if cluster.is_sl_enabled():
            failure_flag = self._check_helmbundle(
                'stacklight-bundle', 'stacklight', region_k8s_client, messages) or failure_flag

        # Check Cluster status.conditions
        failure_flag = self._check_cluster_conditions(
            cluster_name, cluster_namespace, kaas_manager, messages) or failure_flag

        return failure_flag

    def _check_clusterdeploy_stages(self, cluster_name, cluster_namespace, kaas_manager, messages):
        if not settings.FEATURE_FLAGS.enabled("upgrade-history"):
            LOG.info("Skip upgrade_history crds check")
            return False
        failed_stages = []
        ns = kaas_manager.get_namespace(cluster_namespace)
        cluster = ns.get_cluster(cluster_name)

        depl_stages = cluster.get_cluster_deploy_stages_filtered_by_cluster_name(
            cluster_name=cluster.name)
        if depl_stages:
            for stage in depl_stages:
                stage_name = stage.get('name')
                timestamp = stage.get('timestamp')
                message = stage.get('message')
                success = stage.get('success')
                if not success:
                    failed_stages.append(f"Cluster '{cluster_name}'"
                                         f"Failed stage: '{stage_name}'"
                                         f"Message '{message}'"
                                         f"Time '{timestamp}'")
        if failed_stages:
            messages.extend(failed_stages)
            messages.append("Cluster Has failed stages in crd DeployStatus")
            return True
        # No errors found in the crd clusterdeploystatus
        return False

    def _check_deploy_stages(self, cluster_name, cluster_namespace,  kaas_manager, messages):
        # Check Cluster crd clusterdeploystatus
        return self._check_clusterdeploy_stages(cluster_name,
                                                cluster_namespace, kaas_manager, messages)
