import pytest

import kubernetes
import os
import base64
import exec_helpers
from ironicclient import client as ironicclient
from kubernetes.client.rest import ApiException
from si_tests import settings
from si_tests.utils import templates
from si_tests.clients.k8s import K8sCluster
from si_tests.managers.kaas_manager import Manager
from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.proxy_manager import ProxyManager
from si_tests.managers.tungstenfafric_manager import TFManager
from si_tests import logger
from si_tests.utils import utils
import si_tests.utils.templates as templates_utils
import si_tests.utils.waiters as helpers
import json
import yaml

LOG = logger.logger


class ComponentTestResources(object):
    """Manage resources for component tests"""

    def __init__(self, kaas_manager, pod_name, namespace,
                 target_namespace, target_cluster):

        self.kaas_manager = kaas_manager
        self.pod_name = pod_name
        self.ns = namespace
        self.target_namespace = target_namespace
        self.target_cluster = target_cluster
        self.conformance_version = settings.K8S_CONFORMANCE_IMAGE_VERSION
        self.secret_name = '-'.join(('kubeconfig', target_cluster))
        self.run_on_target_cluster = settings.\
            COMPONENT_TEST_RUN_ON_TARGET_CLUSTER

    @property
    def cluster(self):
        ns = self.kaas_manager.get_namespace(
            namespace=self.target_namespace
        )
        return ns.get_cluster(self.target_cluster)

    @property
    def parent_cluster(self):
        """Returns the current Cluster or parent Cluster object"""
        if self.run_on_target_cluster:
            return self.cluster
        if not self.cluster.is_management:
            return self.cluster.get_parent_cluster()
        else:
            return self.kaas_manager.get_mgmt_cluster()

    @property
    def parent_api(self):
        """Returns k8sclient of parent cluster where all component
        test resources (including pods) will be created
        """
        # returns target cluster's api instead of parent's in case of True
        if self.run_on_target_cluster:
            return self.cluster.k8sclient
        if not self.cluster.is_management:
            return self.cluster.get_parent_cluster().k8sclient
        else:
            return self.kaas_manager.api

    def setUp(self):
        # create Namespace
        if not self.parent_api.namespaces.present(name=self.ns):
            LOG.info("Creating Namespace '{0}'".format(self.ns))
            ns_body = kubernetes.client.V1Namespace(api_version='v1',
                                                    kind='Namespace',
                                                    metadata={'name': self.ns})
            self.parent_api.api_core.create_namespace(body=ns_body)
        else:
            LOG.info("Namespace '{0}' already exists".format(self.ns))

        # create ServiceAccount
        if not self.parent_api.serviceaccounts.present(name=self.pod_name,
                                                       namespace=self.ns):
            LOG.info("Creating ServiceAccount '{0}'".format(self.pod_name))
            sa_body = kubernetes.client.V1ServiceAccount(
                api_version='v1',
                kind='ServiceAccount',
                metadata={'name': self.pod_name}
            )
            self.SA = self.parent_api.serviceaccounts.create(
                namespace=self.ns,
                body=sa_body
            )
            LOG.debug(self.SA)
        else:
            LOG.info("ServiceAccount '{0}' already exists"
                     .format(self.pod_name))
            self.SA = self.parent_api.serviceaccounts.get(
                name=self.pod_name, namespace=self.ns)

        # create ClusterRole
        api_rbac = self.parent_api.api_rbac_auth
        cr_present = any([
            cr for cr in api_rbac.list_cluster_role().items
            if cr.metadata.name == self.pod_name])
        if not cr_present:
            LOG.info("Creating ClusterRole '{0}'".format(self.pod_name))
            pr = kubernetes.client.V1PolicyRule(api_groups=['*'],
                                                resources=['*'],
                                                verbs=['*'])
            # skipping api_version='rbac.authorization.k8s.io / v1beta1'
            cr_body = kubernetes.client.V1ClusterRole(
                kind='ClusterRole',
                metadata={'name': self.pod_name, 'namespace': self.ns},
                rules=[pr]
            )
            CR = api_rbac.create_cluster_role(body=cr_body)
            LOG.debug(CR)
        else:
            LOG.info("ClusterRole '{0}' already exists".format(self.pod_name))

        # create ClusterRoleBinding
        crb_present = any([
            crb for crb in api_rbac.list_cluster_role_binding().items
            if crb.metadata.name == self.pod_name])

        if not crb_present:
            LOG.info("Creating ClusterRoleBinding '{0}'"
                     .format(self.pod_name))
            rr = kubernetes.client.V1RoleRef(
                kind="ClusterRole",
                api_group="rbac.authorization.k8s.io",
                name=self.pod_name)
            sb = kubernetes.client.V1Subject(name=self.pod_name,
                                             kind="ServiceAccount",
                                             namespace=self.ns)
            # skipping api_version='rbac.authorization.k8s.io / v1beta1'
            crb_template = kubernetes.client.V1ClusterRoleBinding(
                metadata={'name': self.pod_name},
                kind='ClusterRoleBinding',
                role_ref=rr,
                subjects=[sb]
            )
            CRB = api_rbac.create_cluster_role_binding(
                body=crb_template
            )
            LOG.debug(CRB)
        else:
            LOG.info("ClusterRoleBinding '{0}' already exists"
                     .format(self.pod_name))

        if self.parent_api.secrets.present(name=self.secret_name,
                                           namespace=self.ns):
            LOG.warning("Secret '{0}' already exists. Deleting.."
                        .format(self.secret_name))
            self.parent_api.api_core.delete_namespaced_secret(
                self.secret_name, self.ns
            )

        # create kubeconfig secret
        LOG.info("Creating Secret '{0}'".format(self.secret_name))
        target_kubeconfig = ''
        LOG.info("TARGET_NAMESPACE is {0}, TARGET_CLUSTER is {1}. "
                 "Using this cluster kubeconfig and running tests "
                 "against it.".format(self.target_namespace,
                                      self.target_cluster))

        name, target_kubeconfig = self.cluster.get_kubeconfig_from_secret()
        self.secret = create_secret(self.parent_api, target_kubeconfig,
                                    self.ns, self.secret_name)
        LOG.debug(self.secret)

    def _get_conformance_version(self, k8s_version_obj):
        try:
            status = self.cluster.data['status']
            if status:
                lcmType = status.get('providerStatus', {}).get(
                    'releaseRefs', {}).get('current', {}).get('lcmType', '')
                if lcmType and lcmType == 'ucp':
                    LOG.info("Cluster under testing has lcmType=ucp")
                    k8s_version = \
                        k8s_version_obj.git_version.split("-docker")[0]
                    k8s_version = k8s_version.split(".")[0] + \
                        "." + k8s_version.split(".")[1]

                    if settings.K8S_CONFORMANCE_IMAGE_VERSION:
                        LOG.info(f"Conformance version manually set to {settings.K8S_CONFORMANCE_IMAGE_VERSION}.")
                        conformance_version = settings.K8S_CONFORMANCE_IMAGE_VERSION
                    else:
                        LOG.info('Trying to detect latest suitable conformance version.')
                        conformance_version = utils.get_latest_k8s_image_version(
                            settings.K8S_CONFORMANCE_IMAGE_VERSION_CHECK_PATH,
                            k8s_version
                        )
        except Exception as e:
            LOG.error(e)
            LOG.error("Cannot dynamically identify conformance image version. ")
            raise e

        return conformance_version

    def get_conformance_version(self):
        target_kubectl_client = self.cluster.k8sclient

        k8s_version_obj = target_kubectl_client.api_version.get_code()
        self.conformance_version = self._get_conformance_version(k8s_version_obj)
        LOG.info(f"K8s conformance image version for this ucp cluster is: {self.conformance_version}")
        LOG.warning("For ucp child we need to set special environment variables")
        LOG.info("*Set K8S_CONFORMANCE_RUN_STORAGE_TESTS to False")
        os.environ["K8S_CONFORMANCE_RUN_STORAGE_TESTS"] = "False"
        LOG.info("*Set K8S_CONFORMANCE_NON_BLOCKING_TAINTS to "
                 "com.docker.ucp.manager,"
                 "com.docker.ucp.orchestrator.kubernetes")
        os.environ["K8S_CONFORMANCE_NON_BLOCKING_TAINTS"] = \
            "com.docker.ucp.manager,"\
            "com.docker.ucp.orchestrator.kubernetes"

        exec_helpers.Subprocess().check_call(
            "echo '{0}' > {1}/conformance_image_version".format(
                self.conformance_version, settings.ARTIFACTS_DIR)
        )

        k8s_version = "{0}.{1}".format(
            k8s_version_obj.major,
            k8s_version_obj.minor).replace("+", ".x")

        LOG.info("Target cluster K8s version: {}".format(k8s_version))

        exec_helpers.Subprocess().check_call(
            "echo '{0}' > {1}/k8s_version".format(
                k8s_version, settings.ARTIFACTS_DIR)
        )

    def tearDown(self):
        LOG.info("teardown prerequisites")
        self.SA.delete()
        self.parent_api.api_rbac_auth.delete_cluster_role(
            name=self.pod_name, body=kubernetes.client.V1DeleteOptions()
        )
        self.parent_api.api_rbac_auth.delete_cluster_role_binding(
            name=self.pod_name, body=kubernetes.client.V1DeleteOptions()
        )
        self.secret.delete()
        self.parent_api.api_core.delete_namespace(
            name=self.ns, body=kubernetes.client.V1DeleteOptions()
        )


def create_secret(kubectl_client, content, namespace, name):
    config = ''
    config = base64.b64encode(content.encode('utf-8'))
    secret_body = kubernetes.client.V1Secret(
        api_version='v1',
        kind='Secret',
        metadata={'name': name, 'namespace': namespace},
        data={name: config.decode('utf-8')}
    )
    secret = kubectl_client.secrets.create(namespace=namespace,
                                           body=secret_body)
    LOG.debug(secret)
    return secret


@pytest.fixture(scope='session')
def squid_ip(kaas_manager):
    squid_svc = kaas_manager.api.api_core.read_namespaced_service(
        name='squid-proxy-squid-cache',
        namespace='kaas').to_dict()
    return squid_svc['status']['load_balancer']['ingress'][0]['ip']


@pytest.fixture(scope='session')
def mgmt_k8s_ip(kaas_manager):
    mgmt_cluster = kaas_manager.get_mgmt_cluster().data
    return mgmt_cluster['status']['providerStatus']['loadBalancerHost']


@pytest.fixture(scope='session')
def kaas_manager():
    if settings.KUBECONFIG_MGMT_PATH:
        kubeconfig_path = settings.KUBECONFIG_MGMT_PATH
    else:
        kubeconfig_path = settings.KUBECONFIG_PATH
    kaas_manager = Manager(kubeconfig=kubeconfig_path)
    if not kaas_manager.api.kaas_kaasreleases.available:
        LOG.error(f"Kubeconfig {kubeconfig_path} is not a Management cluster config")
        raise Exception("Incorrect kubeconfig settings")
    return kaas_manager


@pytest.fixture(scope='session')
def ironic_client(kaas_manager):
    """
    Fixture for ironic_client initialization
    Ironic client params:
        token - fake auth
        endpoint - ironic endpoint
        insecure - allow to reqest to https
    @param kaas_manager:
    @return Client
    """
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    ironic_svc = mgmt_cluster.k8sclient.services.get(
        namespace='kaas',
        name='ironic-kaas-bm')
    if not ironic_svc:
        raise LookupError("Ironic service not found")
    ironic_ip = ironic_svc.get_external_addr()
    kwargs = {'token': 'fake-token',
              'endpoint': f'http://{ironic_ip}:6385/',
              'insecure': 'true'}
    return ironicclient.get_client(1, **kwargs)


@pytest.fixture(scope='module',
                params=settings.TARGET_CLUSTERS)
def target_clusters_logs(request):
    yield request.param


@pytest.fixture(scope='session')
def os_manager():
    return OpenStackManager(kubeconfig=settings.KUBECONFIG_PATH)


@pytest.fixture(scope='session')
def tf_manager():
    return TFManager(kubeconfig=settings.KUBECONFIG_PATH)


@pytest.fixture(scope='class')
def restore_tfoperator_cr(tf_manager):
    LOG.info("Backup TFOperator CR")
    tfoperator = tf_manager.tfoperator()
    data = tfoperator.data
    yield
    LOG.info("Restore TFOperator CR")
    res_version = tf_manager.tfoperator().data['metadata']['resource_version']
    name = data['metadata']['name']
    ns = data['metadata']['namespace']
    del data['status']
    del data['metadata']
    data['metadata'] = {'name': name, 'namespace': ns}
    data['metadata']['resourceVersion'] = str(res_version)

    tfoperator.replace(data)
    tf_manager.update_tfvrouter_pods()
    tf_manager.wait_tf_controllers_healthy(timeout=240)
    tf_manager.wait_tfoperator_healthy(timeout=180)
    tf_manager.wait_tf_modules_updated()


@pytest.fixture(scope='class')
def restore_osdpl_cr(os_manager):
    LOG.info("Backup Openstack controller CR")
    osdpl = os_manager.get_osdpl_deployment()
    data = osdpl.data
    yield
    LOG.info("Restore Openstack controller CR")
    res_version = os_manager.get_osdpl_deployment().data['metadata']['resource_version']
    name = data['metadata']['name']
    ns = data['metadata']['namespace']
    del data['status']
    del data['metadata']
    data['metadata'] = {'name': name, 'namespace': ns}
    data['metadata']['resourceVersion'] = str(res_version)

    osdpl.replace(data)
    os_manager.wait_os_deployment_status(timeout=600, status="APPLIED")
    os_manager.wait_openstackdeployment_health_status(timeout=210)


@pytest.fixture(scope='function')
def component_test_resources():
    """Returns the class ComponentTestResources"""
    return ComponentTestResources


@pytest.fixture(scope="module")
def proxy_manager(kaas_manager):
    """Returns the internal proxy control class"""
    return ProxyManager(kaas_manager.api)


@pytest.fixture(scope="module")
def k8s_prerequisites(kaas_manager):
    resources = ComponentTestResources(
        kaas_manager,
        settings.K8S_CONFORMANCE_POD_NAME,
        settings.K8S_CONFORMANCE_NAMESPACE,
        settings.TARGET_NAMESPACE,
        settings.TARGET_CLUSTER,
    )
    resources.setUp()
    resources.get_conformance_version()
    yield resources
    resources.tearDown()


@pytest.fixture(scope="module")
def stacklight_prerequisites(kaas_manager):
    resources = ComponentTestResources(
        kaas_manager,
        settings.STACKLIGHT_TEST_POD_NAME,
        settings.STACKLIGHT_TEST_NAMESPACE,
        settings.TARGET_NAMESPACE,
        settings.TARGET_CLUSTER,
    )
    resources.setUp()
    yield resources
    resources.tearDown()


@pytest.fixture(scope="module")
def hpa_prerequisites(kaas_manager, target_kubectl, target_cluster):
    if len(target_kubectl.pods.list_starts_with(
           'metrics-server', 'kube-system')) == 0:
        msg = "metrics-server is not found. Skipping"
        LOG.warning(msg)
        pytest.skip(msg)

    deployment_name = "si-tests-hpa-{}".format(settings.TARGET_CLUSTER)
    base_image_repo = target_cluster.determine_mcp_docker_registry()

    options = {
        'HPA_TEST_DEP_NAME': deployment_name,
        'HPA_BASE_IMAGE_REPO': base_image_repo,
    }
    templates = templates_utils.render_template(
        settings.HPA_TEST_POD_YAML, options)
    LOG.debug(templates)
    json_body = json.dumps(yaml.load(templates, Loader=yaml.SafeLoader))
    dep = target_kubectl.deployments.create(name=deployment_name,
                                            namespace='default',
                                            body=json.loads(json_body))

    service = kubernetes.client.V1Service()
    service.metadata = kubernetes.client.V1ObjectMeta(name="si-tests-hpa-svc")
    service.spec = kubernetes.client.V1ServiceSpec(type="LoadBalancer")
    service.spec.selector = {"run": "hpa"}
    service.spec.ports = [kubernetes.client.V1ServicePort(port=8080,
                                                          target_port=8080)]
    svc = target_kubectl.services.create(name="si-tests-hpa-svc",
                                         namespace='default',
                                         body=service)

    hpa_body = kubernetes.client.V1HorizontalPodAutoscaler()
    hpa_body.metadata = kubernetes.client.V1ObjectMeta(name="si-tests-hpa")
    scale_target_ref = kubernetes.client.V1CrossVersionObjectReference(
        kind="Deployment", name=deployment_name, api_version="apps/v1")
    hpa_body.spec = kubernetes.client.V1HorizontalPodAutoscalerSpec(
        max_replicas=2, min_replicas=1, target_cpu_utilization_percentage=50,
        scale_target_ref=scale_target_ref)
    hpa = target_kubectl.horizontalpodautoscalers.create(
        namespace='default', name="si-tests-hpa", body=hpa_body)

    LOG.info("Waiting for pods (must be only 1)")
    helpers.wait(lambda: len(target_kubectl.pods.list_starts_with(
        deployment_name, 'default')) == 1,
        timeout=300, interval=15,
        timeout_msg='Timeout waiting for number of pods to be 1. '
        'Actual: {}'.format(
        len(target_kubectl.pods.list_starts_with(
            deployment_name, 'default'))))

    LOG.info("Waiting for LB to have external ip/hostname")
    helpers.wait(lambda: svc.read().to_dict()['status'][
        'load_balancer']['ingress'] is not None,
        timeout=500, interval=15,
        timeout_msg='Timeout waiting for svc loadbalancer')

    address = ''
    if svc.read().to_dict()['status']['load_balancer']['ingress'][0]['ip']:
        address = svc.read().to_dict()['status'][
            'load_balancer']['ingress'][0]['ip']
    else:
        address = svc.read().to_dict()['status'][
            'load_balancer']['ingress'][0]['hostname']
    LOG.info("LB iaddress is {}".format(address))

    yield {"lb_address": address}

    LOG.info("Removing hpa resources")
    svc.delete()
    hpa.delete()
    dep.delete()


@pytest.fixture(scope="module")
def target_kubectl(kaas_manager):
    try:
        kaas_manager.get_kaasreleases()
    except ApiException:
        LOG.info("Cannot use kaas_manager for this cluster. "
                 "target_kubectl will be from KUBECONFIG_PATH")
        yield K8sCluster(kubeconfig=settings.KUBECONFIG_PATH)
    else:
        namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
        cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
        LOG.info("KaaS cluster detected")
        yield cluster.k8sclient


@pytest.fixture(scope="module")
def target_cluster(kaas_manager: Manager):
    namespace = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
    yield cluster


@pytest.fixture(scope="module")
def kaas_ui_prerequisites(kaas_manager):
    secret_cloud_yaml = templates.render_template(settings.CLOUDS_YAML_PATH)
    secret_cloud_yaml = create_secret(kaas_manager.api, secret_cloud_yaml,
                                      settings.KAAS_UI_TEST_NAMESPACE,
                                      'clouds.yaml')
    key = utils.generate_keys()['public']
    secret_key_pub = create_secret(kaas_manager.api, "ssh-rsa " + key,
                                   settings.KAAS_UI_TEST_NAMESPACE,
                                   'key.pub')
    yield kaas_ui_prerequisites
    secret_cloud_yaml.delete()
    secret_key_pub.delete()


@pytest.fixture(scope='function')
def store_cluster_description(request, kaas_manager, _, record_property):
    """Store mgmg + child(optional) cluster configurations after test"""

    def test_fin():
        try:
            # from @pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
            cluster_name = _.split("=")[-1]

            mgmt_cluster = kaas_manager.get_mgmt_cluster()
            kaas_release = mgmt_cluster.get_kaasrelease_version()
            mgmt_description = (f"KaaS Release: '{kaas_release}' "
                                f"ENV_NAME='{settings.ENV_NAME}'\n")

            mgmt_description += mgmt_cluster.get_cluster_description()

            clusters = kaas_manager.get_clusters()

            for cluster in clusters:
                if cluster.name != cluster_name:
                    continue
                if cluster.is_management:
                    description = mgmt_description
                else:
                    description = (mgmt_description +
                                   cluster.get_cluster_description())

                with open(os.path.join(
                        settings.ARTIFACTS_DIR,
                        f"description-{cluster.name}.txt"), "w") as f:
                    f.write(description)

                # Record description to JUnit property
                text = ('**Environment description:**\\n' +
                        '\\n'.join(description.strip().splitlines()) +
                        '\\n')
                record_property("__doc__", (text))

        except Exception:
            pass

    request.addfinalizer(test_fin)


def get_child_cluster_description(kaas_manager, cluster_name):

    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    kaas_release = mgmt_cluster.get_kaasrelease_version()

    description = (f"KaaS Release: '{kaas_release}' "
                   f"ENV_NAME='{settings.ENV_NAME}'\n")
    description += mgmt_cluster.get_cluster_description(system_info=True)

    clusters = kaas_manager.get_clusters()
    for cluster in clusters:
        if cluster.name == cluster_name:
            description += cluster.get_cluster_description(system_info=True)

    return description


def get_mgmt_cluster_description(kaas_manager):

    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    kaas_release = mgmt_cluster.get_kaasrelease_version()

    description = (f"KaaS Release: '{kaas_release}' "
                   f"ENV_NAME='{settings.ENV_NAME}'\n")
    description += mgmt_cluster.get_cluster_description(system_info=True)

    clusters = kaas_manager.get_clusters(namespace=settings.REGION_NAMESPACE)
    for cluster in clusters:
        if cluster.is_regional:
            description += cluster.get_cluster_description(system_info=True)

    return description


@pytest.fixture(scope='function')
def store_updated_child_cluster_description(request, kaas_manager,
                                            _, record_property):
    """Store mgmg + child cluster configurations before/after update"""

    # from @pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
    cluster_name = _.split("=")[-1]

    description = ">>> BEFORE CHILD CLUSTER UPDATE:\n"
    try:
        description += get_child_cluster_description(kaas_manager,
                                                     cluster_name)
        LOG.info(f"\n{description}")
    except Exception as e:
        LOG.error("Error getting the cluster description before update, see debug log")
        LOG.debug(f"Error getting the cluster description before update: {e}")

    yield
    test_skipped = (hasattr(request.node, 'rep_call') and request.node.rep_call.skipped)
    if test_skipped:
        LOG.warning("Test was skipped, so skip updating the cluster description")
        return
    description += "\n>>> AFTER CHILD CLUSTER UPDATE:\n"
    try:
        description += get_child_cluster_description(kaas_manager,
                                                     cluster_name)
        description += "\n"
        LOG.info(f"\n{description}")
    except Exception as e:
        LOG.error("Error getting the cluster description after update, see debug log")
        LOG.debug(f"Error getting the cluster description after update: {e}")

    with open(os.path.join(
            settings.ARTIFACTS_DIR,
            f"description-{cluster_name}.txt"), "a") as f:
        f.write(description)

    # Record description to JUnit property
    text = ('**Environment description:**\\n' +
            '\\n'.join(description.strip().splitlines()) +
            '\\n')
    record_property("__doc__", (text))


@pytest.fixture(scope='function')
def store_updated_mgmt_cluster_description(request, kaas_manager,
                                           record_property):
    """Store mgmg cluster configurations before/after update"""
    mgmt_cluster = kaas_manager.get_mgmt_cluster()

    description = ">>> BEFORE MANAGEMENT CLUSTER UPDATE:\n"
    try:
        description += get_mgmt_cluster_description(kaas_manager)
        LOG.info(f"\n{description}")
    except Exception as e:
        LOG.error("Error getting the cluster description before update, see debug log")
        LOG.debug(f"Error getting the cluster description before update: {e}")

    yield

    description += "\n>>> AFTER MANAGEMENT CLUSTER UPDATE:\n"
    try:
        description += get_mgmt_cluster_description(kaas_manager)
        LOG.info(f"\n{description}")
    except Exception as e:
        LOG.error("Error getting the cluster description after update, see debug log")
        LOG.debug(f"Error getting the cluster description after update: {e}")

    with open(os.path.join(
            settings.ARTIFACTS_DIR,
            f"description-{mgmt_cluster.name}.txt"), "w") as f:
        f.write(description)

    # Record description to JUnit property
    text = ('**Environment description:**\\n' +
            '\\n'.join(description.strip().splitlines()) +
            '\\n')
    record_property("__doc__", (text))


@pytest.fixture(scope='function')
def mariadb_backup_restore_cleanup_helper(kaas_manager):
    """Helper to log backup/restore pod process and cleanup from any test point"""

    LOG.banner('SetUp for MCC MariaDB test started\n')
    # Prefix for backupjob since it created from cronjob
    backup_job_prefix = "mariadb-phy-backup"
    # Name for restorejob since it created as standalone object
    restore_job_name = "mariadb-phy-restore"
    kaas_ns = 'kaas'
    mgmt_cluster = kaas_manager.get_mgmt_cluster()
    iam_cm_name = 'iam-backup-hash'
    iam_keycloak_sts_name = 'iam-keycloak'

    LOG.info('Check and remove (if found) orphaned backup/restore jobs before test start.')
    old_backupjobs = mgmt_cluster.k8sclient.jobs.list(name_prefix=backup_job_prefix, namespace=kaas_ns)
    if old_backupjobs:
        # May be multiple jobs
        for job in old_backupjobs:
            LOG.debug(f"Removing job {job.namespace}/{job.name}")
            job.delete()
    old_restorejob = mgmt_cluster.k8sclient.jobs.list(name_prefix=restore_job_name, namespace=kaas_ns)
    if old_restorejob:
        # Only one restorejob may left
        LOG.debug(f"Removing job {old_restorejob[0].namespace}/{old_restorejob[0].name}")
        old_restorejob[0].delete()

    LOG.banner('SetUp for MCC MariaDB test completed. Good luck!\n')

    yield

    LOG.banner('TearDown for MCC MariaDB test started\n')

    # Dumping logs after any test result, check jobs and remove if exists
    LOG.info('Looking for backup pods that left to gather logs')
    backupjob = mgmt_cluster.k8sclient.jobs.list(name_prefix=backup_job_prefix, namespace=kaas_ns)
    if backupjob:
        bjob_name = backupjob[0].name
        LOG.info(f"Found backupjob with name {backupjob[0].name}. Looking for pods...")
        backup_pods = mgmt_cluster.get_pods_by_job_name(job_name=bjob_name, namespace=kaas_ns)
        if backup_pods:
            bp_logs = backup_pods[0].get_logs()
            # Last 200 line should be enought to understand what happened in case of failure
            formatted = '\n'.join(bp_logs.splitlines()[-200:])
            LOG.info(f">> Log output of {backup_pods[0].namespace}/{backup_pods[0].name}\n")
            LOG.info(f"{formatted}\n")
            LOG.info('>> End of log')
        LOG.info(f"Removing backup job {backupjob[0].namespace}/{backupjob[0].name}")
        backupjob[0].delete()
        if backup_pods:
            # (vastakhov) Evade very rare race condition when pod gonna be removed but due cluster slowness
            # can be listed in pod list and when restore pod list will be generated -
            # can cause fail due will not be exist anymore while we filtering pods by job name.
            # It's caused by direct .read() directive in pod.job_name. In that case we're trying to read deleted object
            LOG.info(f"Waiting for pod {backup_pods[0].namespace}/{backup_pods[0].name} to be removed")
            helpers.wait(lambda: not backup_pods[0].exists(), timeout=150, interval=10,
                         timeout_msg="Backup job pod still exist after 150s of waiting after parent job was removed")
            LOG.info("Backup pod cleared. Continue.")
    LOG.info('\n')
    LOG.info('Looking for restore pods that left to gather pods')
    restorejob = mgmt_cluster.k8sclient.jobs.list(name_prefix=restore_job_name, namespace=kaas_ns)
    if restorejob:
        LOG.info(f"Found restorejob with name {restorejob[0].name}. Looking for pods...")
        restore_pods = mgmt_cluster.get_pods_by_job_name(job_name=restorejob[0].name, namespace=kaas_ns)
        if restore_pods:
            rp_logs = restore_pods[0].get_logs()
            formatted = '\n'.join(rp_logs.splitlines()[-200:])
            LOG.info(f">> Log output of {restore_pods[0].namespace}/{restore_pods[0].name}")
            LOG.info(f"{formatted}\n")
            LOG.info('>> End of log')
        LOG.info(f"Removing restore job {restorejob[0].namespace}/{restorejob[0].name}")
        restorejob[0].delete()

    # Getting image from backup cronjob. Always should be present in cluster.
    backup_cronjob = mgmt_cluster.k8sclient.cronjobs.get(name=backup_job_prefix, namespace=kaas_ns)
    maria_image = backup_cronjob.data['spec']['job_template']['spec']['template']['spec']['containers'][0]['image']
    # List and cleanup base backups
    # TODO(vastakhov): Add removing of whole backup folder in case of adding incr backups into test
    bcups = kaas_manager.keycloak.mariadb_volume_mgmt(namespace=kaas_ns,
                                                      image=maria_image,
                                                      cmd="ls /var/backup/base || echo ''").strip()
    if bcups:
        LOG.info(f"Found backups: {bcups}. Cleaning..")
        kaas_manager.keycloak.mariadb_volume_mgmt(namespace=kaas_ns,
                                                  image=maria_image,
                                                  cmd='rm -rf /var/backup/base/')
        mgmt_cluster.k8sclient.configmaps.get(name=iam_cm_name, namespace=kaas_ns).delete()

    # Rollback backup settings
    changes_made = kaas_manager.keycloak.reset_mcc_mariadb_backup_settings()
    if changes_made:
        LOG.info('Reset completed. Waiting for iam statefulset to be updated')
        sts = mgmt_cluster.k8sclient.statefulsets.get(name=iam_keycloak_sts_name, namespace=kaas_ns)
        sts.wait_for_new_generation()
        LOG.info('Iam statefulset updated. Waiting for pods recreate')
        helpers.wait(lambda: not mgmt_cluster.are_conditions_ready(),
                     timeout=300,
                     interval=10,
                     timeout_msg=f"Cluster {mgmt_cluster.namespace}/{mgmt_cluster.name} does not reach "
                                 f"NOT READY state after MariaDB backup settings were reseted")
        mgmt_cluster.check.check_cluster_readiness()
        LOG.info('Iam updated and in operational state')

    LOG.banner('TearDown completed. Bye!')
