import tarfile
import tempfile
import yaml
import os
import time

from si_tests.managers.openstack_manager import OpenStackManager
from si_tests.managers.openstack_client_manager import OpenStackClientManager
from si_tests.deployments.utils.file_utils import render_yaml_content
from si_tests.deployments.utils import commons
from si_tests.utils import utils

from si_tests import settings


class PerfAnalyzerBase():

    chart_name = None
    endpoints_chart = None
    runner_label_selector = ""
    runner_containers = ["runner", "extractor", "sla"]
    cleanup_network_policies = []

    def __init__(self, scenario_path):
        self.charts_path = os.path.join(settings.PERF_ANALYZER_DIR, 'charts')
        self.scenario_path = scenario_path
        self.os_manager = OpenStackManager()
        self.ocm = OpenStackClientManager(kubeconfig=self.os_manager.kubeconfig)
        self.namespace = self.os_manager.openstack_namespace
        self.init_dependencies()

    def init_dependencies(self):
        executor = commons.get_local_executor()
        helm = self.os_manager.os_helm_manager.binary
        result = executor.execute(f"make HELM={helm}", cwd=settings.PERF_ANALYZER_DIR)
        assert result.exit_code == 0, result

    @property
    def chart_path(self):
        return os.path.join(self.charts_path, self.chart_name)

    @property
    def runner_pods(self):
        pods = self.os_manager.api.pods.list(self.namespace, label_selector=self.runner_label_selector)
        return pods

    def scenario_args(self):
        pass

    def get_scenario_values(self):
        data = self.scenario_args()
        return render_yaml_content(self.scenario_path, data)

    def get_endpoints(self):
        return self.os_manager.os_helm_manager.get_release_values(self.endpoints_chart)["endpoints"]

    def install(self):
        commons.LOG.info("Installing perf-analyzer chart %s", self.chart_name)
        scenario_values = self.get_scenario_values()
        scenario_values["endpoints"] = self.get_endpoints()

        with tempfile.NamedTemporaryFile(mode="w") as values_path:
            yaml.dump(scenario_values, values_path)
            self.os_manager.os_helm_manager.install_chart(
                self.os_manager.kubeconfig, namespace=self.namespace, chart_name=self.chart_name,
                chart_path=self.chart_path, values_path=[values_path.name])
        commons.LOG.info("Installed perf-analyzer chart %s successfully", self.chart_name)

    def remove(self):
        self.os_manager.os_helm_manager.delete_chart(
            self.os_manager.kubeconfig, self.namespace, self.chart_name)

    def prepare(self):
        nps = self.os_manager.api.networkpolicies.list(
            self.namespace)
        for np in nps:
            if np.name in self.cleanup_network_policies:
                commons.LOG.info("Removing networkpolicy %s", np.name)
                np.delete()

    def cleanup(self):
        pass

    def collect(self):
        for pod in self.runner_pods:
            dst = os.path.join(settings.ARTIFACTS_DIR, pod.name)
            try:
                os.mkdir(dst)
            except FileExistsError:
                pass
            pod.cp_from_pod(source_dir="/var/lib/perf-analyzer",
                            destination=dst,
                            compression="-z",
                            container=self.runner_containers[0])
            archive = f"{dst}.tar.gz"
            with tarfile.open(archive, "w:gz") as tar:
                tar.add(dst)
            utils.clean_dir(dst)
            for container in self.runner_containers:
                container_logs = os.path.join(settings.ARTIFACTS_DIR, f"{pod.name}_{container}.log")
                with open(container_logs, 'w') as f:
                    logs = pod.get_logs(container=container)
                    f.write(logs)

    def get_completion_status(self, pod, container):
        pod = pod.read()
        for container_status in pod.status.container_statuses:
            if container_status.name == container:
                if container_status.ready:
                    return "completed"
                if container_status.restart_count > 0:
                    return "failed"
                return "running"

    def is_completed(self, pod):
        completions = []
        for container in self.runner_containers:
            status = self.get_completion_status(pod, container)
            completions.append(status in ["completed", "failed"])
        return completions and all(completions)

    def wait_completed(self, timeout=4200, delay=60):
        commons.LOG.info("Waiting for %s %s is finished", timeout, self.chart_name)
        start = time.time()
        while time.time() - start <= timeout:
            completions = []
            for pod in self.runner_pods:
                completions.append(self.is_completed(pod))
            if completions and all(completions):
                return
            commons.LOG.info("Still waiting for %s completions %s", self.runner_pods, completions)
            time.sleep(delay)
        raise TimeoutError()

    def check_status(self):
        status = {}
        failed = False
        for pod in self.runner_pods:
            status[pod.name] = {}
            for container in self.runner_containers:
                container_status = self.get_completion_status(pod, container)
                if container_status != "completed":
                    failed = True
                status[pod.name][container] = container_status
        if failed:
            raise Exception(f"Some pods failed {status}")


class Dnsperf(PerfAnalyzerBase):

    chart_name = "dnsperf"
    endpoints_chart = "openstack-designate"
    runner_label_selector = "application=dnsperf,component=runner"
    cleanup_network_policies = ["openstack-designate-api-netpol"]

    def scenario_args(self):
        pods = self.os_manager.api.pods.list(
            self.namespace, label_selector="application=designate,component=mdns")
        pod = pods[0].read()
        return {"dns_server": pod.status.pod_ip}

    def prepare(self):
        super().prepare()
        commons.LOG.info("Setting unlimited quotas for designate")
        self.ocm.exec(["openstack", "dns", "quota", "set", "--recordset-records", "-1",
                       "--zone-records", "-1", "--zone-recordsets", "-1", "--zones", "-1"])


class Rabbitmqperf(PerfAnalyzerBase):

    chart_name = "rabbitmqperf"
    endpoints_chart = "openstack-neutron"
    runner_label_selector = "application=rabbitmqperf,component=runner"
    cleanup_network_policies = ["openstack-neutron-rabbitmq-server-netpol"]


class Mariadbperf(PerfAnalyzerBase):

    chart_name = "mariadbperf"
    endpoints_chart = "openstack-keystone"
    runner_label_selector = "application=mariadbperf,component=runner"
    cleanup_network_policies = ["openstack-mariadb-mariadb-server-netpol"]


class Etcdperf(PerfAnalyzerBase):

    chart_name = "etcdperf"
    endpoints_chart = "openstack-cinder"
    runner_label_selector = "application=etcdperf,component=runner"
    cleanup_network_policies = ["etcd-server-netpol"]
