from si_tests.utils import utils
from si_tests.deployments.utils import (extra_context, file_utils, wait_utils,
                                        commons, kubectl_utils)
from si_tests.deployments.utils.namespace import NAMESPACE


def is_baremetal():
    context = file_utils.get_osdpl_content()

    if "baremetal" in context["spec"]["features"].get("services", []):
        return True
    return False


def deploy_vbmc(os_manager, timeout):
    commons.LOG.info("Deploy vmbc")
    auth_keys = ["OS_PASSWORD", "OS_PROJECT_DOMAIN_NAME", "OS_PROJECT_NAME",
                 "OS_REGION_NAME", "OS_USER_DOMAIN_NAME", "OS_USERNAME"]

    commons.LOG.info("Get ssl certs")
    ca_cert, api_cert, api_key = file_utils.get_ssl()

    openstack_spec = file_utils.get_osdpl_content()["spec"]
    tls = {"ca": ca_cert, "crt": api_cert, "key": api_key}
    opts = extra_context.ExtraContext().openstack_opts

    update_values = {
        "endpoints": {
            "cluster_domain_suffix": openstack_spec["internal_domain_name"],
            "identity": {
                "host_fqdn_override": {
                    "public": {
                        "host": openstack_spec["public_domain_name"],
                        "tls": tls}},
                "hosts": {"public": {"tls": tls}}}},
        "conf": {
            "auto_bridge_add": {
                "br-bms": opts.get("ironic_baremetal_network_vbmc_interface")
            }}}

    commons.LOG.info("Get vbmc yaml")
    vbmc_yaml = file_utils.get_release("ci/40-virtualbmc.yaml")
    vbmc_content = file_utils.get_yaml_content(vbmc_yaml)
    vbmc_values = list(filter(
        lambda x: x["name"] == "vbmc-virtualbmc",
        vbmc_content["spec"]["releases"]))[0]["values"]

    ironic_secret = os_manager.api.secrets.get(
        "ironic-keystone-admin", NAMESPACE.openstack)

    def _wait():
        secret_data = ironic_secret.read()
        assert hasattr(secret_data, "data")

    commons.LOG.info("Wait for ironic secret")
    wait = wait_utils.Waiter(os_manager, timeout, namespace=NAMESPACE.vbmc)
    wait.wait_pass(_wait, expected=Exception)

    ironic_data = ironic_secret.read().data

    commons.LOG.info("Update vbmc auth")
    vbmc_admin = vbmc_values["endpoints"]["identity"]["auth"]["admin"]
    for auth_key in auth_keys:
        data = commons.decode(ironic_data[auth_key])
        key = auth_key[3:].lower()
        vbmc_admin[key] = data

    utils.merge_dicts(vbmc_values, update_values)

    new_yaml = file_utils.get_new_name(vbmc_yaml)
    commons.LOG.info("Save new vbmc to %s", new_yaml)
    file_utils.save_to_yaml(vbmc_content, new_yaml)

    kubectl = kubectl_utils.Kubectl()

    commons.LOG.info("Create %s namespace", NAMESPACE.vbmc)
    kubectl.create("namespace", NAMESPACE.vbmc)

    commons.LOG.info("Apply vmbc yaml")
    kubectl.apply(new_yaml)

    vbmc_nodes = len(os_manager.api.nodes.list(
        label_selector="virtualbmc=enabled"))

    daemonsets = ["libvirt-libvirt-default",
                  "openvswitch-db",
                  "openvswitch-vswitchd", "vbmc"]
    for daemonset in daemonsets:
        commons.LOG.info("Wait for %s daemonset", daemonset)
        wait.daemonset(daemonset, replicas=vbmc_nodes)

    commons.LOG.info("Activate activate nodes via doing manage -> provide")
    vbmc_release = list(filter(
        lambda x: x["name"] == "vbmc-virtualbmc",
        vbmc_content["spec"]["releases"]))[0]
    vbmc_release["values"]["manifests"]["job_activate_nodes"] = True

    commons.LOG.info("Save new vbmc to %s", new_yaml)
    file_utils.save_to_yaml(vbmc_content, new_yaml)

    commons.LOG.info("Apply vmbc yaml")
    kubectl.apply(new_yaml)

    def _waiter():
        job = os_manager.api.jobs.get(
            "vbmc-activate-nodes", NAMESPACE.vbmc).read()
        return job.status.succeeded == 1
    wait.wait_pass(_waiter, expected=Exception)

    job_suffix = utils.gen_random_string(3)
    # Force running cell setup unless PRODX-172 is implemented
    commons.LOG.info("Run cell setup")
    kubectl.create(
        "job",
        "--from=cronjob/nova-cell-setup nova-cell-setup-pd01-{}".format(
            job_suffix), namespace=NAMESPACE.openstack)

    commons.LOG.info("vbmc successfully deployed")
