import os
import paramiko
import yaml
import tempfile

from si_tests import settings
from si_tests.deployments.utils import commons, file_utils, wait_utils
from si_tests.deployments.utils.namespace import NAMESPACE
from si_tests.utils import utils
from . import base


class RefappHeat(base.RefappBase):
    def __init__(self, client, kubeconfig=None):
        super().__init__(client, kubeconfig)
        self.stack_name = "refapp-test"
        self._ssh_private_key = None

    @property
    def url(self):
        if self._url is None:
            stack_outputs = self.ostcm.stack.show([self.stack_name, "-c", "outputs"])["outputs"]
            self._url = list(filter(lambda x: x["output_key"] == "app_url", stack_outputs))[0]["output_value"]
        return self._url

    @property
    def _is_active(self):
        """Whether heat stack is active or not"""

        return self.ostcm.stack.exists(self.stack_name)

    @property
    def _stack_status(self):
        """Return stack status"""
        return self.ostcm.stack.show([self.stack_name])["stack_status"]

    @property
    def ssh_private_key(self):
        if self._ssh_private_key is None:
            stack_outputs = self.ostcm.stack.show([self.stack_name, "-c", "outputs"])["outputs"]
            self._ssh_private_key = list(
                filter(lambda x: x["output_key"] == "cluster_private_key", stack_outputs)
            )[0]["output_value"]
        return self._ssh_private_key

    def deploy(self):
        if self._is_active and self._stack_status == "CREATE_FAILED":
            commons.LOG.warning(f"Stack '{self.stack_name}' already exists, but in CREATE_FAILED state. "
                                f"Recreating the stack...")
            self.delete()

        if not self._is_active:
            source = file_utils.join(commons.REFAPP_DIR, "heat-templates")
            if not os.path.exists(source):
                commons.LOG.warning("Refapp source does not exist %s", source)

            client_name = self.ostcm.client.name
            destination = f"{NAMESPACE.openstack}/{client_name}:/tmp/heat-templates"

            commons.LOG.info("Copy heat templates")
            self.kubectl.exec(self.ostcm.client.name, "sh -c 'rm -rf /tmp/heat-templates || true'")
            self.kubectl.cp(source, destination)

            ssh_keys = utils.generate_keys()
            commons.LOG.info("Create heat stack")
            stack_parameters = [
                "--parameter",
                f"cluster_public_key=ssh-rsa {ssh_keys['public']}",
                "--parameter",
                f"cluster_private_key={ssh_keys['private']}",
                "--parameter",
                f"database_admin_password={self.db_passwords['admin']}",
                "--parameter",
                f"app_database_password={self.db_passwords['app']}",
            ]
            if settings.OPENSTACK_REFAPP_IMAGE:
                stack_parameters.extend(["--parameter", f"app_docker_image={settings.OPENSTACK_REFAPP_IMAGE}"])

            if self.os_manager.is_ovn_enabled:
                stack_parameters.extend(["-e", "/tmp/heat-templates/ovn-env.yaml"])

            if self.os_manager.is_tf_enabled:
                tmpl_path = "/tmp/heat-templates/tf/top.yaml"
            else:
                tmpl_path = "/tmp/heat-templates/top.yaml"

            self.ostcm.stack.create([self.stack_name, "-t", tmpl_path, *stack_parameters])

            commons.LOG.info("Waiting for heat stack to create ...")
            wait = wait_utils.Waiter(self.ostcm.os_manager, 3600)

            def f():
                status = self.ostcm.stack.show([self.stack_name])["stack_status"]
                if status == "CREATE_FAILED":
                    resource_list = self.ostcm.stack.resource_list(
                        ['-n', '10', self.stack_name])
                    event_list = self.ostcm.stack.event_list(
                        ['--nested-depth', '10', self.stack_name])
                    commons.LOG.error("Resource info: %s\nHeat stack event list: %s",
                                      resource_list, event_list)
                    raise Exception("Failed to create stack")
                return status == "CREATE_COMPLETE"

            wait.wait(f)

        commons.LOG.info("Refapp is successfully deployed")

    def delete(self):
        commons.LOG.info("Delete RefApp stack")
        self.ostcm.stack.delete([self.stack_name])

        wait = wait_utils.Waiter(self.ostcm.os_manager, 3600)

        def wait_refapp_stack_delete():
            return not self.ostcm.stack.exists(self.stack_name)

        commons.LOG.info("Wait for RefApp heat stack to delete")
        wait.wait(wait_refapp_stack_delete)

        commons.LOG.info("Refapp heat stack successfully deleted")

    def collect_logs(self, dst_dir=settings.ARTIFACTS_DIR):
        """KUBECONFIG configuration parameter is required
           Direct SSH connect to refapp stack instances is required for
           correct log collection. If there is no SSH connection, console
           logs will be collected, but it will take more time due SSH
           connect timeout
        """

        if not self.ostcm.stack.exists(self.stack_name):
            commons.LOG.error(f"Stack {self.stack_name} does not exist. Nothing to collect.")
            return

        stack_info = self.ostcm.stack.show([self.stack_name])
        commons.LOG.info("Collect logs from RefApp stack")
        work_dir = os.path.join(dst_dir, self.stack_name + ".log")
        os.makedirs(work_dir, exist_ok=True)
        with open(os.path.join(work_dir, "stack_info.yaml"), "w") as f:
            yaml.safe_dump(stack_info, f)

        vm_list = self.ostcm.stack.resource_list(
            [
                "-n",
                "10",
                self.stack_name,
                "-c",
                "physical_resource_id",
                "--filter",
                "type=OS::Nova::Server",
                "-f",
                "value",
            ]
        ).split()
        if vm_list:
            instances_dir = os.path.join(work_dir, "instances")
            os.makedirs(instances_dir, exist_ok=True)
            lb_list = self.ostcm.stack.resource_list(
                [
                    "-n",
                    "10",
                    self.stack_name,
                    "-c",
                    "physical_resource_id",
                    "--filter",
                    "type=OS::Octavia::LoadBalancer",
                    "-f",
                    "value",
                ]
            ).split()
            if lb_list:
                current_dir = os.path.join(work_dir, "loadbalancer")
                os.makedirs(current_dir, exist_ok=True)
                for lb_item in lb_list:
                    lb_info = self.ostcm.loadbalancer.show([lb_item])
                    with open(os.path.join(current_dir, lb_info["name"] + ".yaml"), "w") as f:
                        yaml.safe_dump(lb_info, f)
                    amphora_list = self.ostcm.loadbalancer.amphora_list(
                        ["--loadbalancer", lb_item, "-c", "id", "-f", "value"]
                    ).split()
                    if amphora_list:
                        for amphora_item in amphora_list:
                            amphora_info = self.ostcm.loadbalancer.amphora_show([amphora_item])
                            with open(os.path.join(current_dir, "amphora-" + amphora_item + ".yaml"), "w") as f:
                                yaml.safe_dump(amphora_info, f)
                            if amphora_info["compute_id"] is not None:
                                vm_list.append(amphora_info["compute_id"])
            with tempfile.NamedTemporaryFile() as keyfile:
                keyfile.write(self.ssh_private_key.encode())
                keyfile.flush()
                ssh_auth = commons.get_ssh_auth(username='ubuntu', key_filename=keyfile.name)
                for instance_id in vm_list:
                    instance_info = self.ostcm.server.show([instance_id])
                    if instance_info and isinstance(instance_info, dict):
                        current_dir = os.path.join(instances_dir, instance_info["name"])
                        os.makedirs(current_dir, exist_ok=True)
                        with open(os.path.join(current_dir, "instance_info.yaml"), "w") as f:
                            yaml.safe_dump(instance_info, f)
                        try:
                            instance_ip_list = self.ostcm.server_get_floating_ips(instance_id)
                            if instance_ip_list:
                                ssh_connect = commons.get_ssh_client(ip=instance_ip_list[0], auth=ssh_auth)
                                self.ostcm.server_collect_logs(
                                    ssh_connect=ssh_connect, arch_path=current_dir, docker_logs=True
                                )
                            else:
                                self.ostcm.server_collect_console(instance_id, arch_path=current_dir)
                        except (paramiko.ssh_exception.SSHException, TimeoutError):
                            commons.LOG.error(
                                f"Unable connect to {instance_id} by SSH. Do collecting console output."
                            )
                            self.ostcm.server_collect_console(instance_id, arch_path=current_dir)
