#    Copyright 2022 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from si_tests.clients.k8s import base


class V1KaaSMachinePool(base.BaseModel):
    pass


class V1KaaSMachinePoolList(base.BaseModelList):
    pass


class KaaSMachinePool(base.K8sNamespacedResource):
    resource_type = 'machinepool'
    model = V1KaaSMachinePool

    @property
    def node_type(self) -> str:
        if self.data.get('metadata', {}).get('labels', {}).get('cluster.sigs.k8s.io/control-plane'):
            return 'master'
        else:
            return 'node'

    @property
    def cluster_name(self) -> str:
        return self.data.get('metadata', {}).get('labels', {})['cluster.sigs.k8s.io/cluster-name']

    @property
    def region(self) -> str:
        return self.data.get('metadata', {}).get('labels', {}).get('kaas.mirantis.com/region', '')

    @property
    def provider(self) -> str:
        return self.data.get('metadata', {}).get('labels', {})['kaas.mirantis.com/provider']

    def replicas(self) -> int:
        return self.data['spec']['replicas']

    def ready_replicas(self) -> int:
        # If readyReplicas does not exist in status means that 0 readyReplicas available
        return self.data['status'].get('readyReplicas', 0)

    def actual_replicas(self) -> int:
        # If replicas does not exist in status means that 0 replicas available
        return self.data['status'].get('replicas', 0)

    def set_replicas(self, replicas):
        return self.patch(body={"spec": {"replicas": replicas}})


class KaaSMachinePoolManager(base.K8sBaseManager):
    model = V1KaaSMachinePoolList
    resource_class = KaaSMachinePool
    resource_group = 'kaas.mirantis.com'
    resource_version = 'v1alpha1'
    resource_plural = 'machinepools'

    def filter(self, namespace, node_type=None, cluster_name=None):
        machinepools = self.list(namespace=namespace)
        if node_type:
            machinepools = [machinepool.name for machinepool in machinepools
                            if machinepool.node_type == node_type]
        if cluster_name:
            machinepools = [machinepool.name for machinepool in machinepools
                            if machinepool.cluster_name == cluster_name]
        return machinepools
