#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import json
import tempfile
import yaml

import kubernetes
from kubernetes import client

from si_tests.clients.k8s.clusterrolebindings import \
    K8sClusterRoleBindingManager
from si_tests.clients.k8s.rolebindings import \
    K8sRoleBindingManager
from si_tests.clients.k8s.componentstatuses import \
    K8sComponentStatusManager
from si_tests.clients.k8s.configmaps import K8sConfigMapManager
from si_tests.clients.k8s.daemonsets import K8sDaemonSetManager
from si_tests.clients.k8s.deployments import K8sDeploymentManager
from si_tests.clients.k8s.endpoints import K8sEndpointsManager
from si_tests.clients.k8s.events import K8sEventManager
from si_tests.clients.k8s.horizontalpodautoscalers import \
    K8sHorizontalPodAutoscalerManager
from si_tests.clients.k8s.poddisruptionbudget import \
    K8sPodDisruptionBudgetManager
from si_tests.clients.k8s.ingresses import K8sIngressManager
from si_tests.clients.k8s.ipaddrs_ipam_mirantis import \
    IpaddrIpamMirantisManager
from si_tests.clients.k8s.subnets_ipam_mirantis import \
    SubnetIpamMirantisManager
from si_tests.clients.k8s.ipamhosts_ipam_mirantis import \
    IpamHostIpamMirantisManager
from si_tests.clients.k8s.l2template_ipam_mirantis import \
    L2TemplateIpamMirantisManager
from si_tests.clients.k8s.baremetalhostprofile_mirantis import \
    BaremetalHostProfileManager
from si_tests.clients.k8s.jobs import K8sJobManager
from si_tests.clients.k8s.cronjob import K8sCronJobManager
from si_tests.clients.k8s.nodemaintenancerequests import \
    NodeMaintenanceRequestManager
from si_tests.clients.k8s.nodedeletionrequests import \
    NodeDeletionRequestManager
from si_tests.clients.k8s.nodedisablenotifications import \
    NodeDisableNotificationManager
from si_tests.clients.k8s.clusterworkloadlocks import \
    ClusterWorkloadLockManager
from si_tests.clients.k8s.clustermaintenancerequests import \
    ClusterMaintenanceRequestManager
from si_tests.clients.k8s.nodeworkloadlocks import NodeWorkloadLockManager
from si_tests.clients.k8s.kaas_awscredentials import KaaSAWSCredentialManager
from si_tests.clients.k8s.kaas_vspherecredentials import \
    KaaSVsphereCredentialManager
from si_tests.clients.k8s.kaas_equinixmetalcredentials import \
    KaaSEquinixMetalCredentialManager
from si_tests.clients.k8s.kaas_azurecredentials import \
    KaaSAzureCredentialManager
from si_tests.clients.k8s.kaas_rhellicenses import KaaSRHELLicenseManager
from si_tests.clients.k8s.kaas_scope import KaaSScopeManager
from si_tests.clients.k8s.kaas_ansibleextra import KaaSAnsibleExtraManager
from si_tests.clients.k8s.kaas_baremetalhosts import KaaSBareMetalHostManager
from si_tests.clients.k8s.kaas_baremetalhostinventories import KaaSBareMetalHostInventoryManager
from si_tests.clients.k8s.baremetalhostcredential import BaremetalHostCredentialManager
from si_tests.clients.k8s.kaas_cephclusters import KaaSCephClusterManager
from si_tests.clients.k8s.kaas_cephoperationrequests import KaaSCephOperationRequestManager
from si_tests.clients.k8s.kaas_clusterreleases import KaaSClusterReleaseManager
from si_tests.clients.k8s.kaas_clusters import KaaSClusterManager
from si_tests.clients.k8s.kaas_clusterupgradestatuses import KaaSClusterUpgradeStatusesManager
from si_tests.clients.k8s.kaas_clusterdeploystatus import KaaSClusterDeploymentStatusesManager
from si_tests.clients.k8s.kaas_clusterupdateplans import KaaSClusterUpdatePlansManager
from si_tests.clients.k8s.kaas_helmbundles import KaaSHelmbundleManager
from si_tests.clients.k8s.kaas_kaasreleases import KaaSKaaSReleaseManager
from si_tests.clients.k8s.kaas_byocredentials import KaaSByoCredentialManager
from si_tests.clients.k8s.kaas_lcmclusters import KaaSLcmClusterManager
from si_tests.clients.k8s.kaas_lcmclustermkeconfigs import KaaSLcmClusterMkeConfigManager
from si_tests.clients.k8s.kaas_lcmmachines import KaaSLcmMachineManager
from si_tests.clients.k8s.kaas_machines import KaaSMachineManager
from si_tests.clients.k8s.kaas_machineupgradestatuses import KaaSMachineUpgradeStatusesManager
from si_tests.clients.k8s.kaas_metallbconfig import KaaSMetalLBConfigManager
from si_tests.clients.k8s.kaas_metallbconfigtemplate import MetalLBConfigTemplateManager
from si_tests.clients.k8s.kaas_racks import KaaSRackManager
from si_tests.clients.k8s.kaas_multirackclusters import KaaSMultiRackClusterManager
from si_tests.clients.k8s.addresspools import KaaSAddressPoolManager
from si_tests.clients.k8s.bgpadvertisements import KaaSBGPAdvertisementManager
from si_tests.clients.k8s.bgppeers import KaaSBGPPeerManager
from si_tests.clients.k8s.cassandra import CassandraClusterManager
from si_tests.clients.k8s.kaas_machinepools import KaaSMachinePoolManager
from si_tests.clients.k8s.kaas_openstackcredentials import \
    KaaSOpenStackCredentialManager
from si_tests.clients.k8s.kaas_openstackkeypairs import \
    KaaSOpenStackKeypairManager
from si_tests.clients.k8s.kaas_openstackresources import \
    KaaSOpenStackResourceManager
from si_tests.clients.k8s.kaas_awsresources import \
    KaaSAWSResourceManager
from si_tests.clients.k8s.kaas_publickeys import KaaSPublicKeyManager
from si_tests.clients.k8s.kaas_proxies import KaaSProxyManager
from si_tests.clients.k8s.rookcephcluster import \
    RookCephClusterManager
from si_tests.clients.k8s.miracephs import \
    MiracephManager
from si_tests.clients.k8s.miracephhealths import \
    MiracephHealthManager
from si_tests.clients.k8s.miracephsecrets import \
    MiracephSecretManager
from si_tests.clients.k8s.cephosdremoverequests import \
    CephOsdRemoveRequestManager
from si_tests.clients.k8s.cephperftestrequests import \
    CephPerfTestRequestManager
from si_tests.clients.k8s.limitranges import K8sLimitRangeManager
from si_tests.clients.k8s.namespaces import K8sNamespaceManager
from si_tests.clients.k8s.networkpolicies import K8sNetworkPolicyManager
from si_tests.clients.k8s.nodes import K8sNodeManager
from si_tests.clients.k8s.openstack_deployment import \
    OpenStackDeploymentManager
from si_tests.clients.k8s.openstack_deployment_status import \
    OpenStackDeploymentStatusManager
from si_tests.clients.k8s.persistentvolumeclaims import \
    K8sPersistentVolumeClaimManager
from si_tests.clients.k8s.persistentvolumes import \
    K8sPersistentVolumeManager
from si_tests.clients.k8s.pods import K8sPodManager
from si_tests.clients.k8s.replicasets import K8sReplicaSetManager
from si_tests.clients.k8s.replicationcontrollers import \
    K8sReplicationControllerManager
from si_tests.clients.k8s.resourcequotas import K8sResourceQuotaManager
from si_tests.clients.k8s.secrets import K8sSecretManager
from si_tests.clients.k8s.serviceaccounts import \
    K8sServiceAccountManager
from si_tests.clients.k8s.services import K8sServiceManager
from si_tests.clients.k8s.statefulsets import K8sStatefulSetManager
from si_tests.clients.k8s.tfoperator import (TFOperatorManager,
                                             TFOperatorManagerV2)
from si_tests.clients.k8s.tfanalytics import (TFAnalyticsManager,
                                              TFAnalyticsManagerV2)
from si_tests.clients.k8s.tfconfig import (TFConfigManager,
                                           TFConfigManagerV2)
from si_tests.clients.k8s.tfcontrol import (TFControlManager,
                                            TFControlManagerV2)
from si_tests.clients.k8s.tfdbrestore import (TFDbRestoreManager,
                                              TFDbRestoreManagerV2)
from si_tests.clients.k8s.tfdbrepair import TFDbRepairManagerV2
from si_tests.clients.k8s.tfdbbackup import (TFDbBackupManager,
                                             TFDbBackupManagerV2)
from si_tests.clients.k8s.tfimageprecaching import (TFImagePreCachingManager,
                                                    TFImagePreCachingManagerV2)
from si_tests.clients.k8s.tftest import TFTestManager, TFTestManagerV2
from si_tests.clients.k8s.tftool import TFToolManager, TFToolManagerV2
from si_tests.clients.k8s.tfvrouter import TFVrouterManager, TFVrouterManagerV2
from si_tests.clients.k8s.tfwebui import TFWebUIManager, TFWebUIManagerV2
from si_tests.clients.k8s.updategroups import UpdateGroupManager
from si_tests.clients.k8s.iam_users import IAMUserManager
from si_tests.clients.k8s.iam_rolebindings import IAMRoleBindingManager
from si_tests.clients.k8s.iam_rolebindings import IAMClusterRoleBindingManager
from si_tests.clients.k8s.iam_rolebindings import IAMGlobalRoleBindingManager
from si_tests.clients.k8s.kaas_mccupgrades import KaaSMccUpgradesManager
from si_tests.clients.iam import keycloak_client
from si_tests.clients.k8s.license import LicenseManager
from si_tests.clients.k8s.certificate_configuration import CertificateConfigurationManager
from si_tests.clients.k8s.mcc_certificate_request import MCCCertificateRequestManager
from si_tests.clients.k8s.certificate_request import CertificateRequestManager
from si_tests.clients.k8s.issuer import IssuerManager
from si_tests.clients.k8s.containerregistry import ContainerRegistryManager
from si_tests.clients.k8s.kaas_bootstrapregions import KaaSBootstrapRegionManager
from si_tests.clients.k8s.kaas_lcmclusterstate import KaaSLcmClusterStateManager
from si_tests.clients.k8s.kaas_gracefulrebootrequest import GracefulRebootRequestManager
from si_tests.clients.k8s.kaas_tlsconfigs import KaaSTLSConfigManager
from si_tests.clients.k8s.storageclass import K8sStorageClassManager
from si_tests.clients.k8s.kaas_vspherevmtemplates import KaaSVsphereVMTemplateManager
from si_tests.clients.k8s.kaas_cachewarmuprequest import KaaSCacheWarmupRequestManager
from si_tests.clients.k8s.kaas_hostosconfigurations import KaaSHostOSConfigurationManager
from si_tests.clients.k8s.kaas_hostosconfigurationmodules import KaaSHostOSConfigurationModuleManager
from si_tests.clients.k8s.diagnostic_diagnostic import DiagnosticDiagnosticManager
from si_tests.clients.k8s.diagnostic_diagnosticrelease import DiagnosticReleaseManager
from si_tests.clients.k8s.leases import LeasesManager
from si_tests.clients.k8s.kaas_infraconnectivitymonitors import KaaSInfraConnectivityMonitorManager
from si_tests.clients.k8s.cnnc_netcheckertargetsconfigs import CnnCNetCheckerTargetsConfigManager
from si_tests.clients.k8s.cnnc_checkerinventoryconfigs import CnnCCheckerInventoryConfigManager
from si_tests import logger

LOG = logger.logger


class K8sCluster(object):
    def __init__(self, schema="https", user=None, password=None, ca=None,
                 host='localhost', port='443', default_namespace='default',
                 config_data=None, kubeconfig=None, cluster_name='kaas',
                 idp_ca=None, client_id="kaas", idp_issuer_url=None):
        self.default_namespace = default_namespace
        self.__schema = schema
        self.__user = user
        self.__password = password
        self.__ca = ca
        self.__host = host
        self.__port = port
        self.__config_data = config_data
        self.__kubeconfig = kubeconfig
        self.__cluster_name = cluster_name
        self.__idp_ca = idp_ca
        self.__client_id = client_id
        self.__idp_issuer_url = idp_issuer_url

        self.login()
        self.init_apis()
        self.init_managers()

    def generate_config_data(self, offline=False):
        if self.__idp_issuer_url:
            keycloak_url = self.__idp_issuer_url.split(
                "//")[1].split("/")[0]
        else:
            raise Exception("idp issuer url was not found")
        usr = keycloak_client.KeycloakUserClient(
            keycloak_url, self.__user, self.__password,
            client_id=self.__client_id)
        if offline:
            token = usr.get_offline_openid_token()
        else:
            token = usr.get_openid_token()
        id_token = token['id_token']
        refresh_token = token['refresh_token']
        api_server = f'{self.__schema}://{self.__host}:{self.__port}'
        current_context = f'{self.__user}@{self.__cluster_name}'

        config_data = {
            'apiVersion': 'v1',
            'kind': 'Config',
            'preferences': {},
            'current-context': current_context,
            'clusters': [{
                'name': self.__cluster_name,
                'cluster': {
                    'server': api_server,
                },
            }],
            'users': [{
                'name': self.__user,
                'user': {
                    'auth-provider': {
                        'config': {
                            'client-id': self.__client_id,
                            'id-token': id_token,
                            'client-secret': "fake-fake",
                            'idp-certificate-authority-data': self.__idp_ca,
                            'idp-issuer-url': self.__idp_issuer_url,
                            'refresh-token': refresh_token
                        },
                        'name': 'oidc'
                    }
                }
            }],
            'contexts': [{
                'name': current_context,
                'context': {
                    'cluster': self.__cluster_name,
                    'user': self.__user,
                },
            }],
        }
        if self.__ca is not None:
            config_data['clusters'][0]['cluster']['certificate-authority-data'] = self.__ca
        else:
            config_data['clusters'][0]['cluster']['insecure-skip-tls-verify'] = 'true'

        return config_data

    def update_kubeconfig(self, kubeconfig):
        """Update path to the KUBECONFIG file and relogin"""
        self.__kubeconfig = kubeconfig
        self.login()
        self.init_apis()
        self.init_managers()

    def login(self):
        if self.__kubeconfig:
            # Load original kubeconfig
            with open(self.__kubeconfig, 'r') as f:
                kubeconfig_data = yaml.safe_load(f.read())
            # Patch kubeconfig
            for user in kubeconfig_data.get('users', []):
                auth_provider = user.get('user', {}).get('auth-provider', {})
                if auth_provider.get('name', '') == 'oidc':
                    auth_provider_config = auth_provider.get('config')
                    if auth_provider_config and 'client-secret' not in auth_provider_config:
                        LOG.warning(f"Patching 'kubeconfig' OIDC config for user {user.get('name')}")
                        auth_provider_config['client-secret'] = 'si-fake-secret'
            # Init kubernetes client using patched kubeconfig
            with tempfile.NamedTemporaryFile(mode="w") as tmp:
                tmp.write(yaml.safe_dump(kubeconfig_data))
                tmp.flush()
                kubernetes.config.load_kube_config(config_file=tmp.name)
            self.__api_client = client.ApiClient()
        else:
            configuration = type.__call__(client.Configuration)
            config_data = self.__config_data
            if not config_data:
                config_data = self.generate_config_data()
            loader = kubernetes.config.kube_config.KubeConfigLoader(
                config_data
            )
            LOG.debug("Initialized kubeconfig: {}".format(
                json.dumps(config_data, indent=4)))
            loader.load_and_set(configuration)
            self.__api_client = client.ApiClient(configuration=configuration)

    def init_apis(self):
        self.api_core = client.CoreV1Api(self.api_client)
        self.api_apps = client.AppsV1Api(self.api_client)
        self.api_custom = client.CustomObjectsApi(self.api_client)
        self.api_autoscaling = client.AutoscalingV1Api(self.api_client)
        self.api_batch = client.BatchV1Api(self.api_client)
        self.api_batch_v1beta1 = client.BatchApi(self.api_client)
        self.api_policy_v1beta1 = client.PolicyApi(self.api_client)
        self.api_rbac_auth = client.RbacAuthorizationV1Api(self.api_client)
        self.api_version = client.VersionApi(self.api_client)
        self.api_apis = client.ApisApi(self.api_client)
        self.api_admission = client.AdmissionregistrationV1Api(self.api_client)
        self.api_storage = client.StorageV1Api(self.api_client)
        self.api_networking = client.NetworkingV1Api(self.api_client)

    def init_managers(self):
        self.nodes = K8sNodeManager(self)
        self.pods = K8sPodManager(self)
        self.endpoints = K8sEndpointsManager(self)
        self.namespaces = K8sNamespaceManager(self)
        self.openstackdeployment = OpenStackDeploymentManager(self)
        self.openstackdeploymentstatus = OpenStackDeploymentStatusManager(self)
        self.services = K8sServiceManager(self)
        self.serviceaccounts = K8sServiceAccountManager(self)
        self.secrets = K8sSecretManager(self)
        self.statefulsets = K8sStatefulSetManager(self)
        self.events = K8sEventManager(self)
        self.limitranges = K8sLimitRangeManager(self)
        self.jobs = K8sJobManager(self)
        self.cronjobs = K8sCronJobManager(self)
        self.daemonsets = K8sDaemonSetManager(self)
        self.ingresses = K8sIngressManager(self)
        self.deployments = K8sDeploymentManager(self)
        self.horizontalpodautoscalers = K8sHorizontalPodAutoscalerManager(self)
        self.componentstatuses = K8sComponentStatusManager(self)
        self.resourcequotas = K8sResourceQuotaManager(self)
        self.replicationcontrollers = K8sReplicationControllerManager(self)
        self.pvolumeclaims = K8sPersistentVolumeClaimManager(self)
        self.pvolumes = K8sPersistentVolumeManager(self)
        self.replicasets = K8sReplicaSetManager(self)
        self.networkpolicies = K8sNetworkPolicyManager(self)
        self.clusterrolebindings = K8sClusterRoleBindingManager(self)
        self.rolebindings = K8sRoleBindingManager(self)
        self.configmaps = K8sConfigMapManager(self)
        self.pdb = K8sPodDisruptionBudgetManager(self)

        self.kaas_clusters = KaaSClusterManager(self)
        self.rookcephclusters = RookCephClusterManager(self)
        self.miracephs = MiracephManager(self)
        self.miracephhealths = MiracephHealthManager(self)
        self.miracephsecrets = MiracephSecretManager(self)
        self.cephosdremoverequests = CephOsdRemoveRequestManager(self)
        self.cephperftestrequests = CephPerfTestRequestManager(self)
        self.kaas_cephclusters = KaaSCephClusterManager(self)
        self.kaas_cephoperationrequests = KaaSCephOperationRequestManager(self)
        self.kaas_clusterupgradestatus = KaaSClusterUpgradeStatusesManager(self)
        self.kaas_clusterdeploystatus = KaaSClusterDeploymentStatusesManager(self)
        self.kaas_helmbundles = KaaSHelmbundleManager(self)
        self.kaas_lcmclusters = KaaSLcmClusterManager(self)
        self.kaas_lcmclustermkeconfigs = KaaSLcmClusterMkeConfigManager(self)
        self.kaas_lcmcluster_states = KaaSLcmClusterStateManager(self)
        self.kaas_lcmmachines = KaaSLcmMachineManager(self)
        self.kaas_machines = KaaSMachineManager(self)
        self.kaas_machineupgradestatus = KaaSMachineUpgradeStatusesManager(self)
        self.kaas_machinepools = KaaSMachinePoolManager(self)
        self.kaas_clusterupdateplans = KaaSClusterUpdatePlansManager(self)
        # metallb
        self.kaas_metallbconfig = KaaSMetalLBConfigManager(self)
        self.kaas_metallbconfigtemplate = MetalLBConfigTemplateManager(self)
        #
        self.kaas_racks = KaaSRackManager(self)
        self.kaas_multirack_cluster = KaaSMultiRackClusterManager(self)
        self.addresspool = KaaSAddressPoolManager(self)
        self.bgpadvertisement = KaaSBGPAdvertisementManager(self)
        self.bgppeer = KaaSBGPPeerManager(self)
        self.cassandracluster = CassandraClusterManager(self)
        self.kaas_ansibleextras = KaaSAnsibleExtraManager(self)
        self.kaas_baremetalhosts = KaaSBareMetalHostManager(self)
        self.kaas_baremetalhostinventories = KaaSBareMetalHostInventoryManager(self)
        self.kaas_baremetalhostscredentials = BaremetalHostCredentialManager(self)
        self.kaas_openstackkeypairs = KaaSOpenStackKeypairManager(self)
        self.kaas_openstackresources = KaaSOpenStackResourceManager(self)
        self.kaas_awsresources = KaaSAWSResourceManager(self)
        self.kaas_publickeys = KaaSPublicKeyManager(self)
        self.kaas_proxies = KaaSProxyManager(self)
        self.kaas_clusterreleases = KaaSClusterReleaseManager(self)
        self.kaas_kaasreleases = KaaSKaaSReleaseManager(self)
        self.kaas_byocredentials = KaaSByoCredentialManager(self)
        self.kaas_tlsconfigs = KaaSTLSConfigManager(self)
        self.ipam_ipaddrs = IpaddrIpamMirantisManager(self)
        self.ipam_subnets = SubnetIpamMirantisManager(self)
        self.ipam_hosts = IpamHostIpamMirantisManager(self)
        self.l2templates = L2TemplateIpamMirantisManager(self)
        self.baremetalhostprofiles = BaremetalHostProfileManager(self)
        self.kaas_openstackcredentials = KaaSOpenStackCredentialManager(self)
        self.kaas_awscredentials = KaaSAWSCredentialManager(self)
        self.kaas_vspherecredentials = KaaSVsphereCredentialManager(self)
        self.kaas_equinixmetalcredentials = \
            KaaSEquinixMetalCredentialManager(self)
        self.kaas_azurecredentials = \
            KaaSAzureCredentialManager(self)
        self.kaas_rhellicenses = KaaSRHELLicenseManager(self)
        self.kaas_scopes = KaaSScopeManager(self)
        self.tfoperator = TFOperatorManager(self)
        self.tfoperator_v2 = TFOperatorManagerV2(self)
        self.tfanalytics = TFAnalyticsManager(self)
        self.tfanalytics_v2 = TFAnalyticsManagerV2(self)
        self.tfconfig = TFConfigManager(self)
        self.tfconfig_v2 = TFConfigManagerV2(self)
        self.tfcontrol = TFControlManager(self)
        self.tfcontrol_v2 = TFControlManagerV2(self)
        self.tfdbrestores = TFDbRestoreManager(self)
        self.tfdbrestores_v2 = TFDbRestoreManagerV2(self)
        self.tfdbbackup = TFDbBackupManager(self)
        self.tfdbbackup_v2 = TFDbBackupManagerV2(self)
        self.tfdbrepairs = TFDbRepairManagerV2(self)
        self.tfimageprecaching = TFImagePreCachingManager(self)
        self.tfimageprecaching_v2 = TFImagePreCachingManagerV2(self)
        self.tftest = TFTestManager(self)
        self.tftest_v2 = TFTestManagerV2(self)
        self.tftool = TFToolManager(self)
        self.tftool_v2 = TFToolManagerV2(self)
        self.tfvrouter = TFVrouterManager(self)
        self.tfvrouter_v2 = TFVrouterManagerV2(self)
        self.tfwebui = TFWebUIManager(self)
        self.tfwebui_v2 = TFWebUIManagerV2(self)
        self.iam_users = IAMUserManager(self)
        self.iam_rolebindings = IAMRoleBindingManager(self)
        self.iam_clusterrolebindings = IAMClusterRoleBindingManager(self)
        self.iam_globalrolebindings = IAMGlobalRoleBindingManager(self)
        self.nodemaintenancerequests = NodeMaintenanceRequestManager(self)
        self.nodedeletionrequests = NodeDeletionRequestManager(self)
        self.nodeworkloadlocks = NodeWorkloadLockManager(self)
        self.clusterworkloadlocks = ClusterWorkloadLockManager(self)
        self.clustermaintenancerequests = ClusterMaintenanceRequestManager(self)
        self.kaas_mccupgrades = KaaSMccUpgradesManager(self)
        self.license = LicenseManager(self)
        self.certificate_requests = CertificateRequestManager(self)
        self.certificat_configurations = CertificateConfigurationManager(self)
        self.mcc_certificate_requests = MCCCertificateRequestManager(self)
        self.issuers = IssuerManager(self)
        self.containerregistry = ContainerRegistryManager(self)
        self.kaas_bootstrapregions = KaaSBootstrapRegionManager(self)
        self.gracefulrebootrequest = GracefulRebootRequestManager(self)
        self.storageclass = K8sStorageClassManager(self)
        self.vvmt = KaaSVsphereVMTemplateManager(self)
        self.kaas_cachewarmuprequests = KaaSCacheWarmupRequestManager(self)
        self.kaas_hostosconfigurations = KaaSHostOSConfigurationManager(self)
        self.kaas_hostosconfigurationmodules = KaaSHostOSConfigurationModuleManager(self)
        self.diagnostic_diagnostics = DiagnosticDiagnosticManager(self)
        self.diagnostic_diagnosticreleases = DiagnosticReleaseManager(self)
        self.leases = LeasesManager(self)
        self.nodedisablenotifications = NodeDisableNotificationManager(self)
        self.updategroups = UpdateGroupManager(self)
        self.kaas_infraconnectivitymonitors = KaaSInfraConnectivityMonitorManager(self)
        self.cnnc_netcheckertargetsconfigs = CnnCNetCheckerTargetsConfigManager(self)
        self.cnnc_checkerinventoryconfigs = CnnCCheckerInventoryConfigManager(self)

    @property
    def api_client(self):
        return self.__api_client
