import concurrent.futures
import time
import os
import yaml
import json
import numpy as np

from si_tests.managers import helm3_manager
from si_tests.deployments.utils import file_utils
from si_tests import settings
from si_tests import logger


LOG = logger.logger


def install_refapp_chart(chart_name, helm_manager, kubeconfig_path, namespace,
                         chart_path, chart_values_path, timeout):
    LOG.info(f"Creating chart: {namespace}/{chart_name}")
    start = time.time()

    install_result = helm_manager.install_chart(
        kubeconfig_path=kubeconfig_path,
        namespace=namespace,
        chart_name=chart_name,
        chart_path=chart_path,
        values_path=chart_values_path,
        timeout=timeout)
    assert install_result, f"Helm Chart {chart_name} has not been installed"

    end = time.time()
    result = end - start
    LOG.info(f"{namespace}/{chart_name}: SUCCESS in {result:.2f} seconds")
    return result


def get_density_statistics(results):
    arr = np.array(list(results.values()))
    arr = arr[arr != 0]
    return {
        "avg": np.average(arr),
        "med": np.median(arr),
        "min": np.min(arr),
        "max": np.max(arr),
        "90p": np.percentile(arr, 90),
        "95p": np.percentile(arr, 95)
    }


def install_charts_in_parallel(kubectl_client, kubeconfig_path, total_chart_count, chart_per_namespace_limit, namespace,
                               prefix, concurrency, refapp_chart, chart_values_path, warmup=False, timeout="5m"):
    iteration = 0
    results = {}

    while total_chart_count > 0:
        iteration += 1
        ns = namespace + f"-{iteration}"
        charts_per_loop = min(total_chart_count, chart_per_namespace_limit)
        total_chart_count -= charts_per_loop
        digit_count = len(str(charts_per_loop))

        # Creating namespace
        LOG.info(f"Creating namespace: {ns}")
        if kubectl_client.namespaces.present(ns):
            density_ns = kubectl_client.namespaces.get(ns)
            LOG.info(f"Namespace '{ns}' already exists")
        else:
            density_ns = kubectl_client.namespaces.create(
                name=ns,
                body={'metadata': {'name': ns}})
        helm_manager = helm3_manager.Helm3Manager(namespace=ns)

        with concurrent.futures.ThreadPoolExecutor(max_workers=concurrency, thread_name_prefix=prefix) as executor:
            futures = {}

            # Start installing charts in parallel
            for idx in range(charts_per_loop):
                chart_name = prefix + str(idx).zfill(digit_count)
                futures[chart_name] = executor.submit(
                    install_refapp_chart,
                    chart_name,
                    helm_manager,
                    kubeconfig_path,
                    ns,
                    refapp_chart,
                    [chart_values_path],
                    timeout=timeout
                )

        # Get installation time of each chart
        for chart_name, value in futures.items():
            results[f"{ns}-{chart_name}"] = value.result()

        LOG.info(f"Namespace {ns} is filled")
        if warmup:
            LOG.info(f"Deleting density namespace: {ns}")
            density_ns.delete()

    return results


def test_density_refapp_k8s(kaas_manager):
    """Deploying multiple instances of a reference application (refapp)
    Helm chart in a Kubernetes environment to perfom a density test.

    Scenario:
        1. Get the k8s client and kubeconfig of the target cluster for testing.
        2. Set a limit of charts per namespace.
        3. Create density namespace for each iteration.
        4. Install a specified number of refapp Helm charts in parallel.
        5. Measure the installation time for each chart.
        6. Generate statistics on the installation times.
    """
    tg_ns_name = settings.TARGET_NAMESPACE
    tg_cluster_name = settings.TARGET_CLUSTER
    refapp_chart = settings.REFAPP_CHART_URL
    chart_per_namespace_limit = settings.DENSITY_CHART_PER_NS_LIMIT
    concurrency = settings.DENSITY_CONCURRENCY
    total_chart_count = settings.DENSITY_CHART_COUNT
    prefix = settings.DENSITY_PREFIX + '-'
    namespace = settings.DENSITY_NAMESPACE
    density_values = settings.DENSITY_VALUES
    artifacts_dir = settings.ARTIFACTS_DIR
    warmup = settings.DENSITY_WARMUP

    # Prepare values for chart
    if density_values:
        values = yaml.load(density_values, Loader=yaml.SafeLoader)
    chart_values_path = os.path.join(artifacts_dir, "values.yaml")
    file_utils.save_to_yaml(values, chart_values_path)

    # Get target cluster
    tg_ns = kaas_manager.get_namespace(tg_ns_name)
    tg_cluster = tg_ns.get_cluster(tg_cluster_name)
    kubectl_client = tg_cluster.k8sclient

    # Get child kubeconfig
    kubeconfig_name, child_kubeconfig = tg_cluster.get_kubeconfig()
    kubeconfig_path = os.path.join(artifacts_dir, kubeconfig_name)
    LOG.info(f"Saving child cluster kubeconfig to {kubeconfig_path}")
    with open(kubeconfig_path, "w") as f:
        f.write(child_kubeconfig)
    os.chmod(path=kubeconfig_path, mode=400)

    if warmup:
        worker_nodes_count = len(tg_cluster.get_machines(machine_type="worker"))
        LOG.info(f"The number of working nodes: {worker_nodes_count}")

        LOG.info("Start warmup density")
        warmup_results = install_charts_in_parallel(
            kubectl_client=kubectl_client,
            kubeconfig_path=kubeconfig_path,
            total_chart_count=worker_nodes_count,
            chart_per_namespace_limit=chart_per_namespace_limit,
            namespace=f"warmup-{namespace}",
            prefix=prefix,
            concurrency=concurrency,
            refapp_chart=refapp_chart,
            chart_values_path=chart_values_path,
            warmup=warmup,
            timeout="30m")
        LOG.info("Warmup is finished!")

        warmup_statistics = get_density_statistics(warmup_results)
        LOG.info(f"Warmup statistics: {warmup_statistics}")

    LOG.info("Start density")
    results = install_charts_in_parallel(
        kubectl_client=kubectl_client,
        kubeconfig_path=kubeconfig_path,
        total_chart_count=total_chart_count,
        chart_per_namespace_limit=chart_per_namespace_limit,
        namespace=namespace,
        prefix=prefix,
        concurrency=concurrency,
        refapp_chart=refapp_chart,
        chart_values_path=chart_values_path)
    LOG.info("Density is finished!")

    # Get statistics
    statistics = get_density_statistics(results)

    LOG.info(f"Save statistics: {statistics}")
    with open(os.path.join(artifacts_dir, "statistics.json"), "w") as f:
        json.dump(statistics, f)
