#    Copyright 2020 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import yaml

from si_tests import logger
from si_tests import settings
from jsonpath_ng.ext import parse
from si_tests.utils import waiters

LOG = logger.logger


def get_kaas_release(kaas_manager, upgrade_spec):
    LOG.info("Getting KaasRelease object from the cluster ...")

    kaas_release_spec = upgrade_spec.get('kaasrelease', {})
    if 'name' in kaas_release_spec:
        if 'namespace' in kaas_release_spec:
            return kaas_manager.get_kaasrelease(
                name=kaas_release_spec['name'],
                namespace=kaas_release_spec['namespace'])
        else:
            return kaas_manager.get_kaasrelease(
                name=kaas_release_spec['name'])
    else:
        kaas_releases_available = kaas_manager.get_kaasreleases()
        LOG.info('Available KaaS releases are {}'.format(
            [x.name for x in kaas_releases_available]))

        def kaas_releases_enabled_filter(kaas_release):
            name = 'kaas.mirantis.com/active'
            value = kaas_release.data['metadata'].get('labels', {}).get(name)
            if value is None:
                return False
            if isinstance(value, str):
                return value.lower() == 'true'
            if isinstance(value, bool):
                return value
            raise Exception("Bad label value: '{}' == '{}'"
                            .format(name, value))

        kaas_releases_enabled = list(
            filter(kaas_releases_enabled_filter, kaas_releases_available))

        assert len(kaas_releases_enabled) > 0, \
            "No active 'kaasrelease' objects found"

        assert len(kaas_releases_enabled) == 1, (
            "There are more than one active 'kaasrelease' objects: {}".format(
                [x.name for x in kaas_releases_enabled]))

        return kaas_releases_enabled[0]


def get_cluster(kaas_manager, upgrade_spec):
    LOG.info("Getting Cluster object from the cluster ...")

    cluster_spec = upgrade_spec.get('cluster', {})
    cluster_name = cluster_spec.get('name', settings.CLUSTER_NAME)
    cluster_namespace = cluster_spec.get('namespace',
                                         settings.CLUSTER_NAMESPACE)

    namespace = kaas_manager.get_namespace(cluster_namespace)
    cluster = namespace.get_cluster(cluster_name)
    return cluster


def get_patched_kaas_release_body(kaas_release, upgrade_spec):
    LOG.info("Update helm repositories in kaasrelease object")

    body = kaas_release.data
    # FIXME: Workaround for PRODX-2534
    body['apiVersion'] = body.pop('api_version')
    body['metadata']['name'] = '{}-test'.format(body['metadata']['name'])

    for helm_repo in upgrade_spec['helm'].get('repositories', []):
        expr = parse("$.spec.helmRepositories[?name='{}']"
                     .format(helm_repo['name']))
        matches = expr.find(body)
        if matches:
            for match in matches:
                match.value['url'] = helm_repo['url']
        else:
            body['spec']\
                .setdefault('helmRepositories', [])\
                .append(helm_repo)

    for helm_chart in upgrade_spec['helm'].get('releases', []):
        chart = helm_chart.get('chart')
        version = helm_chart.get('version')

        expr = parse("$.spec.regional[?provider='baremetal']"
                     ".helmReleases[?name='{}']".format(helm_chart['name']))
        matches = expr.find(body)
        if matches:
            for match in matches:
                if chart:
                    match.value['chart'] = chart
                if version:
                    match.value['version'] = version
        else:
            expr = parse("$.spec.regional[?provider='baremetal'].helmReleases")
            for match in expr.find(body):
                if 'namespace' not in helm_chart:
                    # default namespace for BM charts
                    helm_chart['namespace'] = 'kaas'
                match.value.append(helm_chart)

    return body


def get_patched_cluster_body(cluster, upgrade_spec):
    LOG.info("Update KAAS BM components version")

    body = cluster.data
    # FIXME: Workaround for PRODX-2534
    body['apiVersion'] = body.pop('api_version')

    for helm_chart in upgrade_spec['helm'].get('releases', []):
        values = helm_chart.get('values')

        if values:
            expr = parse("$.spec.providerSpec.value.kaas"
                         ".regional[?provider='baremetal']"
                         ".helmReleases[?name='{}']"
                         .format(helm_chart['name']))
            for match in expr.find(body):
                match.value['values'] = values

    return body


def ensure_helmbundle(helmbundle_data, name, chart=None, version=None):
    LOG.info("Ensure helmbundle for chart '{}'".format(name))

    chart_status = helmbundle_data['status']\
        .get('releaseStatuses', {}).get(name)

    assert chart_status is not None, (
        "Chart '{}' doesn't exist in helmbundle status".format(name))
    LOG.info("chart_status ok")

    chart_name = chart_status['chart']
    chart_message = chart_status.get('message', '<not set>')
    assert chart_status['success'] is True, \
        "Chart update failed: chart={}, message='{}'".format(chart_name,
                                                             chart_message)
    LOG.info("chart_status['success'] ok")

    assert chart_status['chart'] == name, "Chart name mismatch"
    LOG.info("chart_status['chart'] == '{}'".format(name))

    if version is not None:
        assert chart_status['version'] == version, "Chart version mismatch"
        LOG.info("chart_status['version'] == '{}'".format(version))


def ensure_helmbundles(kaas_manager, name, upgrade_spec):
    namespace = upgrade_spec.get('cluster', {})\
        .get('namespace', settings.CLUSTER_NAMESPACE)
    helmbundle = kaas_manager.get_helmbundle(name=name,
                                             namespace=namespace)
    helmbundle_data = helmbundle.data

    for item in upgrade_spec['helm']['releases']:
        ensure_helmbundle(helmbundle_data, name=item['name'],
                          chart=item.get('chart'),
                          version=item.get('version'))


def verify_charts_upgraded(kaas_manager, kaasrelease_name, cluster_name,
                           upgrade_spec):
    # TODO: Add more checks here
    ensure_helmbundles(kaas_manager, cluster_name, upgrade_spec)


def test_update_kaas_bm_charts(kaas_manager):
    """ Test KAAS BM charts upgrade

    This performs a series of actions to test KaaS BM charts
    upgradability. It requires deployed KaaS Manangement Cluster.

    All k8s management performed via 'kubectl' CLI wrapper.

    Steps to test the upgrade:
        * get kaasrelease object (get exact object by name / namespace
          if those are given or find the active object by label)
        * get cluster object by name / namespace
        * replace kaas-ipam / baremetal-operator versions in kaasrelease
          object, save it under new name
        * replace kaasrelease version in cluster object, save it under
          new name
        * upload new kaasrelease / cluste object to seed node
        * apply new kaasrelease object, disable previous one and enable
          the new
        * apply changes to existing cluster object
        * verify the upgrade completed successfully. This is done in
          'verify_charts_upgraded' and may be extended if additional
          checks were added to this function.

    KAAS_BM_UPGRADE_SPEC env var is used to specify a path to a file
    that contains KaaS BM upgrade spec. The structure is the following:

        kaasrelease:
          name:
          namespace:
          version:
        cluster:
          name:
          namespace:
        helm:
          releases:
            - name:
              chart:
              version:
            ...
          repositories:
            - name:
              url:
            ...

    :param openstack_client:
    :return: None
    """

    upgrade_spec = None

    if settings.KAAS_BM_UPGRADE_SPEC_FILE:
        with open(settings.KAAS_BM_UPGRADE_SPEC_FILE) as f:
            upgrade_spec = yaml.load(f.read(), Loader=yaml.SafeLoader)

    if not upgrade_spec:
        LOG.warning("No KaaS BM upgrade spec given via "
                    "KAAS_BM_UPGRADE_SPEC_FILE, skipping test")
        return

    kaas_release = get_kaas_release(kaas_manager, upgrade_spec)
    cluster = get_cluster(kaas_manager, upgrade_spec)

    kaas_release_patched_body = get_patched_kaas_release_body(
        kaas_release, upgrade_spec)

    cluster_patched_body = get_patched_cluster_body(
        cluster, upgrade_spec)

    cluster_patched_body['spec']['providerSpec']['value']['kaas']['release'] \
        = kaas_release_patched_body['metadata']['name']

    LOG.info("Disable old KaasRelease ({})".format(kaas_release.name))
    kaas_release.patch(
        {'metadata': {'labels': {'kaas.mirantis.com/active': 'false'}}})

    name = kaas_release_patched_body['metadata']['name']
    namespace = kaas_release_patched_body['metadata'].get('namespace')
    LOG.info("Create new KaasRelease object ({})".format(name))
    kaas_manager.api.kaas_kaasreleases.create(
        name=name, namespace=namespace, body=kaas_release_patched_body)

    name = cluster_patched_body['metadata']['name']
    LOG.info("Apply updated cluster object ({})".format(name))
    cluster.patch({'spec': cluster_patched_body['spec']})

    kaasrelease_name = kaas_release_patched_body['metadata']['name']
    cluster_name = cluster_patched_body['metadata']['name']
    waiters.wait_pass(verify_charts_upgraded,
                      predicate_args=(kaas_manager, kaasrelease_name,
                                      cluster_name, upgrade_spec),
                      timeout=600, interval=10,
                      timeout_msg="Timeout while waiting for charts upgrade "
                                  "process to complete")
