import base64
import pytest

from si_tests import settings
from si_tests import logger
from si_tests.clients.dashboard.mke_dashboard_client import MKEDashboardClientPlain
from si_tests.clients.k8s import K8sCluster
from si_tests.utils import waiters, utils
from si_tests.utils import packaging_version as version

LOG = logger.logger


@pytest.mark.usefixtures("introspect_child_target_objects")
def test_attach_byo_ucp(kaas_manager):
    def get_clusterreleases_by_ucptag(ucp_tag):
        LOG.info("Get all clusterreleases that support "
                 "UCP version {}".format(ucp_tag))
        cr_list = \
            kaas_manager.api.kaas_clusterreleases.list_raw().to_dict()['items']

        crs = [x for x in cr_list if 'lcmType' in x['spec'] and x['spec']['lcmType'] in ('ucp', 'byo') and
               x['metadata']['name'].startswith("mke")]
        clusterreleases = []
        for cr in crs:
            machinetypes_params = cr['spec']['machineTypes']['control'] + \
                cr['spec']['machineTypes']['worker']
            for ctl in machinetypes_params:
                if 'ucp_tag' in ctl['params'].keys():
                    if ucp_tag != ctl['params']['ucp_tag']:
                        break
            else:
                clusterreleases.append(cr['metadata']['name'])
                LOG.info("Clusterrelease {0} matches {1} ucp version".format(
                    cr['metadata']['name'], ucp_tag))
        return clusterreleases

    mke_client = MKEDashboardClientPlain(
        settings.UCP_UI_URL, settings.UCP_UI_USER, settings.UCP_UI_PASSWORD)
    mke_client.allow_monitoring_agents()

    ucp_k8sclient = K8sCluster(kubeconfig=settings.UCP_BYO_KUBECONFIG)
    # Let's fetch ucp version from 'ucp-metrics' pod
    LOG.info("Fetching standalone UCP cluster version")
    ucp_metrics = [x for x in
                   ucp_k8sclient.pods.list(namespace='kube-system')
                   if 'ucp-metrics' in x.name][0]
    ucp_tag = \
        ucp_metrics.read().spec.containers[0].image.split("/")[1].split(":")[1]

    LOG.info("UCP version is {}".format(ucp_tag))

    # This is temp hack to exclude mosk clusterrelease version with the same
    # ucp tag from the list
    # TODO: better handling?
    clusterreleases = [x for x in get_clusterreleases_by_ucptag(ucp_tag)
                       if 'mos' not in x]
    assert len(clusterreleases) > 0,\
        "Failed to find suitable clusterrelease, matched UCP version"
    LOG.warning("All mosk clusterrelease versions have been excluded."
                "Final list: {}".format(clusterreleases))

    if len(clusterreleases) > 1:
        LOG.warning("More than 1 clusterrelease versions were found: "
                    "{}".format(clusterreleases))
        clusterrelease_version = \
            str(max([version.parse(x) for x in clusterreleases]))
        LOG.info("Latest clusterrelease "
                 "{} will be used".format(clusterrelease_version))
    else:
        clusterrelease_version = clusterreleases[0]

    with open(settings.UCP_BYO_CACERT, 'rb') as f:
        cacert = base64.b64encode(f.read()).decode("utf-8")
    with open(settings.UCP_BYO_CLIENTCERT, 'rb') as f:
        clientcert = base64.b64encode(f.read()).decode("utf-8")
    with open(settings.UCP_BYO_CLIENTKEY, 'rb') as f:
        clientkey = base64.b64encode(f.read()).decode("utf-8")
    with open(settings.UCP_BYO_KUBECONFIG, 'rb') as f:
        ucp_kubeconfig = base64.b64encode(f.read()).decode("utf-8")

    LOG.info("Wait until all MKE pods is ready")
    waiters.wait(lambda: ucp_k8sclient.pods.list_not_in_phases(target_namespaces="kube-system", check_jobs=True),
                 timeout=1800,
                 interval=60,
                 timeout_msg="MKE pods waiting timeout")

    ns_name = settings.TARGET_NAMESPACE
    cluster_name = settings.TARGET_CLUSTER

    ns = kaas_manager.create_namespace(ns_name)
    region = kaas_manager.get_mgmt_cluster().region_name

    creds = ns.create_byocredential(name='creds',
                                    ucp_host=settings.UCP_BYO_DOCKER_URL,
                                    cacert=cacert,
                                    clientcert=clientcert,
                                    clientkey=clientkey,
                                    ucp_kubeconfig=ucp_kubeconfig,
                                    region=region)

    LOG.info("Attaching UCP cluster")
    ns.attach_ucp_cluster(name=cluster_name, creds_name=creds.name, region=region)

    ucp_cluster = ns.get_cluster(name=cluster_name)
    # if LMA HA is true - need to add stacklight label
    LOG.info("Wait machines appear in cluster")
    workers_cnt = int(settings.WORKER_COUNT)
    waiters.wait(
        lambda: len(ucp_cluster.get_machines(machine_type="worker",
                                             raise_if_empty=False)
                    ) == workers_cnt,
        timeout=700, interval=10, timeout_msg="machines appear timeout"
    )

    for worker_m in ucp_cluster.get_machines(machine_type="worker"):
        LOG.info(f"{worker_m}")
        worker_m.add_labels([{"key": "stacklight", "value": "enabled"}])

    ucp_cluster.check.check_cluster_release(
        expected_clusterrelease=clusterrelease_version,
        timeout=1800, interval=15)
    ucp_cluster.check.check_machines_status(timeout=3600, interval=10)
    ucp_cluster.check.check_k8s_nodes()

    LOG.info("Check cluster provider")
    assert ucp_cluster.provider == utils.Provider.byo,\
        "Wrong child cluster provider: {0} instead of " \
        "BYOClusterProviderSpec".format(ucp_cluster.provider)
    ucp_cluster.check.check_cluster_readiness()
    ucp_cluster.check.check_cluster_nodes()
    ucp_cluster.check.check_k8s_pods(timeout=6000)
    ucp_cluster.check.check_actual_expected_pods()

    LOG.info("Make sure metrics-server pod is added to standalone UCP")
    waiters.wait(
        lambda: len(ucp_k8sclient.pods.list(
            namespace='kube-system',
            name_prefix='metrics-server')) > 0,
        timeout=700, interval=10)
    LOG.info("Make sure helm-controller pod is added to standalone UCP")
    waiters.wait(
        lambda: len(ucp_k8sclient.pods.list(
            namespace='kube-system',
            name_prefix='helm-controller')) > 0,
        timeout=400, interval=10)

    master_label = 'node-role.kubernetes.io/master'
    ctl_nodes = \
        len([x for x in ucp_k8sclient.nodes.list_raw().to_dict()['items']
             if master_label in x['metadata']['labels'].keys()])

    wrk_nodes = \
        len([x for x in ucp_k8sclient.nodes.list_raw().to_dict()['items']
             if master_label not in x['metadata']['labels'].keys()])

    LOG.info("Check number of control nodes")
    assert len(ucp_cluster.get_machines(machine_type='control')) == ctl_nodes,\
        "Number of control nodes in child kaas cluster ({0}) is not " \
        "equal to expected number in UCP cluster ({1})".format(
            ucp_cluster.get_machines(machine_type='control'),
            ctl_nodes)

    LOG.info("Check number of worker nodes")
    assert len(ucp_cluster.get_machines(machine_type='worker')) == wrk_nodes,\
        "Number of worker nodes in child kaas cluster ({0}) is in not " \
        "equal to expected number in UCP cluster ({1})".format(
            ucp_cluster.get_machines(machine_type='worker'),
            ctl_nodes)
