import copy

import pytest
import yaml
from tabulate import tabulate

from si_tests import logger
from si_tests import settings
from si_tests.utils import exceptions as timeout_exceptions
from si_tests.utils import waiters

LOG = logger.logger


@pytest.mark.usefixtures('log_method_time')
def test_cleanup_resources(openstack_client):
    """ Test cleanup resources """
    resources_map = {'routers': [], 'servers': [], 'loadbalancers': [],
                     'fips': [], 'networks': [], 'volumes': [],
                     'security_groups': [], 'keypairs': [], 'stacks': []}

    stack_resources_map = copy.deepcopy(resources_map)
    resource_info_map = {}
    not_deleted_resources = {}
    trigger_cleanup = settings.TRIGGER_RESOURCES_CLEANUP
    trigger_stacks_cleanup = settings.TRIGGER_STACKS_CLEANUP
    resource_to_cleanup_pref = settings.RESOURCE_TO_CLEANUP_PREFIX

    stacks_to_cleanup_pref = settings.STACKS_TO_CLEANUP_PREFIX
    if stacks_to_cleanup_pref:
        stacks_to_delete = openstack_client.get_stacks(name_prefix=tuple(stacks_to_cleanup_pref))
        if trigger_stacks_cleanup:
            resources_map['stacks'] = [st.get('id') for st in stacks_to_delete]
            resource_info_map['stacks'] = {st.get('id'): st.get('stack_name') for st in stacks_to_delete}
    else:
        stacks_to_delete = [
            st for st in openstack_client.get_stacks() if
            not any(
                st.get('stack_name').startswith(pref)
                for pref in settings.FILTERED_STACKS_PREFIX)]
        stacks = [st.get('id') for st in stacks_to_delete]
        if stacks and trigger_stacks_cleanup:
            resources_map['stacks'].extend(stacks)
            resource_info_map['stacks'] = {st.get('id'): st.get('stack_name') for st in stacks_to_delete}

    if trigger_stacks_cleanup:
        def wait_stacks_deleted():
            try:
                LOG.info("Waiting for stacks are deleted")
                waiters.wait(lambda: not [
                    stack.id for stack in openstack_client.heat.stacks.list()
                    if stack.id in resources_map['stacks']], timeout=300)
            except timeout_exceptions.TimeoutError:
                not_deleted_stacks = \
                    [stack.id for stack in openstack_client.heat.stacks.list()
                     if stack.id in resources_map['stacks']]
                not_deleted_resources['stacks'] = not_deleted_stacks
                LOG.warning("Some stacks are not deleted: {}".format(
                    not_deleted_stacks))

        if resources_map['stacks']:
            LOG.info("\nNext 'Stacks' resources are found for deletion:\n" +
                     tabulate(resource_info_map['stacks'].items(), tablefmt="presto", headers=['ID', 'Name']))
            for s in resources_map['stacks']:
                try:
                    openstack_client.delete_stack_by_id(s)
                except Exception as e:
                    LOG.warning("Unable to delete stack {}. "
                                "Reason: {}".format(s, e))
            wait_stacks_deleted()
            del resource_info_map['stacks']
    else:
        # collect resource for each stack for later exclusion
        def _collect_resource_id(resources, resource_name):
            return [r.physical_resource_id for r in resources if r.resource_type == resource_name]

        for stack in stacks_to_delete:
            stack_resources = openstack_client.heat.resources.list(stack_id=stack.get('id'))
            resource_type_mapping = {
                'routers': 'OS::Neutron::Router',
                'servers': 'OS::Nova::Server',
                'loadbalancers': 'OS::Octavia::LoadBalancer',
                'fips': 'OS::Neutron::FloatingIP',
                'networks': 'OS::Neutron::Net',
                'volumes': 'OS::Cinder::Volume',
                'security_groups': 'OS::Neutron::SecurityGroup',
                'keypairs': 'OS::Nova::KeyPair'
            }
            LOG.debug(f"\nStack {stack.get('stack_name')}")
            for k, v in resource_type_mapping.items():
                ids = _collect_resource_id(stack_resources, v)
                stack_resources_map[k].extend(ids)
                LOG.debug(f"{v} - {','.join(ids)}")

    def wait_routers_deleted():
        try:
            LOG.info("Waiting for routers are deleted")
            waiters.wait(lambda: not [
                rt['id'] for rt in openstack_client.neutron.list_routers()[
                    'routers'] if rt['id'] in resources_map['routers']],
                         interval=15, timeout=300)
        except timeout_exceptions.TimeoutError:
            not_deleted_routers = [
                rt['id'] for rt in openstack_client.neutron.list_routers()[
                    'routers'] if rt['id'] in resources_map['routers']]
            not_deleted_resources['routers'] = not_deleted_routers
            LOG.warning("Some routers are not deleted: {}".format(
                not_deleted_routers))

    def wait_servers_deleted():
        try:
            LOG.info("Waiting for servers are deleted")
            waiters.wait(lambda: not [
                sv.id for sv in openstack_client.nova.servers.list()
                if sv.id in resources_map['servers']],
                         interval=15, timeout=300)
        except timeout_exceptions.TimeoutError:
            not_deleted_servers = [
                sv.id for sv in openstack_client.nova.servers.list()
                if sv.id in resources_map['servers']]
            not_deleted_resources['servers'] = not_deleted_servers
            LOG.warning("Some servers are not deleted: {}".format(
                not_deleted_servers))

    def wait_loadbalancers_deleted():
        try:
            LOG.info("Waiting for loadbalancers are deleted")
            waiters.wait(lambda: not [
                lbl['id'] for lbl in
                openstack_client.octavia.load_balancer_list()[
                    'loadbalancers'] if lbl['id'] in resources_map[
                    'loadbalancers']], interval=15, timeout=300)
        except timeout_exceptions.TimeoutError:
            not_deleted_lbs = \
                [lbl['id'] for lbl in
                 openstack_client.octavia.load_balancer_list()[
                     'loadbalancers'] if lbl['id'] in resources_map[
                     'loadbalancers']]
            not_deleted_resources['loadbalancers'] = not_deleted_lbs
            LOG.warning("Some loadbalancers are not deleted: {}".format(
                not_deleted_lbs))

    def wait_fips_deleted():
        try:
            LOG.info("Waiting for floating ips are are deleted")
            waiters.wait(lambda: not [
                fip['id'] for fip in
                openstack_client.neutron.list_floatingips()[
                    'floatingips'] if fip['id'] in resources_map[
                    'fips']], interval=10, timeout=60)
        except timeout_exceptions.TimeoutError:
            not_deleted_fips = \
                [fip['id'] for fip in
                 openstack_client.neutron.list_floatingips()[
                     'floatingips'] if fip['id'] in resources_map[
                     'fips']]
            not_deleted_resources['fips'] = not_deleted_fips
            LOG.warning("Some floating ips are not deleted: {}".format(
                not_deleted_fips))

    def wait_networks_deleted():
        try:
            LOG.info("Waiting for networks are are deleted")
            waiters.wait(lambda: not [
                net['id'] for net in
                openstack_client.neutron.list_networks()[
                    'networks'] if net['id'] in resources_map[
                    'networks']], interval=15, timeout=300)
        except timeout_exceptions.TimeoutError:
            not_deleted_nets = \
                [net['id'] for net in
                 openstack_client.neutron.list_networks()[
                     'networks'] if net['id'] in resources_map[
                     'networks']]
            not_deleted_resources['networks'] = not_deleted_nets
            LOG.warning("Some networks are not deleted: {}".format(
                not_deleted_nets))

    def wait_volumes_deleted():
        try:
            LOG.info("Waiting for volumes are deleted")
            waiters.wait(lambda: not [
                vol.id for vol in openstack_client.cinder.volumes.list()
                if vol.id in resources_map['volumes']],
                         interval=30, timeout=300)
        except timeout_exceptions.TimeoutError:
            not_deleted_volumes = \
                [vol.id for vol in openstack_client.cinder.volumes.list()
                 if vol.id in resources_map['volumes']]
            not_deleted_resources['volumes'] = not_deleted_volumes
            LOG.warning("Some volumes are not deleted: {}".format(
                not_deleted_volumes))

    def wait_security_groups_deleted():
        try:
            LOG.info("Waiting for security groups are deleted")
            waiters.wait(lambda: not [
                sg.get('id') for sg in openstack_client.get_security_groups()
                if sg.get('id') in resources_map['security_groups']],
                         interval=30, timeout=300)
        except timeout_exceptions.TimeoutError:
            not_deleted_sgs = \
                [sg.get('id') for sg in openstack_client.get_security_groups()
                 if sg.get('id') in resources_map['security_groups']]
            not_deleted_resources['security_groups'] = not_deleted_sgs
            LOG.warning("Some security_groups are not deleted: {}".format(
                not_deleted_sgs))

    routers_resource = openstack_client.get_routers(name_prefix=resource_to_cleanup_pref)
    routers_to_delete = [rt for rt in routers_resource if rt.get('id') not in stack_resources_map['routers']]
    resources_map['routers'] = [rt.get('id') for rt in routers_to_delete]
    resource_info_map['routers'] = {rt.get('id'): rt.get('name') for rt in routers_to_delete}

    servers_resource = openstack_client.get_servers(name_prefix=resource_to_cleanup_pref)
    servers_to_delete = [server for server in servers_resource if
                         server.get('id') not in stack_resources_map['servers']]
    resources_map['servers'] = [server.get('id') for server in servers_to_delete]
    resource_info_map['servers'] = {server.get('id'): server.get('name') for server in servers_to_delete}

    loadbalancers_resource = [lb for lb in openstack_client.get_loadbalancers() if lb.get(
        'provisioning_status') == 'ACTIVE']
    loadbalancers_to_delete = [lb for lb in loadbalancers_resource if
                               lb.get('id') not in stack_resources_map['loadbalancers']]
    resources_map['loadbalancers'] = [lb.get('id') for lb in loadbalancers_to_delete]
    resource_info_map['loadbalancers'] = {lb.get('id'): lb.get('name') for lb in loadbalancers_to_delete}

    networks_resource = openstack_client.get_networks(name_prefix=resource_to_cleanup_pref)
    networks_to_delete = [net for net in networks_resource if net.get('id') not in stack_resources_map['networks']]
    resources_map['networks'] = [net.get('id') for net in networks_to_delete]
    resource_info_map['networks'] = {net.get('id'): net.get('name') for net in networks_to_delete}

    keypairs_resource = openstack_client.get_keypairs(name_prefix='bootstrap')
    keypairs_to_delete = [key for key in keypairs_resource if key.get('name') not in stack_resources_map['keypairs']]
    resources_map['keypairs'] = [key.get('id') for key in keypairs_to_delete]
    resource_info_map['keypairs'] = {key.get('id'): key.get('name') for key in keypairs_to_delete}

    vol_name_prefixes = ['pvc-', 'kubernetes', 'kaas']
    volumes_resource = openstack_client.get_volumes(name_prefix=tuple(vol_name_prefixes))
    volumes_to_delete = [vol for vol in volumes_resource if vol.get('id') not in stack_resources_map['volumes']]
    resources_map['volumes'] = [vol.get('id') for vol in volumes_to_delete]
    resource_info_map['volumes'] = {vol.get('id'): vol.get('name') for vol in volumes_to_delete}

    security_groups_resource = openstack_client.get_security_groups(name_prefix=resource_to_cleanup_pref)
    security_groups_to_delete = [sg for sg in security_groups_resource if
                                 sg.get('id') not in stack_resources_map['security_groups']]
    resources_map['security_groups'] = [sg.get('id') for sg in security_groups_to_delete]
    resource_info_map['security_groups'] = {sg.get('id'): sg.get('name') for sg in security_groups_to_delete}

    if trigger_cleanup:

        for k, v in resource_info_map.items():
            LOG.info(f"\nNext '{k.capitalize()}' resources are found for deletion:\n" +
                     tabulate(v.items(), tablefmt="presto", headers=['ID', 'Name']))

        # Some resources are depend from others, so will wait in right order
        if resources_map['servers']:
            for s in resources_map['servers']:
                try:
                    openstack_client.delete_server(s)
                except Exception as e:
                    LOG.warning("Unable to delete server {}. "
                                "Reason: {}".format(s, e))
            wait_servers_deleted()

        if resources_map['loadbalancers']:
            for lb in resources_map['loadbalancers']:
                try:
                    openstack_client.delete_loadbalancer(lb)
                except Exception as e:
                    LOG.warning("LB {} can not be deleted. {}".format(lb, e))
            wait_loadbalancers_deleted()

        if resources_map['routers']:
            for r in resources_map['routers']:
                try:
                    openstack_client.delete_router(r)
                except Exception as e:
                    LOG.warning("Unable to delete router {}. "
                                "Reason: {}".format(r, e))
            wait_routers_deleted()

        if resources_map['networks']:
            for n in resources_map['networks']:
                try:
                    openstack_client.delete_network(n)
                except Exception as e:
                    LOG.warning("Unable to delete network {}. "
                                "Reason: {}".format(n, e))
            wait_networks_deleted()

        if resources_map['keypairs']:
            for key in resources_map['keypairs']:
                try:
                    openstack_client.delete_keypair(key)
                except Exception as e:
                    LOG.warning("Unable to delete keypair {}. "
                                "Reason: {}".format(key, e))

        if resources_map['volumes']:
            for v in resources_map['volumes']:
                try:
                    openstack_client.delete_volume(v)
                except Exception as e:
                    LOG.warning("Unable to delete volume {}. "
                                "Reason: {}".format(v, e))
            wait_volumes_deleted()

        # Cleanup all of floating IPs which are not associated with ports
        resources_map['fips'] = [
            fip.get('id') for fip in openstack_client.get_fips()
            if fip.get('port_id') is None]
        if resources_map['fips']:
            LOG.debug("Next FIPs are found for deletion: \n{}".format(
                yaml.dump(resources_map['fips'])))
            for ip in resources_map['fips']:
                try:
                    openstack_client.delete_fip(ip)
                except Exception as e:
                    LOG.warning("Unable to delete floating IP {}. "
                                "Reason: {}".format(ip, e))
            wait_fips_deleted()

        if resources_map['security_groups']:
            for sg in resources_map['security_groups']:
                try:
                    openstack_client.delete_security_group(sg)
                except Exception as e:
                    LOG.warning("Security group {} can not be "
                                "deleted. {}".format(sg, e))
            wait_security_groups_deleted()

    if not_deleted_resources:
        LOG.warning("Next resources were not deleted: \n{}".format(
            yaml.dump(not_deleted_resources)))
    else:
        if trigger_stacks_cleanup:
            LOG.info("All found stacks have been deleted")
        if trigger_cleanup:
            LOG.info("All found resources have been deleted")
