#    Copyright 2019 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.
import os

from si_tests import logger
from si_tests import settings
from si_tests.managers import bootstrap_manager
from si_tests.managers import si_config_manager

LOG = logger.logger


class TestProvisionMgmCluster():

    def test_provision_seed_node_vm(self, openstack_client, show_step):
        """Bootstrap KaaS on OpenStack VMs

        Scenario:
            1. Erase previous KaaS environment and seed node, if exists
            2. Create a seed VM using heat template
            3. Inject proxy config (optional)
            4. Prepare seed node
        """

        if settings.KAAS_CORE_REPO_URL:
            assert settings.CDN_REGION != 'public', (
                f"'CDN_REGION' is set to '{settings.CDN_REGION}', but we are using custom release artifacts from "
                f"{settings.KAAS_CORE_REPO_URL} which requires to use build artifacts "
                f"from an internal CDN. Please select a correct 'CDN_REGION' for these artifacts")

        # step 001
        if settings.KEEP_ENV_BEFORE:
            LOG.info("Skip erase due KEEP_ENV_BEFORE flag is setted")
        else:
            if openstack_client._current_stack:
                # Erase the stack with the seed node
                show_step(1)
                seed_ip = openstack_client.get_seed_ip()
                old_bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
                old_bootstrap.step_001_erase_env_before()
                LOG.info(f"Remove heat stack {settings.ENV_NAME}")
                openstack_client.delete_stack()
            else:
                LOG.warning("Can't erase environment due "
                            "seed node stack is absent")

        # Create a stack with a seed node
        show_step(2)
        openstack_client.create_seed_node()
        seed_ip = openstack_client.get_seed_ip()
        bootstrap = bootstrap_manager.BootstrapManager(seed_ip=seed_ip)
        bootstrap.wait_for_seed_node_ssh()
        if settings.SI_HEAT_AUTO_GUESS_CONFIG:
            bootstrap.store_stack_info(openstack_client._stack_resources_data())
        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        si_config.store_seed_ip(seed_ip)

        show_step(3)
        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
            remote = bootstrap.remote_seed()

            if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
                remote.upload(
                    os.path.abspath(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE),
                    settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE,
                )
                sudo_mode = True
                if settings.SEED_SSH_LOGIN in settings.SYSTEM_ADMINS:
                    sudo_mode = False
                with remote.sudo(enforce=sudo_mode):
                    remote.execute(f"cp {settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE} {settings.SEED_CERTS_DIR}")
                    remote.execute("update-ca-certificates")
                LOG.info("Ssl certificate has been updated on seed node")

            run_envs = dict()
            if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
                run_envs['PROXY_CA_CERTIFICATE_PATH'] = settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE
            run_envs['HTTP_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['http_proxy'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['HTTPS_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['https_proxy'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR

            if settings.KAAS_PROXYOBJECT_NO_PROXY:
                endpoints = openstack_client.get_os_cloud_endpoints()
                no_proxy = f"{settings.KAAS_PROXYOBJECT_NO_PROXY},{settings.OS_FIP_RANGES},{endpoints}"
                run_envs['NO_PROXY'] = no_proxy
                run_envs['no_proxy'] = no_proxy

            envs_string = '\n'.join(["{}='{}'".format(k, v) for k, v in run_envs.items()])
            with remote.sudo(enforce=True):
                with remote.open('/tmp/environment', mode='a') as file:
                    file.write(envs_string)
                remote.execute("cat /tmp/environment >> /etc/environment")
                remote.execute("rm -f /tmp/environment")

            LOG.info(f"The following proxy objects have been added to /etc/environment: \n{envs_string}")

        show_step(4)
        bootstrap.step_003_prepare_seed_node_base()
