provider "vsphere" {
  vsphere_server = var.kaas_vsphere_server_ip
  user           = var.kaas_vsphere_capi_provider_username
  password       = var.kaas_vsphere_capi_provider_password

  # Enable this if your vSphere server has a self-signed certificate
  allow_unverified_ssl = true
}

data "vsphere_datacenter" "dc" {
  name = var.kaas_vsphere_datacenter_name
}

data "vsphere_resource_pool" "resource_pool" {
  name          = var.kaas_vsphere_resource_pool_path
  datacenter_id = data.vsphere_datacenter.dc.id
}

data "vsphere_datastore_cluster" "datastore_cluster" {
  name          = var.kaas_vsphere_datastorecluster_path
  datacenter_id = data.vsphere_datacenter.dc.id
}

data "vsphere_network" "network" {
  name          = var.kaas_vsphere_network_path
  datacenter_id = data.vsphere_datacenter.dc.id
}

data "vsphere_virtual_machine" "template_vm_linux" {
  name          = var.kaas_vsphere_os_template_path
  datacenter_id = data.vsphere_datacenter.dc.id
}

resource "tls_private_key" "ssh_key" {
  algorithm = "RSA"
  rsa_bits  = "4096"
}

resource "local_file" "ssh_public_key" {
  content  = tls_private_key.ssh_key.private_key_pem
  filename = "./ssh_keys/${var.cluster_name}.pem"
  provisioner "local-exec" {
    command = "chmod 0600 ${local_file.ssh_public_key.filename}"
  }
}

data "vsphere_datastore" "datastore" {
  name          = var.kaas_vsphere_datastore_name
  datacenter_id = data.vsphere_datacenter.dc.id
}

locals {
  ip_range_start = split("-", var.kaas_vsphere_ipam_range)[0]
  ip_range_end   = split("-", var.kaas_vsphere_ipam_range)[1]

  # Note that this naively only works for IP ranges using the same first three octects
  ip_range_first_three_octets = join(".", slice(split(".", local.ip_range_start), 0, 3))
  ip_range_start_fourth_octet = split(".", local.ip_range_start)[3]
  ip_range_end_fourth_octet   = split(".", local.ip_range_end)[3]

  list_of_final_octet  = range(local.ip_range_start_fourth_octet, local.ip_range_end_fourth_octet)
  list_of_ips_in_range = formatlist("${local.ip_range_first_three_octets}.%s", local.list_of_final_octet)

  net_master  = slice(local.list_of_ips_in_range, 0, var.managers_count)
  net_workers = slice(local.list_of_ips_in_range, var.managers_count, var.managers_count+var.workers_count)
}

module "managers" {
  source               = "./modules/virtual_machine"
  quantity             = var.managers_count
  name_prefix          = "manager"
  resource_pool_id     = data.vsphere_resource_pool.resource_pool.id
  datastore_id         = data.vsphere_datastore.datastore.id
  folder               = var.kaas_vsphere_machines_folder
  network_id           = data.vsphere_network.network.id
  template_vm          = data.vsphere_virtual_machine.template_vm_linux
  disk_size            = var.kaas_vsphere_machine_disk
  ram                  = var.kaas_vsphere_machine_ram
  num_cpu              = var.kaas_vsphere_numcpu
  guest_ssh_key_public = tls_private_key.ssh_key.public_key_openssh
  guest_ssh_user       = var.ssh_username
  cluster_name         = var.cluster_name
  is_ipam_enabled      = var.kaas_vsphere_ipam_enabled
  dns                  = var.kaas_vsphere_dns
  gateway              = var.kaas_vsphere_network_gateway
  ip_addresses         = local.net_master
}

module "workers" {
  source               = "./modules/virtual_machine"
  quantity             = var.workers_count
  name_prefix          = "worker"
  resource_pool_id     = data.vsphere_resource_pool.resource_pool.id
  datastore_id         = data.vsphere_datastore.datastore.id
  folder               = var.kaas_vsphere_machines_folder
  network_id           = data.vsphere_network.network.id
  template_vm          = data.vsphere_virtual_machine.template_vm_linux
  disk_size            = var.kaas_vsphere_machine_disk
  ram                  = var.kaas_vsphere_machine_ram
  num_cpu              = var.kaas_vsphere_numcpu
  guest_ssh_key_public = tls_private_key.ssh_key.public_key_openssh
  guest_ssh_user       = var.ssh_username
  cluster_name         = var.cluster_name
  is_ipam_enabled      = var.kaas_vsphere_ipam_enabled
  dns                  = var.kaas_vsphere_dns
  gateway              = var.kaas_vsphere_network_gateway
  ip_addresses         = local.net_workers
}

